/*
 * $Id:$
 */

package csbase.client.applications.xmlviewer;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ButtonGroup;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JToolBar;

import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationProject;
import csbase.client.applications.xmlviewer.actions.XMLViewerCloseAction;
import csbase.client.applications.xmlviewer.actions.XMLViewerFindAction;
import csbase.client.applications.xmlviewer.actions.XMLViewerLocalOpenAction;
import csbase.client.applications.xmlviewer.actions.XMLViewerProjectOpenAction;
import csbase.client.applications.xmlviewer.actions.XMLViewerRefreshAction;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.util.charset.CharsetRadioMenu;
import csbase.client.util.charset.CharsetRadioMenuChangeListener;
import csbase.client.util.xmlpanel.XMLPanelStyleInterface;
import csbase.client.util.xmlpanel.XMLStandardPanel;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * Aplicativo.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class XMLViewer extends ApplicationProject {

  /**
   * Timeout da barra de status.
   */
  public static final int STATUS_BAR_TIMEOUT = 5;

  /**
   * Arquivo corrente.
   */
  private ClientFile currentFile;

  /**
   * Indicativo de arquivo corrente salvo.
   */
  private boolean isSaved = true;

  /**
   * Indicativo de arquivo corrente em modo read-only.
   */
  private boolean isReadOnly = true;

  /**
   * Painel do programa.
   */
  final private XMLStandardPanel xmlPanel;

  /**
   * Ao de abertura de arquivo local.
   */
  final private XMLViewerLocalOpenAction openLocalAction;

  /**
   * Ao de abertura de arquivo de projeto.
   */
  final private XMLViewerProjectOpenAction openProjectAction;

  /**
   * Adapter que acompanha a abertura/fechamento de projetos no desktop.
   */
  final private ProjectTreeAdapter projectTreeAdapter;

  /**
   * Menu de escolha de encoding
   */
  final private CharsetRadioMenu charsetMenu = new CharsetRadioMenu();

  /**
   * Item de menu para estilo nulo.
   */
  final private JRadioButtonMenuItem noStyleItem = new JRadioButtonMenuItem();

  /**
   * Lista de layouts configurada para o programa.
   */
  final private List<XMLPanelStyleInterface> styles =
    new ArrayList<XMLPanelStyleInterface>();

  /**
   * Mapa para converso de estilo para item de menu associado.
   */
  final private HashMap<XMLPanelStyleInterface, JRadioButtonMenuItem> styleItemMap =
    new HashMap<XMLPanelStyleInterface, JRadioButtonMenuItem>();
  /**
   * Charset corrente.
   */
  private Charset currentCharset;

  /**
   * Montagem do menu de arquivos.
   * 
   * @return o menu de arquivos.
   */
  private JMenu buildFileMenu() {
    final JMenu menu = new JMenu(getMyString("file.menu"));
    menu.add(openProjectAction);
    if (allowLocalFiles()) {
      menu.add(openLocalAction);
    }
    menu.add(new XMLViewerCloseAction(this));
    menu.addSeparator();
    menu.add(new ApplicationExitAction(this));
    return menu;
  }

  /**
   * Indica se aplicao deve aceitar arquivos locais.
   * 
   * @return indicativo
   */
  private boolean allowLocalFiles() {
    final String propName = "allow.local.files";
    boolean flag = getBooleanSpecificProperty(propName, false);
    return flag;
  }

  /**
   * Montagem do menu de ajuda
   * 
   * @return um menu de about.
   */
  private JMenu buildOptionMenu() {
    charsetMenu.addCharsetChangeListener(new CharsetRadioMenuChangeListener() {
      @Override
      public void charsetChanged(final Charset charset) {
        currentCharset = charset;
      }
    });
    charsetMenu.setText(getMyString("charset.options.menu"));

    final JMenu menu = new JMenu(getMyString("options.menu"));
    menu.add(charsetMenu);
    menu.add(new XMLViewerFindAction(this));
    menu.addSeparator();
    menu.add(new XMLViewerRefreshAction(this));
    return menu;
  }

  /**
   * Montagem do menu de ajuda
   * 
   * @return um menu de about.
   */
  private JMenu buildHelpMenu() {
    final JMenu helpMenu = new JMenu(getMyString("help.menu"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * Montagem do menu.
   * 
   * @return o menu do programa.
   */
  private JMenuBar buildMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(buildFileMenu());
    menuBar.add(buildOptionMenu());
    JMenu layMenu = buildStylesMenu();
    if (layMenu != null) {
      menuBar.add(layMenu);
    }
    menuBar.add(buildHelpMenu());
    return menuBar;
  }

  /**
   * Montagem da toolbar.
   * 
   * @return o painel.
   */
  private JToolBar buildToolBar() {
    final String appName = getName();
    final JToolBar toolBar = new JToolBar(appName);
    toolBar.setFloatable(false);

    toolBar.add(openProjectAction);
    if (allowLocalFiles()) {
      toolBar.add(openLocalAction);
    }

    toolBar.addSeparator();
    toolBar.add(new XMLViewerFindAction(this));

    return toolBar;
  }

  /**
   * Cria o listener para alterao/abertura/fechamento de projetos.
   * 
   * @return o listener
   */
  private ProjectTreeAdapter buildProjectTreeAdapter() {
    final ProjectTreeAdapter pta = new ProjectTreeAdapter() {
      @Override
      public void projectChanged(final CommonClientProject proj) {
        closeDependentWindows();
        final boolean hasProj = (proj != null);
        openProjectAction.setEnabled(hasProj);
        if (!hasProj && currentFile != null) {
          final ClientFileType type = currentFile.getClientFileType();
          if (type != ClientFileType.REMOTE) {
            return;
          }
          try {
            resetFile();
          }
          catch (Exception e) {
            e.printStackTrace();
          }
        }
      }
    };
    return pta;
  }

  /**
   * Construtor
   * 
   * @param id identificador da aplicao.
   * @throws ApplicationException em caso de falha
   */
  public XMLViewer(String id) throws ApplicationException {
    super(id);
    final Client client = Client.getInstance();
    this.currentCharset = client.getSystemDefaultCharset();
    this.projectTreeAdapter = buildProjectTreeAdapter();

    final String attTitle = getMyString("attribute.title");
    final String txtTitle = getMyString("text.title");
    xmlPanel = new XMLStandardPanel(attTitle, txtTitle);
    xmlPanel.setSearchPanelVisible(false);
    openLocalAction = new XMLViewerLocalOpenAction(this);
    openProjectAction = new XMLViewerProjectOpenAction(this);

    initConfiguredStylists();

    final ApplicationFrame applicationFrame = getApplicationFrame();
    final JMenuBar menu = buildMenuBar();
    applicationFrame.setJMenuBar(menu);
    applicationFrame.setSize(new Dimension(800, 600));
    buildMainFrame(applicationFrame);

    initProjectListener();
  }

  /**
   * @throws ApplicationException
   */
  private void initConfiguredStylists() throws ApplicationException {
    final List<String> props = getStringListSpecificProperty("style");
    if (props == null) {
      return;
    }
    for (String prop : props) {
      try {
        @SuppressWarnings("unchecked")
        Class<? extends XMLPanelStyleInterface> cls =
          (Class<? extends XMLPanelStyleInterface>) Class.forName(prop);
        Constructor<? extends XMLPanelStyleInterface> constructor =
          cls.getConstructor();
        final XMLPanelStyleInterface layout = constructor.newInstance();
        styles.add(layout);
      }
      catch (Exception e) {
        throw new ApplicationException(e);
      }
    }
  }

  /**
   * Cria menu de layouts
   * 
   * @return o meneu
   */
  private JMenu buildStylesMenu() {
    if (styles == null || styles.size() == 0) {
      return null;
    }

    final JMenu stylesMenu = new JMenu();
    stylesMenu.setText(getMyString("styles.menu"));
    final ButtonGroup grp = new ButtonGroup();
    for (final XMLPanelStyleInterface style : styles) {
      final String styleName = style.getName();
      final AbstractAction action = new AbstractAction() {
        @Override
        public void actionPerformed(final ActionEvent e) {
          xmlPanel.setStyle(style);
        }
      };
      final JRadioButtonMenuItem item = new JRadioButtonMenuItem();
      styleItemMap.put(style, item);
      item.setAction(action);
      item.setText(styleName);
      grp.add(item);
      stylesMenu.add(item);
    }

    // Item para anular estilo.
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        xmlPanel.setStyle(null);
      }
    };
    final String noStyleText = getMyString("no.style.item");
    noStyleItem.setAction(action);
    noStyleItem.setText(noStyleText);
    grp.add(noStyleItem);
    stylesMenu.add(noStyleItem);
    noStyleItem.setSelected(true);

    return stylesMenu;
  }

  /**
   * @param applicationFrame
   */
  private void buildMainFrame(final ApplicationFrame applicationFrame) {
    final JToolBar toolbar = buildToolBar();
    final Container contentPane = applicationFrame.getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(xmlPanel, BorderLayout.CENTER);
    contentPane.add(toolbar, BorderLayout.NORTH);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() {
    if (currentFile != null) {
      try {
        currentFile.close(true);
      }
      catch (IOException ioe) {
        handleError(ioe);
      }
    }
  }

  /**
   * Tratador de exceo da aplicao.
   * 
   * @param e exceo
   */
  final public void handleException(final Exception e) {
    final ApplicationFrame frame = getApplicationFrame();
    final String title = getName();
    final String err = e.getMessage();
    final String stackMsg = getMyString("exception.stack.message");
    final String generalMsg = getMyString("exception.error.message");
    final String msg = generalMsg + "\n" + " \u2022 " + err + "\n\n" + stackMsg;
    if (StandardDialogs.showYesNoDialog(frame, title, msg) == 0) {
      showExceptionStack(e);
    }
  }

  /**
   * Busca texto de internacionalizao.
   * 
   * @param tag tag
   * @return texto
   */
  private String getMyString(final String tag) {
    return getClassString(this.getClass(), tag);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected boolean userCanKillApplication() {
    if (isReadOnly) {
      return true;
    }
    return isSaved;
  }

  /**
   * Abre arquivo.
   * 
   * @param file arquivo de entrada.
   * @throws Exception em caso de erro.
   */
  public void openXML(final ClientFile file) throws Exception {
    if (file.size() <= 0) {
      final String err = getMyString("empty.file.error");
      throw new IOException(err);
    }

    currentFile = file;
    xmlPanel.setXMLClientFile(file, null, currentCharset);

    SwingThreadDispatcher.invokeLater(new Runnable() {
      @Override
      public void run() {
        updateTitle();
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(final String name, final Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(ApplicationProject.PROJECT_FILE_MESSAGE)) {
      final ClientProjectFile file = (ClientProjectFile) value;
      final String fileType = file.getType();
      final List<String> fileTypes = getFileTypes();
      if (!fileTypes.contains(fileType)) {
        return;
      }
      try {
        openXML(file);
        XMLPanelStyleInterface style = getTipicalStyle(file.getType());
        setStyle(style);
      }
      catch (Exception e) {
        handleException(e);
      }
    }
  }

  /**
   * Verifica se existe projeto corrente.
   * 
   * @return indicativo.
   */
  private boolean hasProject() {
    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    final ProjectTree projectTree = desktopFrame.getTree();
    if (projectTree == null) {
      return false;
    }
    final CommonClientProject project = projectTree.getProject();
    final boolean hasProj = project != null;
    return hasProj;
  }

  /**
   * Inicializa o listener de projeto.
   */
  private void initProjectListener() {
    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    final ProjectTree projectTree = desktopFrame.getTree();
    if (projectTree == null) {
      final String err = "projectTree == null";
      throw new IllegalStateException(err);
    }
    if (projectTreeAdapter == null) {
      final String err = "projectTreeAdapter == null";
      throw new IllegalStateException(err);
    }
    projectTree.addProjectTreeListener(projectTreeAdapter);

    final boolean hasProject = hasProject();
    openProjectAction.setEnabled(hasProject);
  }

  /**
   * Atualizao de ttulo.
   */
  private void updateTitle() {
    final ApplicationFrame frame = getApplicationFrame();
    final String appName = getName();
    if (currentFile == null) {
      frame.setTitle(appName);
    }
    else {
      final String path = currentFile.getStringPath();
      frame.setTitle(appName + " - " + path);
    }
  }

  /**
   * Limpa a definio de arquivo corrente.
   */
  public void resetFile() {
    try {
      xmlPanel.setXMLClientFile(null, null, currentCharset);
    }
    catch (Exception e) {
      this.showExceptionStack(e);
    }
    currentFile = null;
    updateTitle();
  }

  /**
   * Exibe o painel de busca.
   */
  public void showFindPanel() {
    xmlPanel.setSearchPanelVisible(true);
  }

  /**
   * Atualizao.
   */
  public void refreshAll() {
    xmlPanel.refreshAll();
  }

  /**
   * Consulta o estilo tpico a ser usado dado um tipo de arquivo.
   * 
   * @param fileType tipo de arquivo
   * @return estilo (ou {@code null}).
   */
  public XMLPanelStyleInterface getTipicalStyle(String fileType) {
    for (XMLPanelStyleInterface style : styles) {
      final List<String> fileTypes = style.getTipicalFileTypes();
      if (fileTypes.contains(fileType)) {
        return style;
      }
    }
    return null;
  }

  /**
   * Ajusta estilo do programa.
   * 
   * @param style estilo
   */
  public void setStyle(XMLPanelStyleInterface style) {
    xmlPanel.setStyle(style);
    updateStyleMenu(style);
  }

  /**
   * Atualizao de estado do menu de estilos com base no ajustado.
   * 
   * @param style estilo selecionado
   */
  private void updateStyleMenu(XMLPanelStyleInterface style) {
    if (style == null) {
      noStyleItem.setSelected(true);
      return;
    }
    final JRadioButtonMenuItem chk = styleItemMap.get(style);
    chk.setSelected(true);
  }
}
