package csbase.client.ias;

import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.text.MessageFormat;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.srvproxies.UserGroupProxy;
import csbase.client.remote.srvproxies.UserProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserInfo;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;

/**
 * Dilogo para incluso de dados de um ou mais usurios.
 * 
 * <p>
 * NOTA 1: Caso o mtodo de autenticao do servidor no seja validao local, o
 * dilogo exibe um <i>checkbox</i> permitindo uma autenticao local "forada".
 * Isto  necessrio devido a uma circunstncia atual na Petrobras de
 * descentralizao na administrao de servidores LDAP. Como existem muitas
 * configuraes diferentes, o sistema tem dificuldade em conectar-se a alguns
 * desses servidores, o que impossibilita usurios cadastrados nesses servidores
 * de acessarem o sistema. Para contornar esse problema, foi criada essa
 * possibilidade de login local forado para o administrador poder atribuir a
 * usurios enquadrados nessa situao, enquanto uma soluo mais geral no for
 * encontrada.
 * </p>
 * 
 * <p>
 * NOTA 2: a sigla "DTO" usada na documentao refere-se ao padro <i>Data
 * Transfer Object</i>, utilizado por um objeto que centraliza as informaes do
 * usurio (ver {@link UserInfoDialog#userInfo}).
 * </p>
 * 
 * @author Tecgraf
 */
public class IncludeUserInfoDialog extends UserInfoDialog {

  /** Combo de grupo de usurios ao qual o usurio pertence */
  private JComboBox userGroupComboBox;
  /** Checkbox indicando login local forado. */
  private JCheckBox forceLocalLoginCheckBox;
  /** Boto de incluso de usurio */
  private JButton includeUserButton;
  /** Boto de perfis */
  private JButton roleButton;
  /** Boto de permisses */
  private JButton permissionButton;
  /** Texto para exibir a UN, caso ela no exista na userGroupComboBox. **/
  private JLabel groupLabel;

  /**
   * Cria o dilogo de incluso de usurios.
   * 
   * @param ownerWindow janela que criou este dilogo.
   */
  public IncludeUserInfoDialog(Window ownerWindow) {
    super(ownerWindow, ClientUtilities.addSystemNameToTitle(LNG
      .get("ias.user.inclusion_title")));
  }

  /**
   * {@inheritDoc} <br>
   * Acrescenta ao DTO as selees de grupo de usurio e atributo "login local
   * forado". <br>
   * OBS: Os dilogos de informaes complementares (perfis, permisses e
   * senhas) se encarregam de atualizar o DTO, portanto no  necessrio
   * atualizar essas informaes aqui.
   */
  @Override
  protected void updateUserInfo() {
    super.updateUserInfo();
    UserGroupName userGroupName =
      (UserGroupName) userGroupComboBox.getSelectedItem();
    UserGroup userGroup = userGroupName.getUserGroup();
    userInfo.setAttribute(UserGroup.USERGROUP_ID, userGroup.getId());
    if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
      Boolean forceLocalLogin =
        new Boolean(forceLocalLoginCheckBox.isSelected());
      userInfo.setAttribute(User.FORCE_LOCAL_LOGIN, forceLocalLogin);
    }
  }

  /**
   * {@inheritDoc} <br>
   * Acrescenta ao painel o boto "Incluir".
   * 
   * @see IncludeUserInfoDialog#makeIncludeUserButton()
   */
  @Override
  protected JPanel makeLowerPanel() {
    JPanel buttonPanel = super.makeLowerPanel();
    includeUserButton = makeIncludeUserButton();
    includeUserButton.setEnabled(false);
    buttonPanel.add(includeUserButton, 0);
    return buttonPanel;
  }

  /**
   * Cria o boto de incluso de usurio, atribuindo-lhe como ao o mtodo
   * {@link IncludeUserInfoDialog#createUser()}.
   * 
   * @return o boto de incluso de usurio.
   */
  private JButton makeIncludeUserButton() {
    JButton button = new JButton(LNG.get("IAS_USER_INCLUDE"));
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        createUser();
      }
    });
    return button;
  }

  /**
   * Executa a ao de criar o usurio, usando os dados preenchidos nos campos
   * do dilogo. Se a criao for bem-sucedida, limpa os campos, possibilitando
   * mltiplas incluses em srie. O mtodo assume que qualquer erro ocorrido na
   * criao do usurio dever ser reportado no ponto em que ocorrer, portanto
   * no se preocupa com a exibio de mensagem de erro, apenas de sucesso.
   * 
   * @return <code>true</code> se o usurio tiver sido criado com sucesso,
   *         <code>false</code> se este no tiver sido criado, seja por falta de
   *         informaes, seja por erro na criao.
   */
  protected boolean createUser() {
    updateUserInfo();
    // Caso o servidor aceite troca de senhas, pede uma senha para o login.
    // Caso contrrio, cadastra como senha local o prprio login do usurio, a
    // qual s ser usada caso o administrador force o login local para esse
    // usurio.
    if (ClientRemoteMonitor.getInstance().canChangePasswords()
      || (forceLocalLoginCheckBox != null && forceLocalLoginCheckBox
        .isSelected())) {
      IncludePasswordDialog dialog = new IncludePasswordDialog(userInfoDialog);
      String password = dialog.getNewPassword();
      if (password == null) {
        return false;
      }
      userInfo.setAttribute(User.PASSWORD, password);
    }
    else {
      userInfo.setAttribute(User.PASSWORD, userInfo.getLogin());
    }
    Date currentDate = Calendar.getInstance().getTime();
    userInfo.setAttribute(User.CREATION_DATE, currentDate);
    User user =
      UserProxy.createUser(userInfoDialog, dialogTitle,
        LNG.get("IAS_WAITING_USER_CREATION"), userInfo);
    if (user == null) {
      return false;
    }
    JOptionPane.showMessageDialog(userInfoDialog, MessageFormat.format(
      LNG.get("IAS_USER_INCLUDED_WITH_SUCCESS"),
      new Object[] { user.getLogin() }), dialogTitle,
      JOptionPane.INFORMATION_MESSAGE);
    clearFields();
    return true;
  }

  /**
   * Limpa todos os campos de dados do usurio (menos o combo de UserGroup e o
   * checkbox de login local forado - estes permanecem com a ltima seleo
   * para facilitar a incluso sucessiva de vrios usurios com essas mesmas
   * caractersticas. <br>
   * Desabilita tambm o boto de incluir.
   */
  protected void clearFields() {
    // #TODO Passar a limpeza dos campos superiores para a classe pai. 
    login.setText("");
    name.setText("");
    emailsModel.clear();
    addEmailTf.setText("");
    login.requestFocus();
    clearRoleSelection();
    clearPermissionSelection();
    includeUserButton.setEnabled(false);
  }

  /**
   * Limpa a seleo de permisses.
   */
  private void clearPermissionSelection() {
    userInfo.setAttribute(User.PERMISSION_IDS, new Object[] {});
  }

  /**
   * Limpa a seleo de perfis.
   */
  private void clearRoleSelection() {
    userInfo.setAttribute(User.ROLE_IDS, new Object[] {});
  }

  /**
   * {@inheritDoc} <br>
   * Adiciona listeners de mudana nos dados aos campos do dilogo.
   */
  @Override
  protected void makeDialog() {
    try {
      super.makeDialog();
    }
    catch (IllegalStateException ex) {
      StandardErrorDialogs.showErrorDialog(ownerWindow, dialogTitle,
        ex.getMessage());
    }
  }

  /**
   * Constri o combo de grupos de usurios disponveis, ordenados
   * alfabeticamente.
   * 
   * @return o combo de grupos de usurios disponveis.
   */
  private JComboBox makeUserGroupComboBox() {
    JComboBox combo = new JComboBox();
    Vector<UserGroup> userGroups =
      UserGroupProxy.getAllUserGroups(userInfoDialog, dialogTitle,
        LNG.get("IAS_WAITING_ALL_USERGROUPS"));
    if (userGroups == null) {
      throw new IllegalStateException(LNG.get("ias.usergroup.retrieval_error"));
    }
    Collections.sort(userGroups, UserGroup.getNameComparator());
    int userGroupSize = userGroups.size();
    for (int i = 0; i < userGroupSize; i++) {
      combo.addItem(new UserGroupName(userGroups.get(i)));
    }
    if (userGroupSize > 0) {
      combo.setSelectedIndex(0);
    }
    return combo;
  }

  /**
   * Cria o componente responsvel por informar se deve ser forado login local.
   * 
   * @return componente responsvel por informar se deve ser forado login
   *         local.
   */
  private JCheckBox makeForceLocalLoginCheckBox() {
    JCheckBox checkbox = new JCheckBox(LNG.get("IAS_USER_FORCE_LOCAL_LOGIN"));
    return checkbox;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<JComponent[]> createDataComponents() {
    List<JComponent[]> components = super.createDataComponents();
    JLabel userGroupLabel = new JLabel(LNG.get("IAS_USER_USERGROUP"));
    JLabel unLabel = new JLabel("");
    userGroupComboBox = makeUserGroupComboBox();

    groupLabel = makeGroupLabel();

    components.add(new JComponent[] { userGroupLabel, userGroupComboBox });
    if (!ClientRemoteMonitor.getInstance().isLocalLogin()) {
      JLabel forceLoginLabel = new JLabel(LNG.get("IAS_USER_AUTHENTICATION"));
      forceLocalLoginCheckBox = makeForceLocalLoginCheckBox();
      components.add(new JComponent[] { forceLoginLabel,
          forceLocalLoginCheckBox });
    }

    components.add(new JComponent[] { unLabel, groupLabel });
    return components;
  }

  private JLabel makeGroupLabel() {
    JLabel label = new JLabel();
    label.setVisible(false);
    return label;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void addDataComponentsListeners() {
    super.addDataComponentsListeners();
    ItemListener itemListener = new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        fireOnDataChanged();
      }
    };
    if (userGroupComboBox != null) {
      userGroupComboBox.addItemListener(itemListener);
    }
    if (forceLocalLoginCheckBox != null) {
      forceLocalLoginCheckBox.addItemListener(itemListener);
    }
  }

  /**
   * {@inheritDoc} <br>
   * Se o usurio confirmar a incluso, verifica se os dados requeridos esto
   * preenchidos (ver {@link UserInfoDialog#requiredFieldsAreFilled()}. Se no
   * estiverem, pede o preenchimento e cancela o fechamento.
   */
  @Override
  protected boolean confirmClose() {
    // #TODO Usar constantes para todos os tags de propriedades.
    Object[] options =
      { LNG.get("IAS_INCLUDE"), LNG.get("IAS_NOT_INCLUDE"),
          LNG.get("IAS_CANCEL") };
    int selected =
      JOptionPane.showOptionDialog(userInfoDialog,
        LNG.get("IAS_USER_ADD_CONFIRMATION"),
        LNG.get("IAS_USER_ADD_NOT_DONE_TITLE"), JOptionPane.DEFAULT_OPTION,
        JOptionPane.INFORMATION_MESSAGE, null, options, options[0]);
    if (options[selected].equals(LNG.get("IAS_INCLUDE"))) {
      if (requiredFieldsAreFilled()) {
        return createUser();
      }
      StandardErrorDialogs.showErrorDialog(userInfoDialog, dialogTitle,
        LNG.get("IAS_FILL_ALL_FIELDS"));
      return false;
    }
    if (options[selected].equals(LNG.get("IAS_NOT_INCLUDE"))) {
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   *
   * Acrescenta os botes de seleo de permisses e seleo de perfis ao
   * painel.
   */
  @Override
  protected JPanel makeMiddlePanel() {
    JPanel panel = new JPanel();
    roleButton = makeRoleButton();
    panel.add(roleButton);
    permissionButton = makePermissionButton();
    panel.add(permissionButton);

    final JComponent[] buttons =
      new JComponent[] { roleButton, permissionButton };
    GUIUtils.matchPreferredSizes(buttons);
    return panel;
  }

  /**
   * Cria o boto de seleo de permisses, atribuindo-lhe como ao a criao
   * do dilogo de seleo de permisses, alm habilitao do boto de incluso
   * aps a seleo, caso os requisitos estejam preenchidos.
   * 
   * @return boto de seleo de permisses.
   */
  private JButton makePermissionButton() {
    JButton button = new JButton(LNG.get("IAS_USER_PERMISSIONS_BUTTON"));
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        new UserPermissionDefinitionDialog(userInfoDialog, userInfo, null, true)
          .showDialog();
        enableIncludeUserButtonIfAllowed();
      }
    });
    return button;
  }

  /**
   * Cria o boto de seleo de perfis, atribuindo-lhe como ao a criao do
   * dilogo de seleo de perfis, alm da habilitao do boto de incluso aps
   * a seleo, caso os requisitos estejam preenchidos.
   * 
   * @return boto de seleo de perfis.
   */
  private JButton makeRoleButton() {
    JButton button = new JButton(LNG.get("IAS_USER_ROLES_BUTTON"));
    button.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        new UserRoleDefinitionDialog(userInfoDialog, userInfo, null, true)
          .showDialog();
        enableIncludeUserButtonIfAllowed();
      }
    });
    return button;
  }

  /**
   * Adiciona um KeyListener ao campo de login usando o callBack Object para
   * devolver o contedo do campo  classe que o chamou.
   * 
   * @param callBack o callback.
   */
  protected void sendUserInfoCallBackObject(
    final UserInfoCallBackInterface callBack) {
    if (callBack == null) {
      return;
    }

    //Se tecla "Enter" for pressionada
    getLogin().addKeyListener(new KeyAdapter() {
      @Override
      public void keyPressed(KeyEvent e) {
        if (e.getKeyCode() == 10 && getLogin().getText().length() > 0) {
          addUserInfo(callBack.obtainUserInfo(login.getText()));
        }
      }
    });
    //Se campo de texto perdeu o foco
    getLogin().addFocusListener(new FocusAdapter() {
      @Override
      public void focusLost(FocusEvent e) {
        if (getLogin().getText().length() > 0) {
          addUserInfo(callBack.obtainUserInfo(login.getText()));
        }
      }
    });
  }

  /**
   * Carrega as informaes de usurio nos campos de texto.
   * 
   * @param info as informaes de usurio
   */
  protected void addUserInfo(UserInfo info) {
    if (info == null) {
      return;
    }

    if (info.getAttributes().get(User.NAME) != null) {
      name.setText(info.getAttributes().get(User.NAME).toString());
    }

    //Esvazia o componente de emails antes de carrega-lo
    emailsModel.removeAllElements();
    if (info.getAttributes().get(User.EMAILS) != null) {
      for (String email : (String[]) info.getAttributes().get(User.EMAILS)) {
        emailsModel.addElement(email.toString());
      }
    }

    //Primeiramente desmarca qualquer seleo
    userGroupComboBox.setSelectedIndex(-1);
    if (info.getAttributes().get(User.USER_GROUP) != null) {
      for (int i = 0; i < userGroupComboBox.getModel().getSize(); i++) {
        UserGroupName groupName =
          (UserGroupName) userGroupComboBox.getModel().getElementAt(i);
        if (groupName.getUserGroup().getUserGroupInfo().name.equals(info
          .getAttribute(User.USER_GROUP))) {
          userGroupComboBox.setSelectedIndex(i);
          break;
        }
      }
    }

    //Retorna se algum elemento da combo tiver sido selecionada
    if (userGroupComboBox.getSelectedIndex() != -1) {
      return;
    }

    String hasLabel = null;
    try {
      Configuration conf =
        ConfigurationManager.getInstance().getConfiguration(
          csbase.client.ias.IncludeUserInfoDialog.class);
      hasLabel = conf.getOptionalProperty("show.label");
    }
    catch (ConfigurationManagerException e) {
      e.printStackTrace();
    }

    if (hasLabel != null && hasLabel.equals("true")) {
      groupLabel.setText("UN: " + info.getAttribute(User.USER_GROUP));
      groupLabel.setVisible(true);
      super.getUserInfoDialog().pack();

    }
    else {
      groupLabel.setVisible(false);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @return true se e somente se todos os campos requeridos estiverem
   *         preenchidos, false caso contrrio.
   */
  @Override
  protected boolean unsavedDataInput() {
    return requiredFieldsAreFilled();
  }

  /**
   * Habilita o boto de incluir somente se as condies para incluso forem
   * atendidas (campos requeridos preenchidos). Caso contrrio, desabilita.
   */
  private void enableIncludeUserButtonIfAllowed() {
    includeUserButton.setEnabled(requiredFieldsAreFilled());
  }

  /**
   * {@inheritDoc}<br>
   * Chama o {@link #enableIncludeUserButtonIfAllowed() mtodo} que trata o
   * estado habilitado do boto de incluir usurio.
   */
  @Override
  protected void fireOnDataChanged() {
    enableIncludeUserButtonIfAllowed();
  }

  @Override
  protected boolean requiredFieldsAreFilled() {
    return super.requiredFieldsAreFilled() &&
      userGroupComboBox.getSelectedIndex() != -1;
  }
}
