/*
 * $Author: oikawa $ $Date: 2014-03-19 11:16:56 -0300 (Wed, 19 Mar 2014) $
 * $Release: $
 */
package csbase.client.util.sga;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.border.TitledBorder;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.kernel.ClientException;
import csbase.client.remote.srvproxies.SGAProxy;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;

/**
 * Define um painel para seleo de SGAs.
 * 
 * @author ururahy
 * 
 */
public abstract class SGASelectionPanel extends JPanel {
  /**
   * A lista de listeners de seleo de SGAs.
   */
  private List<SGASelectionListener> sgaSelectionListeners;

  /**
   * O conjunto de plataformas disponveis para seleo.
   */
  protected Set<String> platformSet;

  /**
   * O conjunto de requisitos do comando.
   */
  protected Set<String> requirements;

  /**
   * O painel de filtragem de plataformas.
   */
  protected JPanel platformFilterPanel;

  /**
   * O rtulo da filtragem de plataformas.
   */
  protected JLabel platformFilterLabel;

  /**
   * A caixa de seleo de filtragem de plataformas.
   */
  protected JComboBox platformFilterComboBox;

  /**
   * O texto (localizado) que define o item de seleo
   */
  protected String allPlatformsText;

  /**
   * Boto de opo que indica seleo manual de sgas.
   */
  protected JRadioButton manualServerRadioButton;

  /**
   * Boto de opo que indica seleo automtica de sgas.
   */
  protected JRadioButton automaticServerRadioButton;

  /**
   * Grupo de botes de opo que indicam o tipo de seleo de sgas (manual ou
   * automtica).
   */
  private ButtonGroup serverSelectionButtonGroup;

  /**
   * O configurador do algoritmo a ser executado.
   */
  protected AlgorithmConfigurator configurator;

  /**
   * Borda com ttulo do painel de seleo.
   */
  private TitledBorder serverBorder;

  /**
   * Nome da propriedade que indica a escolha automtica de sgas como a opo
   * default.
   */
  private static final String AUTOMATIC_SELECTION = "automatic.selection";

  /**
   * Construtor.
   * 
   * @param configurator O configurador do algoritmo a ser executado.
   */
  protected SGASelectionPanel(AlgorithmConfigurator configurator) {
    this.configurator = configurator;
    this.allPlatformsText = LNG.get("SGASelectionPanel.label.all_platforms");
    this.platformSet = new HashSet<String>();
    this.platformSet.addAll(configurator.getPlatforms());
    this.requirements = new HashSet<String>();
    this.requirements.addAll(configurator.getRequirements());
    this.sgaSelectionListeners = new LinkedList<SGASelectionListener>();
    createGUIComponents();
  }

  /**
   * Obtm a coleo de servidores escolhidos manualmente pelo usurio.
   * 
   * @return A coleo de servidores selecionada pelo usurio.
   */
  abstract Collection<SGASet> getManualServerCollection();

  /**
   * Obtm o servidor selecionado (em modo seleo simples) ou o primeiro
   * servidor selecionado (em modo seleo mltipla).
   * 
   * @return O SGASet do servidor selecionado.
   */
  abstract SGASet getManualSelectedServer();

  /**
   * Indica se existe algum sga disponvel para execuo.
   * 
   * @return verdadeiro se existe algum sga disponvel para execuo ou falso
   *         caso contrrio.
   */
  abstract boolean hasAvailableServers();

  /**
   * Habilita/Desabilita componentes da interface grfica com o usurio de
   * acordo com as interaes do usurio com essa interface. Por exemplo, se a
   * seleo automtica de servidores for habilitada, a caixa de seleo manual
   * de SGAs deve ser desabilitada.
   */
  abstract void updateEnabledServers();

  /**
   * Recarrega o combo box dos SGAs, filtrados pela plataforma selecionada.
   */
  abstract void reloadSGAs();

  /**
   * Sinaliza se a seleo de servidores est completa. Se a resposta for
   * afirmativa e a configurao de parmetros do algoritmo selecionado tambm
   * estiver sido concluda, a execuo do algoritmo est pronta para ocorrer.
   * 
   * @return Se a seleo de servidores est completa ou no.
   */
  public abstract boolean isSelectionReady();

  /**
   * Popula o painel de seleo de sgas. Cada tipo de algoritmo (simples ou
   * mltiplo) possui um tipo de painel.
   */
  public abstract void populateGUIComponents();

  /**
   * Adiciona um listener a este painel.
   * 
   * @param listener O listener.
   * 
   * @throws IllegalArgumentException Se o parmetro listener estiver nulo.
   */
  public void addSGASelectionListener(SGASelectionListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.sgaSelectionListeners.add(listener);
  }

  /**
   * Obtm a plataforma selecionada.
   * 
   * @return A plataforma selecionada.
   */
  public String getSelectedPlatform() {
    return (String) this.platformFilterComboBox.getSelectedItem();
  }

  /**
   * <p>
   * Obtm uma coleo no-modificvel de servidores selecionados, se o painel
   * estiver em modo manual.
   * </p>
   * <p>
   * Os objetos desta coleo so do tipo {@link SGASet}.
   * </p>
   * 
   * @return A coleo de servidores. Retornar uma coleo vazio se no
   *         houverem elementos selecionados.
   * 
   * @see java.util.Collections#unmodifiableCollection(java.util.Collection)
   */
  public Collection<SGASet> getSelectedServerCollection() {
    if (isManualServerSelection()) {
      return this.getManualServerCollection();
    }
    return null;
  }

  /**
   * Obtm o servidor selecionado (em modo seleo simples) ou o primeiro
   * servidor selecionado (em modo seleo mltipla), se o painel estiver em
   * modo manual.
   * 
   * @return O servidor selecionado.
   */
  public SGASet getSelectedServer() {
    return this.getManualSelectedServer();
  }

  /**
   * Carrega o combo box dos SGAs, filtrados pela plataforma selecionada.
   * 
   * @return .
   * 
   * @throws ClientException Em caso de erro.
   */
  protected Vector<SGASet> loadServers() throws ClientException {
    Map<String, SGASet> sgaSetMap = new HashMap<String, SGASet>();
    Collection<String> sgaNames = SGAProxy.getAllSGANames();
    if (sgaNames == null) {
      throw new ClientException(
        LNG.get("SGASelectionPanel.exception.load_sgas"));
    }
    for (String sgaName : sgaNames) {
      SGASet sgaSet = SGAProxy.getSGASet(sgaName);
      sgaSetMap.put(sgaName, sgaSet);
    }
    Vector<String> selectedPlataforms = new Vector<String>();
    String selectedPlatform = getPlatformFilter();
    if (selectedPlatform == null) {
      selectedPlataforms.addAll(this.platformSet);
    }
    else {
      selectedPlataforms.add(selectedPlatform);
    }
    Vector<SGASet> validSGAs = new Vector<SGASet>();
    for (String sgaName : sgaSetMap.keySet()) {
      SGASet sgaSet = sgaSetMap.get(sgaName);
      if (sgaSet != null && sgaSet.isPlatformSupported(selectedPlataforms)
        && sgaSet.hasRequirements(this.requirements)
        && sgaSet.mayExecuteCommand()) {
        validSGAs.add(sgaSet);
      }
    }
    Collections.sort(validSGAs);
    return validSGAs;
  }

  /**
   * Remove o listener da lista de observadores deste painel.
   * 
   * @param listener O listener.
   * 
   * @throws IllegalArgumentException Se o parmetro listener estiver nulo.
   */
  public void removeSGASelectionListener(SGASelectionListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.sgaSelectionListeners.remove(listener);
  }

  /**
   * Verifica se a seleo de servidores  manual.
   * 
   * @return .
   */
  public boolean isManualServerSelection() {
    return this.manualServerRadioButton.isEnabled()
      && this.manualServerRadioButton.isSelected();
  }

  /**
   * Este mtodo retorna o filtro de plataformas selecionado ou null caso no
   * haja nenhuma restrio selecionada..
   * 
   * @return O filtro de plataformas selecionado.
   */
  public String getPlatformFilter() {
    String selectedPlatform = getSelectedPlatform();
    if (selectedPlatform.equals(allPlatformsText)) {
      return null;
    }
    return selectedPlatform;
  }

  /**
   * Cria os componentes da GUI.
   */
  private void createGUIComponents() {
    this.serverBorder =
      new TitledBorder(BorderFactory.createEtchedBorder(),
        LNG.get("SGASelectionPanel.label.server"));
    this.platformFilterLabel =
      new JLabel(LNG.get("SGASelectionPanel.label.platform_filter"));
    this.platformFilterComboBox = makePlatformFilterComboBox();
    createPlatformFilterPanel();
    boolean automaticSelection = false;
    try {
      Configuration cnf =
        ConfigurationManager.getInstance().getConfiguration(this.getClass());
      automaticSelection =
        cnf.getOptionalBooleanProperty(AUTOMATIC_SELECTION, false);
    }
    catch (ConfigurationManagerException e) {
      automaticSelection = false;
    }

    this.manualServerRadioButton =
      new JRadioButton(LNG.get("SGASelectionPanel.radiobutton.manual_server"));
    this.manualServerRadioButton.setSelected(!automaticSelection);
    this.manualServerRadioButton
      .addItemListener(new ManualServerRadioButtonItemListener());
    this.manualServerRadioButton.setEnabled(false);
    this.automaticServerRadioButton =
      new JRadioButton(
        LNG.get("SGASelectionPanel.radiobutton.automatic_server"));
    this.automaticServerRadioButton.setSelected(automaticSelection);
    this.automaticServerRadioButton
      .addItemListener(new ManualServerRadioButtonItemListener());
    this.automaticServerRadioButton.setEnabled(false);
    this.serverSelectionButtonGroup = new ButtonGroup();
    this.serverSelectionButtonGroup.add(this.manualServerRadioButton);
    this.serverSelectionButtonGroup.add(this.automaticServerRadioButton);
    this.setBorder(this.serverBorder);
  }

  /**
   * Cria um painel para a insero do filtro de plataformas.
   */
  private void createPlatformFilterPanel() {
    this.platformFilterPanel = new JPanel();
    this.platformFilterPanel.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.WEST;
    c.insets = new Insets(2, 2, 2, 2);
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0.0;
    c.weighty = 0.0;
    c.fill = GridBagConstraints.NONE;
    this.platformFilterPanel.add(this.platformFilterLabel, c);
    c.weightx = 1.0;
    c.gridx++;
    this.platformFilterPanel.add(this.platformFilterComboBox, c);
  }

  /**
   * Notifica os listeners que houve alterao na lista de servidores.
   */
  protected void fireChangedServer() {
    Iterator<SGASelectionListener> listenerIterator =
      this.sgaSelectionListeners.iterator();
    while (listenerIterator.hasNext()) {
      SGASelectionListener listener = listenerIterator.next();
      listener.wasChangedServer();
    }
  }

  /**
   * Cria um componente do tipo JComboBox para a seleo de plataformas.
   * 
   * @return O componente JComboBox criado e populado com as plataformas
   *         disponveis.
   */
  private JComboBox makePlatformFilterComboBox() {
    JComboBox combo = null;
    Vector<String> allPlatforms = new Vector<String>(platformSet);
    Collections.sort(allPlatforms);
    allPlatforms.add(0, allPlatformsText);
    combo = new JComboBox(allPlatforms);
    combo.setSelectedItem(allPlatformsText);
    combo.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        reloadSGAs();
        fireChangedPlatform();
      }
    });
    return combo;
  }

  /**
   * Notifica os listeners que houve alterao na seleo de plataformas.
   */
  protected void fireChangedPlatform() {
    Iterator<SGASelectionListener> listenerIterator =
      this.sgaSelectionListeners.iterator();
    while (listenerIterator.hasNext()) {
      SGASelectionListener listener = listenerIterator.next();
      listener.wasChangedPlatform();
    }
  }

  /**
   * Define um listener de mudanas no tipo da seleo de plataformas (manual ou
   * automtica).
   */
  private final class ManualServerRadioButtonItemListener implements
    ItemListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void itemStateChanged(ItemEvent e) {
      updateEnabledServers();
      fireChangedServer();
    }
  }
}
