package csbase.client.util.table;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * Modelo de Linhas Abstrato.
 * 
 * @param <R> O valor armazenado na linha.
 *
 * @author lmoreira
 */
public abstract class AbstractRowModel<R> implements RowModel<R> {

  /**
   * Os observadores.
   */
  private List<RowModelListener<R>> listeners;
  
  /**
   * Cria o modelo.
   */
  protected AbstractRowModel() {
    listeners = new LinkedList<RowModelListener<R>>();
  }
  
  public void addRowModelListener(RowModelListener<R> listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Dispara o evento {@link RowModelListener#rowWasAdded(Object, int)}.
   * 
   * @param newRowValue O valor da linha.
   * @param newRowIndex O ndice da linha.
   */
  public final void fireRowWasAdded(R newRowValue, int newRowIndex) {
    for (RowModelListener<R> listener : listeners) {
      listener.rowWasAdded(newRowValue, newRowIndex);
    }
  }

  /**
   * Dispara o evento {@link RowModelListener#rowWasRemoved(Object, int)}.
   * 
   * @param oldRowValue O valor da linha.
   * @param oldRowIndex O ndice da linha.
   */
  public final void fireRowWasRemoved(R oldRowValue, int oldRowIndex) {
    for (RowModelListener<R> listener : listeners) {
      listener.rowWasRemoved(oldRowValue, oldRowIndex);
    }
  }

  public boolean isEmpty() {
    return getRowCount() == 0;
  }
  
  public Iterator<R> iterator() {
    return getRows().iterator();
  }

  public boolean removeRowModelListener(RowModelListener<R> listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }
}
