package csbase.client.applications;

import javax.swing.JPanel;

/**
 * Painel genrico de uma aplicao, que contm algumas facilidades comuns a
 * todos os painis de uma aplicao, commo por exemplo, a obteno do valor de
 * uma propriedade de internacionalizao.
 * 
 * @param <T> Tipo da aplicao que est criando o painel.
 */
public abstract class AbstractSimpleApplicationPanel<T extends Application>
  extends JPanel {
  /**
   * Aplicao que contm o painel.
   */
  private T application;

  /**
   * Constri o painel.
   * 
   * @param application a aplicao
   */
  public AbstractSimpleApplicationPanel(T application) {
    this.application = application;
  }

  /**
   * Obtm a referncia para a aplicao.
   * 
   * @return a aplicao
   */
  final public T getApplication() {
    return application;
  }

  /**
   * Constri o painel.
   */
  protected abstract void buildPanel();

  /**
   * Obtm o texto de uma propriedade de internacionalizao, usando uma chave
   * sem o prefixo padro que  o nome da classe. Obtm dinamicamente o nome da
   * classe para usar como prefixo da busca.
   * 
   * @param key propriedade desejada, sem o prefixo padro.
   * @return o texto de internacionalizao
   */
  final public String getClassString(final String key) {
    @SuppressWarnings("unchecked")
    final Class<? extends AbstractSimpleApplicationPanel<T>> clazz =
      (Class<? extends AbstractSimpleApplicationPanel<T>>) this.getClass();
    if (!application.hasClassString(clazz, key)) {
      final String dummyText = "<<" + clazz.getSimpleName() + "." + key + ">>";
      return dummyText;
    }
    final String text = application.getClassString(clazz, key);
    return text;
  }

  /**
   * Obtm o texto de uma propriedade de internacionalizao
   * 
   * @param key propriedade desejada
   * @return o texto de internacionalizao
   */
  final public String getString(final String key) {
    if (!application.hasString(key)) {
      final String dummyText = "<<" + key + ">>";
      return dummyText;
    }
    final String text = application.getString(key);
    return text;
  }

  /**
   * Obtm o texto de uma propriedade de internacionalizao que contenha
   * parmetros, usando uma chave sem o prefixo padro que  o nome da classe.
   * Obtm dinamicamente o nome da classe para usar como prefixo da busca.
   * 
   * @param key propriedade para a mensagem que contm parmetros.
   * @param args argumentos para a mensagem.
   * 
   * @return mensagem preenchida pelos argumentos
   * 
   * @see java.text.MessageFormat
   */
  public final String getClassString(final String key, final Object[] args) {
    @SuppressWarnings("unchecked")
    final Class<? extends AbstractSimpleApplicationPanel<T>> clazz =
      (Class<? extends AbstractSimpleApplicationPanel<T>>) this.getClass();
    if (!application.hasClassString(clazz, key)) {
      final String dummyText = "<<" + clazz.getSimpleName() + "." + key + ">>";
      return dummyText;
    }
    final String text = application.getClassString(this.getClass(), key, args);
    return text;
  }

  /**
   * Obtm o texto de uma propriedade de internacionalizao que contenha
   * parmetro.
   * 
   * @param key propriedade para a mensagem que contm parmetros.
   * @param args argumentos para a mensagem.
   * 
   * @return mensagem preenchida pelos argumentos
   * 
   * @see java.text.MessageFormat
   */
  public final String getString(final String key, final Object[] args) {
    if (!application.hasString(key)) {
      final String dummyText = "<<" + key + ">>";
      return dummyText;
    }
    final String text = application.getString(key, args);
    return text;
  }
}
