package csbase.client.desktop;

import java.awt.Dialog.ModalityType;
import java.awt.Window;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.util.StandardErrorDialogs;

/**
 * A classe <code>Task</code> modela uma tarefa possivelmente demorada executada
 * no cliente. Uma thread  criada para a execuo da tarefa, para que eventos
 * de interface (como redesenho) possam ser atendidos. As interaes de teclado
 * e mouse, contudo, so bloqueadas. Caso a durao da tarefa ultrapasse um
 * valor limite configurvel, um dilogo modal contendo uma barra de progresso
 * ser exibido. Este dilogo ser fechado automaticamente ao final da tarefa.
 * Aps um intervalo de valor configurvel, um boto para fechamento do dilogo
 *  oferecido ao usurio. O texto do boto (Fechar ou Cancelar)  tambm
 * configurvel, pois depende do tipo de tarefa, e do efeito de seu fechamento.
 * 
 * Essa classe permite aninhamento. Isto , se dentro do mtodo performTask uma
 * outra tarefa for criada e executada, o retorno visual para seu acompanhamento
 *  dado na mesma janela da tarefa anterior, evitando que mltiplas janelas de
 * acompanhamento sejam abertas.
 * 
 *  importante notar que tarefas aninhadas devem, obrigatoriamente, usar os
 * mtodos beforeTaskUI e afterTaskUI para qualquer cdigo que atue sobre a
 * interface grfica.
 * 
 * @param <R> A classe do resultado da tarefa.
 * @author Tecgraf/PUC-Rio
 */
public abstract class Task<R> extends tecgraf.javautils.gui.Task<R> {

  /**
   * Nome da propriedade que indica opcionalmente o tempo esperado para o
   * aparecimento do boto de cancelar aps o lanamento da tarefa. A
   * propriedade  definida com base no {@link ConfigurationManager} do sistema.
   * 
   * @see #DEFAULT_CANCEL_DELAY
   */
  private static final String CANCEL_DELAY_PROP_NAME = "cancel.delay.seconds";

  /**
   * Construtor.
   */
  protected Task() {
    super();
  }

  /**
   * Construtor.
   * 
   * @param modality tipo de bloqueio das janelas
   */
  protected Task(ModalityType modality) {
    super(modality);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Window getDefaultParentWindow() {
    DesktopFrame desktop = DesktopFrame.getInstance();
    if (desktop != null) {
      return DesktopFrame.getInstance().getDesktopFrame();
    }
    else {
      return super.getDefaultParentWindow();
    }
  }

  /**
   * Tratamento de erros especfico para cada tipo de tarefa. Esse tratamento 
   * chamado na EDT. Essa implementao representa um tratamento mnimo, apenas
   * para erros no cliente. Qualquer exceo lanada durante a execuo do
   * mtodo performTask  repassada para este mtodo para tratamento.
   * 
   * @param error - exceo que representa o erro durante execuo da tarefa.
   */
  @Override
  protected void handleError(Exception error) {
    StandardErrorDialogs.showErrorDialog(parentWindow, taskTitle,
      LNG.get("desktop.msg.clientbug"), error, getAdditionalInfo());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected int getCancelDelaySecs() {
    int defaultCancelDelaySecs = super.getCancelDelaySecs();
    try {
      final ConfigurationManager cnfManager =
        ConfigurationManager.getInstance();
      final Configuration configuration =
        cnfManager.getConfiguration(this.getClass());
      final int delay =
        configuration.getOptionalIntegerProperty(CANCEL_DELAY_PROP_NAME,
          defaultCancelDelaySecs);
      return delay;
    }
    catch (ConfigurationManagerException e) {
      return defaultCancelDelaySecs;
    }
  }

}
