/*
 * $Id$
 */
package csbase.client.facilities.algorithms.executor;

import java.awt.Window;
import java.io.IOException;
import java.io.InputStream;
import java.rmi.RemoteException;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.flowapplication.FlowApplication;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.client.desktop.RemoteTask;
import csbase.client.facilities.algorithms.parameters.FlowParametersValues;
import csbase.client.facilities.algorithms.parameters.NodeParametersValues;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.sga.CommandRequestedListener;
import csbase.exception.OperationFailureException;
import csbase.exception.ParseException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.ClientFile;
import csbase.logic.CommandInfo;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfigurator.ConfiguratorType;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.flows.Flow;
import csbase.logic.algorithms.flows.FlowAlgorithmParser;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;
import tecgraf.javautils.core.lng.LNG;

/**
 * Executor programtico de comandos do tipo fluxo de algoritmos.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class FlowCommandExecutor extends AbstractCommandExecutor {

  /**
   * Obtm o configurador do fluxo de algoritmos.
   * 
   * @param algorithmName nome do algoritmo.
   * @param versionId verso do algoritmo.
   * @param owner janela-me da ao.
   * 
   * @return o configurador do fluxo de algoritmos, ou null, caso no seja um
   *         configurador de fluxo
   */
  public FlowAlgorithmConfigurator getConfigurator(String algorithmName,
    AlgorithmVersionId versionId, Window owner) {

    CommandConfiguration commandConfig =
      new CommandConfiguration(algorithmName, versionId, owner);

    final AlgorithmConfigurator configurator = commandConfig.getConfigurator();
    final ConfiguratorType cnfType = configurator.getConfiguratorType();
    if (cnfType == ConfiguratorType.FLOW) {
      return (FlowAlgorithmConfigurator) configurator;
    }
    return null;
  }

  /**
   * Executa um comando a partir de um configurador de um fluxo de algoritmos
   * que esteja ou no instalado no servidor.
   * 
   * @param flow o fluxo de algoritmos
   * @param paramValuesList lista dos valores de parmetros dos algoritmos a
   *        serem inicializados no fluxo, ou null, caso j estejam preenchidos
   * @param description descrio do comando
   * @param openFlowApplication se true, a aplicao a ser aberta para
   *        edio/execuo do fluxo  a de Construtor de Fluxos, caso
   *        contrrio, abre-se o Executor de Algoritmos
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da ao.
   * @param observers Observadores remotos do comando executado
   * 
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   * 
   * @throws RemoteException caso tenha havido falha na chamada remota
   * @throws OperationFailureException caso o escalonador de comandos no esteja
   *         inicializado
   */
  public boolean executeIterativeCommand(Flow flow,
    Map<Integer, NodeParametersValues> paramValuesList, String description,
    boolean openFlowApplication, String sgaServerName, Window owner,
    CommandObserver... observers) throws RemoteException,
    OperationFailureException {
    return executeFlowApplication(flow, paramValuesList, description, owner);
  }

  /**
   * Executa um comando a partir de um configurador de um fluxo de algoritmos
   * que esteja ou no instalado no servidor.
   * 
   * @param flow o fluxo de algoritmos
   * @param paramValuesList lista dos valores de parmetros dos algoritmos a
   *        serem inicializados no fluxo, ou null, caso j estejam preenchidos
   * @param description descrio do comando
   * @param openFlowApplication se true, a aplicao a ser aberta para
   *        edio/execuo do fluxo  a de Construtor de Fluxos, caso
   *        contrrio, abre-se o Executor de Algoritmos
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da ao.
   * @param observers Observadores remotos do comando executado
   * 
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   * 
   * @throws RemoteException caso tenha havido falha na chamada remota
   * @throws OperationFailureException caso o escalonador de comandos no esteja
   *         inicializado
   */
  public String executeCommand(Flow flow,
    Map<Integer, NodeParametersValues> paramValuesList, String description,
    boolean openFlowApplication, String sgaServerName, Window owner,
    CommandObserver... observers) throws RemoteException,
    OperationFailureException {

    Graph graph = createGraph(flow, owner);
    if (!configureGraphNodesValues(graph, paramValuesList, owner)) {
      return null;
    }
    FlowAlgorithmConfigurator conf =
      new FlowAlgorithmConfigurator(graph.toFlow());

    final String executed =
      executeCommand(conf, description, sgaServerName, owner, observers);
    return executed;
  }

  /**
   * Executa um comando de um fluxo de algoritmos que esteja instalado no
   * servidor.
   * 
   * @param paramValuesList lista dos valores de parmetros dos algoritmos a
   *        serem inicializados no fluxo, ou null, caso j estejam preenchidos
   * @param description descrio do comando
   * @param openFlowApplication se true, a aplicao a ser aberta para
   *        edio/execuo do fluxo  a de Construtor de Fluxos, caso
   *        contrrio, abre-se o Executor de Algoritmos
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da ao.
   * @param observers Observadores remotos do comando executado
   * 
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   * 
   * @throws RemoteException caso tenha havido falha na chamada remota
   * @throws OperationFailureException caso o escalonador de comandos no esteja
   *         inicializado
   */
  public String executeCommand(FlowParametersValues paramValuesList,
    String description, boolean openFlowApplication, String sgaServerName,
    Window owner, CommandObserver... observers) throws RemoteException,
    OperationFailureException {
    FlowAlgorithmConfigurator flowConfigurator =
      getConfigurator(paramValuesList.getFlowName(),
        paramValuesList.getFlowVersionId(), owner);
    return executeCommand(flowConfigurator.getFlow(),
      paramValuesList.getParametersValuesList(), description,
      openFlowApplication, sgaServerName, owner, observers);
  }

  /**
   * Executa um fluxo de algoritmos a partir de um arquivo (.flx). A execuo do
   * fluxo  feita programaticamente, com a escolha automtica do sga, e sem
   * necessidade de ter um fluxo instalado no servidor.
   * 
   * @param file arquivo que contm o fluxo de algoritmos
   * @param paramValuesList lista dos valores de parmetros dos algoritmos a
   *        serem inicializados no fluxo, ou null, caso j estejam preenchidos
   * @param description descrio do comando
   * @param openFlowApplication se true, a aplicao a ser aberta para
   *        edio/execuo do fluxo  a de Construtor de Fluxos, caso
   *        contrrio, abre-se o Executor de Algoritmos
   * @param sgaServerName nome do servidor do SGA a ser usado para execuo. Se
   *        for null, indica que a seleo do servidor ser automtica.
   * @param owner janela-me da ao.
   * @param observers Observadores remotos do comando executado
   * @return retorna true, se o comando foi executado com sucesso, caso
   *         contrrio, retorna false
   * 
   * @throws IOException em caso de erro de IO
   * @throws OperationFailureException no caso de falha
   * @throws ParseException no caso de erro de parser do fluxo
   */
  public String executeCommand(ClientFile file,
    Map<Integer, NodeParametersValues> paramValuesList, String description,
    boolean openFlowApplication, String sgaServerName, Window owner,
    CommandObserver... observers) throws ParseException,
    OperationFailureException, IOException {
    try (InputStream inputStream = file.getInputStream()) {
      FlowAlgorithmParser parser = new FlowAlgorithmParser();
      Flow flow = parser.read(inputStream);
      return executeCommand(flow, paramValuesList, description,
        openFlowApplication, sgaServerName, owner, observers);
    }
  }

  /**
   * Abre a aplicao do Construtor de Fluxo para a execuo de um fluxo de
   * algoritmos que esteja ou no instalado no servidor, dependendo de como o
   * configurador do algoritmo foi criado.
   * 
   * O usurio pode interagir tanto na parametrizao do algoritmo, quanto na
   * submisso do comando para execuo.
   * 
   * @param flow o fluxo de algoritmos
   * @param paramValuesList lista dos valores de parmetros dos algoritmos a
   *        serem inicializados no fluxo
   * @param commandDescription descrio do comando
   * @param owner janela-me da ao.
   * 
   * @return retorna true se a aplicao foi aberta com sucesso, caso contrrio,
   *         retorna false
   */
  private boolean executeFlowApplication(Flow flow,
    Map<Integer, NodeParametersValues> paramValuesList,
    String commandDescription, Window owner) {
    try {
      ApplicationManager applicationManager = ApplicationManager.getInstance();
      FlowApplication flowApp =
        applicationManager.runApplication(FlowApplication.class);
      final ApplicationFrame frame = flowApp.getApplicationFrame();
      Graph graph = createGraph(flow, frame);
      if (!configureGraphNodesValues(graph, paramValuesList, owner)) {
        applicationManager.killApplication(flowApp);
        return false;
      }

      flowApp.setGraph(graph, commandDescription);
      flowApp.hideAlgorithmTreePanel();

      flowApp.addCommandRequestedListener(new CommandRequestedListener() {
        @Override
        public void commandsWereRequested(Set<CommandInfo> submittedCommands) {
          if (!submittedCommands.isEmpty()) {
            Iterator<CommandInfo> iterator = submittedCommands.iterator();
            setCommandId(iterator.next().getId());
            notifyCommandRequestedListeners(submittedCommands);
          }
        }
      });
      return true;
    }
    catch (ApplicationException e) {
      StandardErrorDialogs.showExceptionDialog(owner, owner.getName(), e);
      return false;
    }
  }

  /**
   * Configura os valores dos parmetros do grafo que representa o fluxo. S 
   * vlido quando a execuo do fluxo  interativa e utilizando a aplicao do
   * construtor de fluxo.
   * 
   * @param graph grafo.
   * @param paramValuesList lista com os valores dos parmetros a serem
   *        inicializados nos ns do grafo
   * @param owner janela-me da ao.
   * 
   * @return retorna true, se a configurao de todos os ns do grafo foi feita
   *         com sucesso, caso contrrio, retorna false
   */
  private boolean configureGraphNodesValues(Graph graph,
    Map<Integer, NodeParametersValues> paramValuesList, Window owner) {
    if (paramValuesList == null) {
      return true;
    }
    for (Entry<Integer, NodeParametersValues> entry : paramValuesList
      .entrySet()) {
      final GraphNode graphNode = graph.getNode(entry.getKey());
      if (graphNode == null) {
        final String msgTag = "FlowCommandExecutor.msg.error.invalid.node";
        final String msg = LNG.get(msgTag);
        StandardErrorDialogs.showErrorDialog(owner, msg);
        return false;
      }
      try {
        NodeParametersValues value = entry.getValue();
        final Map<String, String> allValues = value.getAllParametersValues();
        graphNode.setParameterValuesByName(allValues);
      }
      catch (ParseException pe) {
        final String msgTag = "FlowCommandExecutor.msg.error.invalid.params";
        final String msg = LNG.get(msgTag);
        StandardErrorDialogs.showErrorDialog(owner, msg, pe);
        return false;
      }
      catch (ParameterNotFoundException e) {
        final String msgTag = "FlowCommandExecutor.msg.error.invalid.params";
        final String msg = LNG.get(msgTag);
        StandardErrorDialogs.showErrorDialog(owner, msg, e);
        return false;
      }
    }
    return true;
  }

  /**
   * Cria remotamente o grafo que representa o fluxo.
   * 
   * @param flow o fluxo
   * @param frame janela da aplicao que originou a ao.
   * @return o grafo criado
   */
  private Graph createGraph(final Flow flow, final Window frame) {
    RemoteTask<Graph> task = new RemoteTask<Graph>() {
      @Override
      protected void performTask() throws Exception {
        Graph graph = new Graph(frame, flow);
        setResult(graph);
      }
    };
    final String title = "Criando o grafo do fluxo";
    final String msg = "Carregando informaes do fluxo.";
    if (!task.execute(frame, title, msg)) {
      return null;
    }
    return task.getResult();
  }

}
