/*
 * $Id: UserManagerPanel.java 175310 2016-08-03 14:38:03Z fpina $
 */

package csbase.client.ias;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.Comparator;
import java.util.Observable;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.table.TableModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.remote.srvproxies.UserGroupProxy;
import csbase.client.remote.srvproxies.UserProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.SwingObserverWrapper;
import csbase.logic.AdministrationEvent;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserOutline;

/**
 * A classe <code>UserManagerPanel</code> representa o painel de administrao
 * dos usurios do sistema. Mostra uma tabela com a identificao e o UserGroup
 * de cada usurio. O administrador pode inclui e remover um usurio, alm de
 * modificar os dados de um usurio qualquer.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class UserManagerPanel extends ManagerPanel {
  /**
   * A janela onde esse painel est sendo usado.
   */
  private JFrame parent;

  /**
   * Boto importar
   */
  private JButton importCSVButton;

  /**
   * Boto exportar
   */
  private JButton exportCVSButton;

  /**
   * Dilogo para incluso de usurio
   **/
  private UserInfoDialog dialog;

  /**
   * Objeto de callBack para ser usado do dialog
   */
  private UserInfoCallBackInterface callBack;
  
  /** Formatador de data. */
  public static SimpleDateFormat dateFormat = new SimpleDateFormat("dd/MM/yyyy");

  /**
   * Remove os observadores de usurios e UserGroups.
   */
  @Override
  public void beforeClose() {
    User.deleteObserver(new SwingObserverWrapper(this));
    UserGroup.deleteObserver(new SwingObserverWrapper(this));
  }

  /**
   * Exibe a janela de incluso de usurio.
   */
  @Override
  public void add() {
    Vector<UserGroup> userGroups =
      UserGroupProxy.getAllUserGroups(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ALL_USERGROUPS"));
    if ((userGroups == null) || (userGroups.size() == 0)) {
      StandardDialogs.showInfoDialog(parent, parent.getTitle(),
        LNG.get("IAS_NO_USERGROUPS"));
      return;
    }
    this.dialog = new IncludeUserInfoDialog(parent);
    ((IncludeUserInfoDialog) dialog).sendUserInfoCallBackObject(callBack);
    this.dialog.setVisible(true);

  }

  /**
   * Exibe a janela de modificao de usurio.
   * 
   * @param id identificador do usurio a ser modificado.
   */
  @Override
  public void modify(Object id) {
    if (User.getLoggedUser().isAdmin()) {
      dialog = new AdminModifyUserInfoDialog(parent, id);
    }
    else {
      dialog = new ModifyUserInfoDialog(parent, id);
    }
    dialog.setVisible(true);
  }

  /**
   * Exibe a janela de remoo de usurio.
   * 
   * @param id identificador do usurio a ser removido
   */
  @Override
  public void delete(Object id) {
    User user =
      UserProxy.getUser(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_USER_INFO"), id);
    if (user == null) {
      displayError(LNG.get("IAS_USERMGR_USER_NOT_FOUND"));
      return;
    }
    if (user.getId().equals(User.getAdminId())) {
      displayError(LNG.get("IAS_USERMGR_ADMIN_CANNOT_BE_REMOVED_ERROR"));
      return;
    }
    Object[] options =
      { LNG.get("IAS_USERMGR_BUTTON_REMOVE"),
          LNG.get("IAS_USERMGR_BUTTON_CANCEL") };
    int res =
      StandardDialogs.showOptionDialog(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_USERMGR_USER_REMOVAL_CONFIRMATION"),
          new Object[] { user.getLogin() }), options);
    if (res == JOptionPane.YES_OPTION) {
      UserProxy.deleteUser(parent, parent.getTitle(), MessageFormat.format(
        LNG.get("IAS_USERMGR_WAITING_REMOVE_USER"),
        new Object[] { user.getLogin() }), id);
    }
  }

  /**
   * Recarrega a tabela de usurios, para refletir quaisquer mudanas que possam
   * ter ocorrido. Este mtodo  chamado pelo mtodo update sempre que ocorre
   * modificao nos dados de algum UserGroup, o que pode potencialmente
   * invalidar a exibio do nome do UserGroup de vrios usurios; para garantir
   * que isso no ocorra, este mtodo recarrega toda a tabela de usurios,
   * sincronizando-a com as informaes persistidas em arquivo.
   */
  public void reload() {
    Vector<UserOutline> userOutlines =
      UserProxy.getAllOutlines(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ADMIN_DATA"));
    if (userOutlines == null) {
      userOutlines = new Vector<UserOutline>();
    }
    super.setItems(userOutlines);
  }

  /**
   * Estende a funcionalidade do mtodo update da classe pai, modificando o
   * comportamento no caso especfico de receber uma notificao de alterao em
   * UserGroup; nessa situao, toda a tabela  recarregada (ver mtodo reload).
   * 
   * @param observable o objeto que est sendo observado
   * @param arg o objeto que indica a mudana ocorrida
   */
  @Override
  public void doUpdate(Observable observable, Object arg) {
    AdministrationEvent action = (AdministrationEvent) arg;
    Object item = action.item;
    if ((item instanceof UserGroup)) {
      if (action.type == AdministrationEvent.MODIFY) {
        reload();
      }
    }
    else {
      super.doUpdate(observable, arg);
    }
  }

  /**
   * Mostra uma janela com uma mensagem.
   * 
   * @param title ttulo da janela de mensagem
   * @param message mensagem a ser exibida
   */
  protected void displayMessage(String title, String message) {
    StandardDialogs.showInfoDialog(parent, title, message);
  }

  /**
   * Mostra uma janela com uma mensagem de erro.
   * 
   * @param message mensagem de erro a ser exibida
   */
  protected void displayError(String message) {
    StandardErrorDialogs.showErrorDialog(parent, parent.getTitle(), message);
  }

  /**
   * Cria um painel de gerncia de usurios cadastrados no sistema. Registra-se
   * como observador das classes User e UserGroup, pois mudanas em quaisquer
   * das duas podem implicar em alteraes na exibio dos dados.
   * 
   * @param parent Janela "pai"
   */
  public UserManagerPanel(JFrame parent) {
    this.parent = parent;
    Vector<UserOutline> userOutlines =
      UserProxy.getAllOutlines(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ADMIN_DATA"));
    if (userOutlines == null) {
      userOutlines = new Vector<UserOutline>();
    }
    ObjectTableProvider provider = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        UserOutline outline = (UserOutline) item;
        if (outline == null) {
          return null;
        }
        
        String date = null;
        if (outline.getCreationDate() != null) {
          date = dateFormat.format(outline.getCreationDate());
        }
        
        return new Object[] { outline.getId(), outline.getLogin(),
            outline.getName(), outline.getUserGroup(), date };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { null, LNG.get("IAS_USER"),
            LNG.get("IAS_USER_NAME"), LNG.get("IAS_USERGROUP"),
            LNG.get("IAS_USER_CREATION_DATE") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { Object.class, String.class, String.class,
            String.class, String.class };
      }
    };
    TableModel model =
      new ObjectTableModel<UserOutline>(userOutlines, provider);
    Comparator<?>[] comparators =
      { null, ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase()};
    make(model, comparators, true);
    User.addObserver(new SwingObserverWrapper(this));
    UserGroup.addObserver(new SwingObserverWrapper(this));
  }

  /**
   * Recebe a interface de callBack que ser usada no UserInfoDialog
   * 
   * @param callBack callBack que ser usada no UserInfoDialog.
   */
  public void setUserInfoCallBackObject(UserInfoCallBackInterface callBack) {
    this.callBack = callBack;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel makeButtons() {
    JPanel panel = super.makeButtons();
    panel.add(new JPanel());

    this.createImportButton(panel);
    this.createExportButton(panel);

    GUIUtils.matchPreferredSizes(new JComponent[] { importCSVButton,
        exportCVSButton });
    return panel;
  }

  /**
   * Criao de boto e adio ao painel.
   * 
   * @param panel painel
   */
  private void createImportButton(JPanel panel) {
    this.importCSVButton = new JButton(LNG.get("IAS_IMPORT_USERS_BUTTON"));
    this.importCSVButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        new ImportUsersDialog(parent).showDialog();
      }
    });
    panel.add(this.importCSVButton);
  }

  /**
   * Criao de boto e adio ao painel.
   * 
   * @param panel painel
   */
  private void createExportButton(JPanel panel) {
    this.exportCVSButton = new JButton(LNG.get("IAS_EXPORT_USERS_BUTTON"));
    this.exportCVSButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        new ExportUsersDialog(parent).showDialog();
      }
    });
    panel.add(this.exportCVSButton);
  }
  
}
