package csbase.client.applications.flowapplication.graph;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import tecgraf.vix.Filter;
import tecgraf.vix.TypeVS;

/**
 * Grade de pontos.
 */
public final class Grid extends Filter {
  /**
   * Distncia entre 2 pontos consecutivos.
   */
  static final int DISTANCE = 12;

  /**
   * Cor dos pontos.
   */
  private static final Color POINT_COLOR = Color.BLACK;

  /**
   * Trao dos pontos.
   */
  private static final Stroke STROKE = new BasicStroke(0f);

  /**
   * A caixa envolvente da grade.
   */
  private Rectangle2D bounds;

  /**
   * Flag que indica se o grid est habilitado.
   */
  private boolean isEnabled;

  /**
   * Cria uma grade.
   * 
   * @param vs O vs que estar ligado a esta grade.
   */
  public Grid(final TypeVS vs) {
    changeVS(null, vs);
    this.bounds = new Rectangle();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    if (isEnabled) {
      g.setStroke(STROKE);
      g.setPaint(POINT_COLOR);
      for (int x = (int) this.bounds.getX(); x < this.bounds.getMaxX(); x +=
        DISTANCE) {
        for (int y = (int) this.bounds.getY(); y < this.bounds.getMaxY(); y +=
          DISTANCE) {
          g.drawLine(x, y, x, y);
        }
      }
    }
    super.callbackRepaint(g);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    return this.bounds;
  }

  /**
   * Indica se o grid est habilitado.
   * 
   * @return {@code true} grid habilitado e {@code false} caso contrrio.
   */
  public boolean isEnabled() {
    return isEnabled;
  }

  /**
   * Muda o estado habilitado/desabilitado do grid.
   * 
   * @param isEnabled {@code true} para habilitar e {@code false} para
   *        desabilitar.
   */
  public void setEnabled(final boolean isEnabled) {
    this.isEnabled = isEnabled;
  }

  /**
   * Modifica a caixa envolvente desta grade.
   * 
   * @param x1 A abcissa do ponto superior-esquerdo.
   * @param y1 A ordenada do ponto superior-esquerdo.
   * @param x2 A abcissa do ponto inferior-direito.
   * @param y2 A ordenada do ponto inferior-direito.
   */
  public void setBounds2D(final double x1, final double y1, final double x2,
    final double y2) {
    this.bounds = new Rectangle();
    final double minX = Math.floor(x1 / DISTANCE) * DISTANCE;
    final double minY = Math.floor(y1 / DISTANCE) * DISTANCE;
    final double maxX = Math.ceil(x2 / DISTANCE) * DISTANCE;
    final double maxY = Math.ceil(y2 / DISTANCE) * DISTANCE;
    this.bounds.setFrameFromDiagonal(minX, minY, maxX, maxY);
  }

  /**
   * Cria uma nova caixa envolvente ajustada a grade que  a menor caixa que 
   * capaz de englobar a caixa passada.
   * 
   * @param bounds2D A caixa envolvente.
   * 
   * @return A caixa com o tamanho ajustado.
   */
  static Rectangle2D adjustBounds(final Rectangle2D bounds2D) {
    final double x1 = Math.floor(bounds2D.getMinX() / DISTANCE) * DISTANCE;
    final double y1 = Math.floor(bounds2D.getMinY() / DISTANCE) * DISTANCE;
    final double x2 = Math.ceil(bounds2D.getMaxX() / DISTANCE) * DISTANCE;
    final double y2 = Math.ceil(bounds2D.getMaxY() / DISTANCE) * DISTANCE;
    final Point2D p1 = new Point();
    p1.setLocation(x1, y1);
    final Point2D p2 = new Point();
    p2.setLocation(x2, y2);
    final Rectangle2D newBounds = new Rectangle();
    newBounds.setFrameFromDiagonal(p1, p2);
    return newBounds;
  }

  /**
   * Cria um novo ponto coincidindo com a grade que  o ponto mais prximo do
   * ponto passado.
   * 
   * @param point O pointo.
   * 
   * @return O ponto passado.
   */
  static Point2D adjustPoint(final Point2D point) {
    final double newX = Math.round(point.getX() / DISTANCE) * DISTANCE;
    final double newY = Math.round(point.getY() / DISTANCE) * DISTANCE;
    final Point2D newPoint = new Point();
    newPoint.setLocation(newX, newY);
    return newPoint;
  }

  /**
   * Incluso de margem.
   * 
   * @param bounds2D bounds
   * @return retngulo
   */
  static Rectangle2D includeMargin(final Rectangle2D bounds2D) {
    final double x0 = bounds2D.getMinX() - DISTANCE;
    final double y0 = bounds2D.getMinY() - DISTANCE;
    final double x1 = bounds2D.getMaxX() + DISTANCE;
    final double y1 = bounds2D.getMaxY() + DISTANCE;
    final Rectangle2D newBounds = new Rectangle();
    newBounds.setFrameFromDiagonal(x0, y0, x1, y1);
    return newBounds;
  }
}
