/*
 * $Id$
 */

package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.AllocateAreaTask;

/**
 * Ao que trata alterao na alocao de projetos. Projetos com espao alocado
 * que sero desalocados e projeto esperando alocao que sero alocados.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class DeallocateAreaAction extends ProjectsManagerAction {

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   */
  public DeallocateAreaAction(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    boolean ret = runTask(projects);
    if (ret) {
      projectsManager.removeProjects(projects);
      projectsManager.refreshProjectsTable();
    }
    else {
      String errMsg =
        getString("DeallocateAreaAction.deallocation.error.message");
      String errTitle =
        getString("DeallocateAreaAction.deallocation.error.title");
      StandardDialogs.showErrorDialog(appFrame, errTitle, errMsg);
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(List<ProjectsManagerData> projects) throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    ApplicationFrame appFrame = projectsManager.getApplicationFrame();
    AllocateAreaTask aat =
      new AllocateAreaTask(projectsManager, projects, false);
    String taskTitle = getString("DeallocateAreaAction.title");
    String taskMsg = getString("DeallocateAreaAction.message");
    aat.execute(appFrame, taskTitle, taskMsg);
    if (aat.wasCancelled()) {
      String err = getString("DeallocateAreaAction.cancelled.message");
      aat.showError(err);
      return false;
    }
    if (aat.getStatus() != true) {
      Exception exception = aat.getError();
      throw exception;
    }
    return aat.getResult();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ProjectsManagerUI.DEALLOCATED;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    List<ProjectsManagerData> prjList = new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    int i = 0;
    while (i < prjList.size()) {
      ProjectsManagerData pmd = prjList.get(i);
      if (pmd.getProjectSpaceAllocation() != ProjectSpaceAllocation.WAITING_DEALLOCATION) {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
      else {
        i++;
      }
    }

    ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      String deniedMessage =
        getString("DeallocateAreaAction.project.selection.denied.message")
          + getString("DeallocateAreaAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("DeallocateAreaAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      String someDeniedMessage =
        getString("DeallocateAreaAction.project.selection.some.denied.message")
          + getString("DeallocateAreaAction.project.requirements.message");
      int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("DeallocateAreaAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

}
