/*
 * $Author: fpina $ $Date: 2016-09-22 18:12:51 -0300 (Thu, 22 Sep 2016) $
 * $Release: $
 */
package csbase.client.util.sga;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;

import javax.swing.JComboBox;

import csbase.client.kernel.ClientException;
import csbase.logic.SGASet;
import csbase.logic.algorithms.AlgorithmConfigurator;

/**
 * Define um painel para seleo de SGAs.
 * 
 * @author ururahy
 * 
 */
public class SGASingleSelectionPanel extends SGASelectionPanel {

  /**
   * Combo de seleo de sgas.
   */
  private JComboBox sgaServerComboBox;

  /**
   * Define um painel para seleo de SGAs.
   * 
   * @param configurator o configurador de algoritmos, que informa se o
   *        algoritmo  de execuo simples.
   */
  public SGASingleSelectionPanel(AlgorithmConfigurator configurator) {
    super(configurator);
    this.sgaServerComboBox = makeSGAServerComboBox();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<SGASet> getManualServerCollection() {
    Collection<SGASet> selectedServerCollection = new LinkedList<SGASet>();
    SGASet selectedServer = getSelectedServer();
    if (selectedServer != null) {
      selectedServerCollection.add(selectedServer);
    }
    return Collections.unmodifiableCollection(selectedServerCollection);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public SGASet getManualSelectedServer() {
    return (SGASet) this.sgaServerComboBox.getSelectedItem();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean hasAvailableServers() {
    if (this.sgaServerComboBox.getItemCount() == 0) {
      return false;
    }
    return true;
  }

  /**
   * Verifica se a seleo de servidores est pronta.
   *
   * A seleo  considerada pronta, se e somente se 1 dos seguintes critrios
   * for atendido:
   * 
   * <ul>
   * <li>
   * A seleo de servidor estiver no modo automtico.</li>
   * <li>
   * A seleo de servidor estiver no modo manual e existir ao menos um servidor
   * selecionado.</li>
   * </ul>
   *
   * @return .
   */
  @Override
  public boolean isSelectionReady() {
    if (!this.hasAvailableServers()) {
      return false;
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void populateGUIComponents() {
    this.removeAll();
    this.setLayout(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();
    c.anchor = GridBagConstraints.WEST;
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.insets = new Insets(2, 2, 2, 2);
    c.weightx = 1.0;
    c.weighty = 0.0;
    c.fill = GridBagConstraints.HORIZONTAL;
    this.add(this.platformFilterPanel, c);
    c.insets = new Insets(2, 2, 2, 2);
    c.gridx = 1;
    c.gridy++;
    this.add(this.automaticServerRadioButton, c);
    c.gridy++;
    this.add(this.manualServerRadioButton, c);
    c.weightx = 0.0;
    c.gridy++;
    c.insets = new Insets(2, 20, 2, 2);
    this.add(this.sgaServerComboBox, c);
    this.revalidate();

    this.updateEnabledServers();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void updateEnabledServers() {
    if (this.hasAvailableServers()) {
      if (this.manualServerRadioButton.isSelected()) {
        this.sgaServerComboBox.setEnabled(true);
      }
      else {
        this.sgaServerComboBox.setEnabled(false);
      }
      this.manualServerRadioButton.setEnabled(true);
      this.automaticServerRadioButton.setEnabled(true);
    }
    else {
      this.manualServerRadioButton.setEnabled(false);
      this.automaticServerRadioButton.setEnabled(false);
      this.sgaServerComboBox.setEnabled(false);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void reloadSGAs() {
    try {
      sgaServerComboBox.removeAllItems();
      for (SGASet sgaSet : this.loadServers()) {
        sgaServerComboBox.addItem(sgaSet);
      }
      updateEnabledServers();
    }
    catch (ClientException e) {
      e.printStackTrace();
    }
  }

  /**
   * Cria um componente do tipo JComboBox para a seleo de SGAs.
   * 
   * @return O componente JComboBox criado e populado com os SGAs disponveis.
   */
  private JComboBox makeSGAServerComboBox() {
    JComboBox combo = null;
    try {
      combo = new JComboBox(this.loadServers());
      combo.addItemListener(new ItemListener() {
        @Override
        public void itemStateChanged(ItemEvent e) {
          fireChangedServer();
        }
      });
    }
    catch (ClientException e) {
      e.printStackTrace();
    }
    return combo;
  }
}
