package csbase.client.util.svgpanel;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.net.URI;
import java.net.URL;

import javax.swing.JPanel;

import com.kitfox.svg.SVGCache;
import com.kitfox.svg.app.beans.SVGIcon;

/**
 * Painel que mostra somente um arquivo SVG
 * 
 * @author Tecgraf/PUC-Rio
 */
public class SVGPanel extends JPanel {

  /**
   * Alinhamento horizontal
   * 
   * @author Tecgraf/PUC-Rio
   */
  public enum HorizontalAlignment {
    /**
     *  esquerda
     */
    LEFT,

    /**
     * Centro
     */
    CENTER,

    /**
     *  direita
     */
    RIGHT
  }

  /**
   * Alinhamento vertical
   * 
   * @author Tecgraf/PUC-Rio
   */
  public enum VerticalAlignment {
    /**
     * Topo
     */
    TOP,

    /**
     * Centro
     */
    CENTER,

    /**
     * Base
     */
    BOTTOM
  }

  /**
   * Alinhamento horizontal.
   */
  private HorizontalAlignment horizontalAlignment = HorizontalAlignment.CENTER;

  /**
   * Retorna: alinhamento; conforme atributo {@link #horizontalAlignment}.
   * 
   * @return o valor
   */
  public final HorizontalAlignment getHorizontalAlignment() {
    return horizontalAlignment;
  }

  /**
   * Retorna: alinhamento; conforme atributo {@link #verticalAlignment}.
   * 
   * @return o valor
   */
  public final VerticalAlignment getVerticalAlignment() {
    return verticalAlignment;
  }

  /**
   * Alinhamento vertical.
   */
  private VerticalAlignment verticalAlignment = VerticalAlignment.CENTER;

  /**
   * SVG icon associado.
   */
  private SVGIcon icon;

  /**
   * Tamanho orginal do arquivo: largura
   */
  private int iconOrigWidth;

  /**
   * Tamanho orginal do arquivo: altura
   */
  private int iconOrigHeight;

  /**
   * Ajuste a imagem com base no path de resources de uma classe
   * 
   * @param clazz classe
   * @param path path do resource
   */
  final public void setImageFromClassResource(Class<?> clazz, final String path) {
    if (clazz == null) {
      final String err = "Null class detected!";
      throw new IllegalArgumentException(err);
    }

    if (path == null || path.trim().isEmpty()) {
      final String err = "Null or empty resource path!";
      throw new IllegalArgumentException(err);
    }
    final URL url = clazz.getResource(path);
    setImageFromURL(url);
  }

  /**
   * Ajuste da imagem com base em uma URL.
   * 
   * @param url URL.
   */
  final public void setImageFromURL(final URL url) {
    if (url == null) {
      final String err = "Null URL detected!";
      throw new IllegalArgumentException(err);
    }
    final URI uri = SVGCache.getSVGUniverse().loadSVG(url);
    icon.setSvgURI(uri);
    iconOrigWidth = icon.getIconWidth();
    iconOrigHeight = icon.getIconHeight();
  }

  /**
   * Ajuste da imagem com base num cone SVG.
   * 
   * @param icon cone.
   */
  final public void setImage(final SVGIcon icon) {
    if (icon == null) {
      final String err = "Null icon detected!";
      throw new IllegalArgumentException(err);
    }
    this.icon = icon;
    iconOrigWidth = icon.getIconWidth();
    iconOrigHeight = icon.getIconHeight();
  }

  /**
   * Construtor
   */
  public SVGPanel() {
    this.icon = new SVGIcon();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void paintComponent(Graphics g) {
    final int width = getWidth();
    final int height = getHeight();
    if (icon.getSvgURI() == null) {
      g.setColor(Color.WHITE);
      g.fillRect(0, 0, width, height);
      return;
    }
    g.setColor(getBackground());
    g.fillRect(0, 0, width, height);
    final double ratio = width / (double) height;
    final double iconRatio = iconOrigWidth / (double) iconOrigHeight;
    final int w;
    final int h;
    if (ratio < 1) {
      w = width;
      h = (int) Math.round(w / iconRatio);
    }
    else {
      h = height;
      w = (int) Math.round(iconRatio * h);
    }
    icon.setPreferredSize(new Dimension(w, h));
    icon.setAntiAlias(true);
    icon.setScaleToFit(true);

    final int x;
    switch (horizontalAlignment) {
      case LEFT:
        x = 0;
        break;
      case RIGHT:
        x = (width - icon.getIconWidth());
        break;
      case CENTER:
        x = (int) Math.round((width - w) / 2.0);
        break;
      default:
        throw new IllegalArgumentException("bad enum!");
    }

    final int y;
    switch (verticalAlignment) {
      case TOP:
        y = 0;
        break;
      case BOTTOM:
        y = height - icon.getIconHeight();
        break;
      case CENTER:
        y = (int) Math.round((height - h) / 2.0);
        break;
      default:
        throw new IllegalArgumentException("bad enum!");
    }

    icon.paintIcon(this, g, x, y);
    //    g.setColor(Color.RED);
    //    g.drawRect(x, y, w, h);
  }

  /**
   * Ajuste de alinhamento vertical.
   * 
   * @param aligment alinhamento
   */
  public void setVerticalAligment(VerticalAlignment aligment) {
    verticalAlignment = aligment;
    repaint();

  }

  /**
   * Ajuste de alinhamento horizontal
   * 
   * @param aligment alinhamento
   */
  public void setHorizontalAligment(HorizontalAlignment aligment) {
    horizontalAlignment = aligment;
    repaint();
  }
}
