package csbase.client.preferences.types;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.client.facilities.configurabletable.table.ColumnState;
import csbase.client.facilities.configurabletable.table.ConfigurableTable;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.editors.PVTablesEditor;
import csbase.client.preferences.util.PreferenceBundle;

/**
 * Preferncia que encapsula um objeto do tipo {@link ConfigurableTable}. Este
 * valor deve ser usado da seguinte forma:<br>
 * 
 * \@Value(type=PVTables.class, defaultValue="")
 * 
 * Para salvarmos os estados das colunas de tabelas, os identificadores das
 * colunas no podem contr os seguintes caracteres: <br>
 * - igual (=) <br>
 * - ponto e virgula (;) <br>
 * - colchetes ([]) <br>
 * - vrgula (,) <br>
 * 
 * @see PreferenceValue PreferenceCategory PreferencePolicy
 * 
 * @author Tecgraf
 */
public class PVTables extends PreferenceValue<Map<String, List<ColumnState>>> {

  /**
   * Construtor usado na instanciao desta classe por reflexo.
   * 
   * @param name nome da preferncia que possui este valor.
   * @param value valor da preferncia.
   * @param defaultValue valor default da preferncia.
   * @param policy poltica de visibilidade do valor.
   * @param preferenceBundle objeto responsvel pela internacionalizao.
   */
  public PVTables(PreferenceDefinition name, String value, String defaultValue,
    PreferencePolicy policy, PreferenceBundle preferenceBundle) {
    super(name, value, defaultValue, policy, preferenceBundle);

    setPreferenceEditorClass(PVTablesEditor.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getClassName() {
    return getClass().getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    for (Entry<String, List<ColumnState>> entry : getValue().entrySet()) {

      String tableId = entry.getKey();
      List<ColumnState> states = entry.getValue();

      builder.append(tableId);
      builder.append("=");
      builder.append(serializeStates(states));
      builder.append(";");
    }
    return builder.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, List<ColumnState>> toValue(String value) {
    Pattern p = Pattern.compile("(.*?)=(.*?);");

    Map<String, List<ColumnState>> result =
      new HashMap<String, List<ColumnState>>();

    Matcher m = p.matcher(value);
    while (m.find()) {
      result.put(m.group(1), deserializeState(m.group(2)));
    }

    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PreferenceValue<Map<String, List<ColumnState>>> clone() {
    return new PVTables(name, this.toString(), defaultValue, policy,
      preferenceBundle);
  }

  /**
   * Mtodo utilitrio que recebe uma lista de tabelas e carrega as
   * visibilidades das colunas de cada tabela.
   * 
   * @param tables - lista de tabelas.
   */
  public void loadTables(List<ConfigurableTable<?>> tables) {
    if (tables == null) {
      throw new IllegalArgumentException("tables no pode ser nulo.");
    }

    for (ConfigurableTable<?> table : tables) {
      loadTable(table);
    }
  }

  /**
   * Mtodo utilitrio que recebe uma tabelas e carrega as visibilidades das
   * colunas.
   * 
   * @param table - tabela.
   */
  public void loadTable(ConfigurableTable<?> table) {
    if (table == null) {
      throw new IllegalArgumentException("table no pode ser nulo.");
    }

    Map<String, List<ColumnState>> v = getValue();
    List<ColumnState> columnsState = v.get(table.getId());

    if (columnsState != null && !columnsState.isEmpty()) {
      table.setColumnsState(columnsState);
      table.updateColumns();
    }
  }

  /**
   * Mtodo utilitrio que recebe uma lista de tabelas e armazena a informao
   * de visibilidade das suas colunas.
   * 
   * @param tables - lista com as tabelas.
   */
  public void storeTables(List<ConfigurableTable<?>> tables) {
    if (tables == null) {
      throw new IllegalArgumentException("tables no pode ser nulo.");
    }

    for (ConfigurableTable<?> table : tables) {
      storeTable(table);
    }
  }

  /**
   * Mtodo utilitrio que recebe uma tabela e armazena a visibilidade das suas
   * colunas.
   * 
   * @param table - tabela.
   */
  public void storeTable(ConfigurableTable<?> table) {
    if (table == null) {
      throw new IllegalArgumentException("table no pode ser nulo.");
    }

    Map<String, List<ColumnState>> v = getValue();
    v.put(table.getId(), table.getColumnsState());
  }

  /**
   * Transforma uma lista de estados de colunas em uma string.
   * 
   * @param states - lista com os estados de colunas.
   * @return String que representa essa lista serializada.
   */
  private String serializeStates(List<ColumnState> states) {
    StringBuilder builder = new StringBuilder();

    for (int i = 0; i < states.size(); i++) {
      ColumnState state = states.get(i);

      builder.append("[");
      builder.append(state.getColumnId());
      builder.append(",");
      builder.append(state.isVisible());
      builder.append("]");

      if (i != states.size() - 1) {
        builder.append(",");
      }
    }

    return builder.toString();
  }

  /**
   * Transforma uma string com os estados de colunas em uma lista de objetos que
   * representam os estados de colunas.
   * 
   * @param states - string com os estados de colunas.
   * @return lista com os objetos que representam os estados de colunas.
   */
  private List<ColumnState> deserializeState(String states) {
    List<ColumnState> result = new ArrayList<ColumnState>();

    String[] statesArray = states.split(",(?=\\[)");
    for (String state : statesArray) {
      String columnId = state.substring(state.indexOf("[") + 1, state.indexOf(
        ","));

      String visible = state.substring(state.indexOf(",") + 1, state.indexOf(
        "]"));

      result.add(new ColumnState(columnId, Boolean.valueOf(visible)));
    }

    return result;
  }
}
