/*
 * ProjectFileChooser.java $Author: fpina $ $Revision: 176168 $ - $Date:
 * 2006-08-16 18:31:12 -0300 (Wed, 16 Aug 2006) $
 */
package csbase.client.project;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.border.Border;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.TreeFilterPanelListener;
import csbase.client.kernel.ClientException;
import csbase.client.util.ClientUtilities;
import csbase.client.util.CountDown;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileFilter;
import csbase.logic.filters.ProjectFileCompositeAndFilter;
import csbase.logic.filters.ProjectFileCompositeOrFilter;
import csbase.logic.filters.ProjectFileDirectoryFilter;
import csbase.logic.filters.ProjectFileNameFilter;
import csbase.logic.filters.ProjectFileNotFilter;
import csbase.logic.filters.ProjectFileTypeFilter;
import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.SwingThreadDispatcher;

/**
 * A classe <code>ProjectFileChooser</code> mostra um dilogo modal para seleo
 * de arquivos de projeto.
 *
 * @author Maria Julia
 */
public abstract class ProjectFileChooser {
  /** Representa o modo de seleo de arquivo. */
  public static final int FILE_ONLY = 0;

  /** Representa o modo de seleo de diretrio. */
  public static final int DIRECTORY_ONLY = 1;

  /** Representa o modo de seleo de arquivo e diretrio. */
  public static final int FILE_AND_DIRECTORY = 2;

  /** A rvore de arquivos de um projeto */
  protected ProjectTree projectTree;

  /** Seleo do tipo de arquivo */
  //TODO ITP Agora esta armazenado o codigo na combobox. Cuidado
  protected ProjectFileTypeComboBox fileTypeList;

  /**
   * Campo de texto correspondente ao nome de um arquivo selecionado ou que est
   * sendo criado.
   */
  protected JTextField fileNameText;

  /** Array de arquivos selecionados na arvore */
  protected ProjectTreePath[] selectedPaths;

  /** O dilogo de seleo que ser mostrado para se efetuar a seleo. */
  protected DesktopComponentDialog dialog;

  /** Boto que confirma a seleo. */
  private JButton confirmButton;

  /** Representa o ouvinte de eventos de seleo na rvore de projeto. */
  private ProjectTreeSelectionListener treeSelectionListener;

  /**
   * Representa o modo de seleo escolhido.
   *
   * @see #FILE_ONLY
   * @see #DIRECTORY_ONLY
   * @see #FILE_AND_DIRECTORY
   */
  protected int mode;

  /** Representa o label para seleo/edio do nome do arquivo/diretrio. */
  private JLabel fileNameLabel;

  /** Painel de filtro para a rvore de projetos. */
  TreeFilterPanel treeFilterPanel;

  /**
   * Constri uma viso de dilogo para seleo de um arquivo de um projeto.
   * Diretrio corrente ser obtido do nome do arquivo ou (caso este seja null)
   * ser a raiz do projeto.
   *
   * @param owner janela pai onde o dilogo ser exibido
   * @param title ttulo do dilogo
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled define se o usurio pode selecionar mais de
   *        um arquivo
   * @param mode modo de seleo do dilogo
   * @see #FILE_ONLY
   * @see #DIRECTORY_ONLY
   * @see #FILE_AND_DIRECTORY
   * @param defaultFileType tipo de arquivo default do dilogo
   * @param visualFilter filtro de visualizao da rvore
   * @param selectionFilter filtro de seleo da rvore
   * @param canEditText define se o atributo {@link #fileNameText} ser editvel
   * @param defaultFullFileName valor sugerido inicialmente para o nome do
   *        arquivo (atributo {@link #fileNameText})
   * @param useFilter se um painel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  protected ProjectFileChooser(Window owner, String title,
    CommonClientProject project, boolean isMultiSelectionEnabled, int mode,
    String defaultFileType, ProjectFileFilter visualFilter,
    ProjectFileFilter selectionFilter, boolean canEditText,
    String defaultFullFileName, boolean useFilter) throws ClientException {
    this(owner, title, project, isMultiSelectionEnabled, mode, defaultFileType,
      visualFilter, selectionFilter, canEditText, defaultFullFileName, null,
      useFilter);
  }

  /**
   * Constri uma viso de dilogo para seleo de um arquivo de um projeto. Se
   * houver defaultFullFileName, o diretrio corrente ser obtido do nome do
   * arquivo. Seno, caso defaultFullFileName seja null, permite a especificao
   * de um diretrio corrente para abertura do chooser. Caso no seja
   * especifciado, e ambos sejam nulos, o diretrio corrente ser a raiz do
   * projeto.
   *
   * @param owner janela pai onde o dilogo ser exibido
   * @param title ttulo do dilogo
   * @param project projeto do qual o usurio vai selecionar um arquivo
   * @param isMultiSelectionEnabled define se o usurio pode selecionar mais de
   *        um arquivo
   * @param mode modo de seleo do dilogo
   * @see #FILE_ONLY
   * @see #DIRECTORY_ONLY
   * @see #FILE_AND_DIRECTORY
   * @param defaultFileType tipo de arquivo default do dilogo
   * @param visualFilter filtro de visualizao da rvore
   * @param selectionFilter filtro de seleo da rvore
   * @param canEditText define se o atributo {@link #fileNameText} ser editvel
   * @param defaultFullFileName valor sugerido inicialmente para o nome do
   *        arquivo (atributo {@link #fileNameText})
   * @param currentDirectory diretrio corrente para abertura do chooser, pode
   *        ser null, faz sentido usar apenas se no houver defaultFullFileName
   * @param useFilter se um painel de filtro deve ser exibido
   * @throws ClientException exceo lanada caso ocorra algum erro na criao
   *         da rvore de projeto
   */
  protected ProjectFileChooser(Window owner, String title,
    CommonClientProject project, boolean isMultiSelectionEnabled, int mode,
    String defaultFileType, ProjectFileFilter visualFilter,
    ProjectFileFilter selectionFilter, boolean canEditText,
    String defaultFullFileName, String[] currentDirectory, boolean useFilter)
    throws ClientException {
    this.mode = mode;
    this.dialog = new DesktopComponentDialog(owner, title);
    this.buildProjectTree(title, project, isMultiSelectionEnabled,
      visualFilter, selectionFilter);
    JPanel contentPanel = new JPanel(new BorderLayout(0, 5));
    contentPanel.add(new JScrollPane(this.projectTree.getTree()),
      BorderLayout.CENTER);
    if (useFilter) {
      contentPanel.add(makeTextFieldPanel(canEditText), BorderLayout.SOUTH);
    }
    else {
      contentPanel.add(makeTextFieldPanel(defaultFileType, canEditText),
        BorderLayout.SOUTH);
    }
    if (useFilter) {
      this.projectTree.setVisualFilter(visualFilter);
      treeFilterPanel =
        new TreeFilterPanel(this.projectTree, defaultFileType, mode);
      treeFilterPanel.setVisible(true);
      contentPanel.add(treeFilterPanel, BorderLayout.NORTH);
    }
    JPanel mainPanel = new JPanel(new BorderLayout(0, 5));
    mainPanel.add(contentPanel, BorderLayout.CENTER);
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);
    this.dialog.getContentPane().add(mainPanel);
    this.dialog.pack();
    this.dialog.center(owner);
    this.dialog.getRootPane().setDefaultButton(this.confirmButton);
    this.dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        ProjectFileChooser.this.selectedPaths = null;
        ProjectFileChooser.this.close();
      }
    });
    /*
     * Ao chamar o mtodo projectTree.setSelectionPath, deve-se usar null quando
     * o arquivo est na raz do projeto, porm, quando o arquivo est em um
     * subdiretrio, o caminho a ser usado NO deve incluir o diretrio do
     * projeto.
     */
    String selectionPath[] = null;
    String defaultName = null;
    if (defaultFullFileName != null) {
      String defaultPath[] = FileUtils.splitPath(defaultFullFileName);
      if (defaultPath.length > 1) {
        selectionPath = new String[defaultPath.length - 1];
        System.arraycopy(defaultPath, 0, selectionPath, 0,
          defaultPath.length - 1);
      }
      defaultName = defaultPath[defaultPath.length - 1];
    }
    else if (currentDirectory != null && currentDirectory.length > 0) {
      // path para algum subdiretrio dentro do projeto
      selectionPath = currentDirectory;
    }
    applyTypeFilter();
    this.projectTree.setSelectionPath(selectionPath);
    this.fileNameText.requestFocus();
    this.dialog.center();

    if (mode == FILE_ONLY) {
      if (defaultName != null) {
        this.fileNameText.setText(defaultName);
        this.confirmButton.setEnabled(true);
      }
    }
    else {
      // diretrios e/ou arquivos
      // habilitar o boto de confirmao, pois o prprio projeto estar
      // selecionado por default caso no tenha um texto padro
      this.confirmButton.setEnabled(true);
      if (mode == DIRECTORY_ONLY) {
        // no mostramos campo de texto para seleo de diretrios
        this.fileNameText.setVisible(false);
        this.fileNameLabel.setVisible(false);
      }
      else {
        // arquivos e diretrios
        if (defaultName != null) {
          this.fileNameText.setText(defaultName);
        }
      }
    }
  }

  protected static ProjectFileFilter createVisualFilter(int mode,
    boolean recursive, String nameFilter, String[] fileTypes) {

    boolean useTypeFilter = fileTypes != null && fileTypes.length != 0;
    boolean useNameFilter = nameFilter != null && !nameFilter.isEmpty();

    // Filtro de nomes, tipos e arquivo/diretrio.
    ProjectFileCompositeAndFilter andTypeFilter =
      new ProjectFileCompositeAndFilter();

    if (useNameFilter) {
      andTypeFilter.addChild(new ProjectFileNameFilter(nameFilter));
    }

    if (useTypeFilter) {
      andTypeFilter.addChild(new ProjectFileTypeFilter(fileTypes));
    }

    ProjectFileDirectoryFilter onlyDirectoriesFilter = ProjectFileDirectoryFilter.getInstance();

    switch (mode) {
    case FILE_ONLY:
      // O filtro and deve ser aplicado a apenas arquivos.
      ProjectFileNotFilter onlyFilesFilter = new ProjectFileNotFilter(onlyDirectoriesFilter);
      andTypeFilter.addChild(onlyFilesFilter);
      break;
    case DIRECTORY_ONLY:
      // O filtro and deve ser aplicado a apenas arquivos.
      andTypeFilter.addChild(onlyDirectoriesFilter);
      break;
    }

    ProjectFileCompositeOrFilter orFilter = new ProjectFileCompositeOrFilter();

    orFilter.addChild(andTypeFilter);

    // Mostra qualquer diretrio
    orFilter.addChild(onlyDirectoriesFilter);

    return orFilter;
  }

  /**
   * Cria o painel com os botes do dilogo.
   * <ul>
   * <li>o boto de confirmao da seleo.</li>
   * <li>o boto de cancelamento da seleo.</li>
   * </ul>
   *
   * @return O painel com os botes.
   */
  private JPanel makeButtonPanel() {
    JPanel butPanel = new JPanel();
    this.confirmButton = new JButton(LNG.get("PRJ_FILE_CHOOSER_CONFIRM"));
    this.confirmButton.setEnabled(false);
    this.confirmButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        if (ProjectFileChooser.this.handleAction()) {
          ProjectFileChooser.this.close();
        }
      }
    });

    JButton cancelButton = new JButton(LNG.get("PRJ_FILE_CHOOSER_CANCEL"));
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        ProjectFileChooser.this.selectedPaths = null;
        ProjectFileChooser.this.close();
      }
    });

    ClientUtilities.adjustEqualSizes(this.confirmButton, cancelButton);
    butPanel.add(this.confirmButton);
    butPanel.add(cancelButton);
    return butPanel;
  }

  /**
   * Cria o painel com os campos de nome de arquivo e de filtro.
   *
   * @param defaultFileType O tipo padro que ser selecionado na criao do
   *        dilogo.
   * @param canEditText Define se o campo texto poder ser alterado pelo
   *        usurio.
   * @return O painel com os campos de arquivo e filtro.
   */
  private JPanel makeTextFieldPanel(String defaultFileType, boolean canEditText) {
    this.fileNameText = new JTextField(30);
    this.fileNameText.setEditable(canEditText);
    this.fileNameLabel = new JLabel(LNG.get("PRJ_FILE_CHOOSER_FILE_NAME"));
    this.fileNameText.getDocument().addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        this.textChanged();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        this.textChanged();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        this.textChanged();
      }

      public void textChanged() {
        ProjectFileChooser.this.updateConfirmButton();
      }
    });

    JComponent[][] components = null;
    JLabel fileTypeLabel =
      new JLabel(LNG.get("PRJ_FILE_CHOOSER_FILTER_NAME"));

    ProjectFileTypeComboBox.Mode mode;
    switch (this.mode) {
    case FILE_ONLY:
      mode = ProjectFileTypeComboBox.Mode.FILE_ONLY;
      break;
    case DIRECTORY_ONLY:
      mode = ProjectFileTypeComboBox.Mode.DIRECTORY_ONLY;
      break;
    case FILE_AND_DIRECTORY:
      mode = ProjectFileTypeComboBox.Mode.FILE_AND_DIRECTORY;
      break;
    default:
      String s = String.format("Modo %s invlido.\n", this.mode);
      throw new IllegalStateException(s);
    }
    /* Tipos de arquivos */
    fileTypeList = new ProjectFileTypeComboBox(mode, true);
    if (defaultFileType != null) {
      this.fileTypeList.selectTypeCode(defaultFileType);
    }
    this.fileTypeList.addActionListener(e -> {
      applyTypeFilter();
    });
    components =
      new JComponent[][] { { fileNameLabel, this.fileNameText }, {
        fileTypeLabel, this.fileTypeList }, };
    return GUIUtils.createBasicGridPanel(components);
  }

  /**
   * Aplica o filtro de tipos de arquivo.
   */
  private void applyTypeFilter() {
    ProjectFileFilter filter;
    if (fileTypeList.isAllItemSelected()) {
      filter = getVisualFilter();
    }
    else {
      filter = getVisualFilter(fileTypeList.getSelectedTypeCode());
    }
    projectTree.setVisualFilter(filter);
    if (fileTypeList.isAllItemSelected()) {
      filter = getSelectionFilter();
    }
    else {
      filter = getSelectionFilter(fileTypeList.getSelectedTypeCode());
    }
    projectTree.setSelectionFilter(filter);
    ProjectFileChooser.this.updateConfirmButton();
  }

  /**
   * Cria o painel com o campos de nome de arquivo.
   *
   * @param canEditText Define se o campo texto poder ser alterado pelo
   *        usurio.
   *
   * @return O painel com o campo de arquivo e filtro.
   */
  private JPanel makeTextFieldPanel(final boolean canEditText) {
    this.fileNameText = new JTextField(30);
    this.fileNameText.setEditable(canEditText);
    this.fileNameLabel = new JLabel(LNG.get("PRJ_FILE_CHOOSER_FILE_NAME"));
    this.fileNameText.getDocument().addDocumentListener(new DocumentListener() {
      @Override
      public void changedUpdate(DocumentEvent e) {
        this.textChanged();
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        this.textChanged();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        this.textChanged();
      }

      public void textChanged() {
        ProjectFileChooser.this.updateConfirmButton();
      }
    });

    JComponent[][] components = null;
    components = new JComponent[][] { { fileNameLabel, this.fileNameText } };
    JPanel txtPanel = GUIUtils.createBasicGridPanel(components);
    return txtPanel;
  }

  /**
   * Habilita ou desabilita o {@link #confirmButton} de acordo com o estado
   * atual do dilogo.
   */
  private void updateConfirmButton() {
    if (mode == FILE_ONLY) {
      boolean hasFilename = fileNameText.getText().trim().length() > 0;
      this.confirmButton.setEnabled(hasFilename && getSelectedPath() != null);
    }
    else {
      this.confirmButton.setEnabled(getSelectedPath() != null);
    }
  }

  /**
   * Executa a ao de seleo.  executada quando o usurio clica no boto
   * {@link #confirmButton} ou quando ele executa um duplo clique num arquivo
   * selecionado.
   *
   * @return true, se a ao foi executada com sucesso, ou false, caso
   *         contrrio.
   */
  protected abstract boolean handleAction();

  private void buildProjectTree(String title, CommonClientProject project,
    boolean isMultiSelectionEnabled, ProjectFileFilter visualFilter,
    ProjectFileFilter selectionFilter) throws ClientException {
    this.projectTree =
      ProjectTreeFactory.getInstance().makeProjectTree(this.dialog, title,
        project, visualFilter, selectionFilter);
    this.projectTree.setSingleSelectionMode(!isMultiSelectionEnabled);
    this.projectTree.getTree().addMouseListener(
      new ProjectTreeMouseAdapter(this.projectTree) {
        @Override
        public void doubleClickAction(ClientProjectFile file) {
          if (ProjectFileChooser.this.handleAction()) {
            ProjectFileChooser.this.close();
          }
        }
      });
    this.treeSelectionListener = new ProjectTreeSelectionListener() {
      @Override
      public void update(ProjectTreeSelectionEvent event) {
        ProjectFileChooser.this.makeSelection(event.getSelectedFiles());
        ProjectFileChooser.this.updateConfirmButton();
      }
    };
    this.projectTree.addTreeSelectionListener(this.treeSelectionListener);
  }

  protected abstract void makeSelection(ClientProjectFile[] projectFileArray);

  protected abstract ProjectFileFilter getSelectionFilter();

  protected abstract ProjectFileFilter getSelectionFilter(String fileType);

  protected abstract ProjectFileFilter getSelectionFilter(String nameFilter,
    String fileType);

  /**
   * Obtm o caminho do ltimo arquivo selecionado na rvore.
   *
   * @return o caminho do arquivo selecionado
   */
  public ProjectTreePath getSelectedPath() {
    if (this.selectedPaths == null || this.selectedPaths.length == 0) {
      return null;
    }
    return this.selectedPaths[this.selectedPaths.length - 1];
  }

  /**
   * Fecha o dilogo e libera os ouvintes que esse dilogo inseriu na rvore de
   * projetos.
   */
  private void close() {
    this.projectTree.removeTreeSelectionListener(this.treeSelectionListener);
    this.projectTree.release();
    this.dialog.close();
  }

  /**
   * Obtm o arquivo que est atualmente selecionado.
   *
   * @return O arquivo selecionado ou null, caso no haja seleo.
   */
  protected ClientProjectFile getSelectedFile() {
    return this.projectTree.getSelectedFile();
  }

  /**
   * Obtm os arquivos que esto atualmente selecionados.
   *
   * @return Os arquivos atualmente selecionados ou null, caso no haja seleo.
   */
  protected ClientProjectFile[] getSelectedFiles() {
    return this.projectTree.getSelectedFiles();
  }

  /**
   * Obtm os paths atualmente selecionados.
   *
   * @return Os paths atualmente selecionados ou null, caso no haja seleo.
   */
  public ProjectTreePath[] getSelectedPaths() {
    return this.selectedPaths;
  }

  /**
   * @return um ttulo especial para o dilogo. Null para usar o padro.
   */
  protected String getTitle() {
    return null;
  }

  private ProjectFileFilter getVisualFilter() {
    return createVisualFilter(this.mode, false, null, null);
  }

  private ProjectFileFilter getVisualFilter(String fileType) {
    return createVisualFilter(this.mode, false, null, new String[]{fileType});
  }

  private ProjectFileFilter getVisualFilter(final String nameFilter, final String fileType) {
    return createVisualFilter(mode, true, nameFilter, new String[]{fileType});
  }

  /**
   * Painel de filtro para a rvore de projetos. Esta classe  um
   * <i>singleton</i>.
   *
   * @author Tecgraf
   */
  public class TreeFilterPanel extends JPanel {
    /**
     * Tempo em que aps o ltimo caracter digitado, o campo de texto deve ficar
     * ocioso para que comece a filtragem.
     */
    private static final long TEXTFIELD_COUNT_DOWN = 1;
    /** Unidade de tempo do {@link #TEXTFIELD_COUNT_DOWN}. */
    private final TimeUnit TEXTFIELD_COUNT_DOWN_UNIT = TimeUnit.SECONDS;
    /** Caixa de texto para o filtro por nome do arquivo. */
    private JTextField filterTextField;
    /** Caixa de seleo para o filtro por tipo de arquivo. */
    private ProjectFileTypeComboBox typeComboBox;
    /** Referncia para a rvore de projetos. */
    private ProjectTree projectTree;
    /** Boto para limpar a caixa de texto contendo o critrio de filtro. */
    private JButton clearFieldButton;

    /**
     * Conjunto de objetos interessados em mudanas no estado deste
     * <i>singleton</i>.
     */
    private List<TreeFilterPanelListener> listeners =
      new ArrayList<TreeFilterPanelListener>();

    /**
     * Cria o painel de filtro.
     *
     * @param tree referncia para a rvore de projetos.
     * @param defaultFileType referncia para a janela de detalhes da rvore de
     *        projetos.
     * @param mode o tipo padro que ser selecionado na criao do
     *        typeComboBox. Pode ser nulo. Neste caso, a opo todos os arquivos
     *        ser selecionada.
     */
    public TreeFilterPanel(ProjectTree tree, String defaultFileType, int mode) {
      super(new GridBagLayout());
      this.projectTree = tree;
      filterTextField = new JTextField(10);
      filterInput(filterTextField);
      filterTextField.addKeyListener(new KeyAdapter() {
        @Override
        public void keyReleased(KeyEvent e) {
          clearFieldButton.setEnabled(filterTextField.getText().length() > 0);
        }
      });
      filterTextField.setToolTipText(LNG
        .get("TreeFilterPanel.filterTextField.tooltip"));
      filterTextField.addKeyListener(new KeyAdapter() {
        final CountDown countDown = new CountDown(TEXTFIELD_COUNT_DOWN,
          TEXTFIELD_COUNT_DOWN_UNIT, new Runnable() {
            @Override
            public void run() {
              SwingThreadDispatcher.invokeLater(new Runnable() {
                @Override
                public void run() {
                  filterTree();
                }
              });
            }
          });

        @Override
        public void keyReleased(KeyEvent e) {
          countDown.restart();
        }
      });
      clearFieldButton = new JButton(ApplicationImages.ICON_CLEARMSG_16);
      clearFieldButton.setMargin(new Insets(0, 0, 0, 0));
      clearFieldButton.setEnabled(false);
      clearFieldButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          filterTextField.setText("");
          clearFieldButton.setEnabled(false);
          filterTextField.requestFocusInWindow();
          filterTree();
        }
      });
      ProjectFileTypeComboBox.Mode comboBoxMode;
      switch (mode) {
      case FILE_ONLY:
        comboBoxMode = ProjectFileTypeComboBox.Mode.FILE_ONLY;
        break;
      case DIRECTORY_ONLY:
        comboBoxMode = ProjectFileTypeComboBox.Mode.DIRECTORY_ONLY;
        break;
      case FILE_AND_DIRECTORY:
        comboBoxMode = ProjectFileTypeComboBox.Mode.FILE_AND_DIRECTORY;
        break;
      default:
        String s = String.format("Modo %s invlido.\n", mode);
        throw new IllegalStateException(s);
      }
      /* Tipos de arquivos */
      typeComboBox = new ProjectFileTypeComboBox(comboBoxMode, true);
      if (defaultFileType != null && !defaultFileType.isEmpty()) {
        typeComboBox.selectTypeCode(defaultFileType);
        filterTree();
      }
      typeComboBox.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          filterTree();
        }
      });
      JPanel entryPanel = new JPanel(new GridBagLayout());
      Insets margins = new Insets(5, 5, 5, 5);
      Border border =
        BorderFactory.createTitledBorder(LNG
          .get("TreeFilterPanel.border.title"));
      entryPanel.setBorder(border);
      entryPanel.add(new JLabel(LNG.get("TreeFilterPanel.label.name")),
        new GBC(0, 0).insets(margins).west());
      entryPanel.add(filterTextField, new GBC(1, 0).insets(margins).west()
        .weightx(1).horizontal());
      entryPanel.add(clearFieldButton, new GBC(2, 0).insets(margins).west());
      entryPanel.add(new JLabel(LNG.get("TreeFilterPanel.label.type")),
        new GBC(0, 1).insets(margins).west());
      entryPanel.add(typeComboBox, new GBC(1, 1).insets(margins).west()
        .weightx(1).width(2).horizontal());
      this.add(entryPanel, new GBC(0, 0).both());
    }

    /**
     * Adiciona um filtro de entrada ao componente especificado. O usurio no
     * conseguir digitar nada alm dos caracteres permitidos.
     *
     * @param cmp componente cuja entrada ser filtrada.
     */
    private void filterInput(final Component cmp) {
      KeyboardFocusManager.getCurrentKeyboardFocusManager()
        .addKeyEventDispatcher(new KeyEventDispatcher() {
          /**
           * Desconsidera os eventos de teclado que forem dirigidos para um
           * determinado componente e no forem considerados "vlidos".
           *
           * @param e informaes sobre o evento ocorrido.
           *
           * @return true se o evento puder ser repassado, false se ele tiver
           *         que ser descartado.
           */
          @Override
          public boolean dispatchKeyEvent(KeyEvent e) {
            boolean discardEvent = false;
            if (e.getID() == KeyEvent.KEY_TYPED) {
              if (cmp.isFocusOwner() && !isValid(e.getKeyChar())) {
                discardEvent = true;
              }
            }
            return discardEvent;
          }

          /**
           * Verifica se o caracter especificado  vlido. So considerados
           * caracteres vlidos dgitos, letras, ".", "_", "?", "*" e "$".
           *
           * @param keyChar caracter a ser verificado.
           *
           * @return true se o caracter for vlido.
           */
          private boolean isValid(char keyChar) {
            if (Character.isLetterOrDigit(keyChar)) {
              return true;
            }
            else if ("?*$._".indexOf(keyChar) >= 0) {
              return true;
            }
            return false;
          }
        });
    }

    /**
     * Filtra a rvore de projetos, considerando o filtro para nome digitado
     * pelo usurio na caixa de texto e a seleo de tipo de arquivo.
     */
    private void filterTree() {
      String nameFilter = filterTextField.getText();
      String typeFilter;
      if (typeComboBox.isAllItemSelected()) {
        typeFilter = null;
      }
      else {
        typeFilter = typeComboBox.getSelectedTypeCode();
      }
      ProjectFileFilter filter =
        ProjectFileChooser.this.getVisualFilter(nameFilter, typeFilter);
      projectTree.setVisualFilter(filter);
      filter =
        ProjectFileChooser.this.getSelectionFilter(nameFilter, typeFilter);
      projectTree.setSelectionFilter(filter);
    }

    /**
     * Adiciona um objeto interesado em mudanas no estado deste
     * <i>singleton</i>.
     *
     * @param l objeto interessado em mudanas neste objeto.
     */
    public void addListener(TreeFilterPanelListener l) {
      listeners.add(l);
    }

    /**
     * Remove um objeto interesado em mudanas no estado deste <i>singleton</i>.
     *
     * @param l objeto interessado em mudanas neste objeto.
     *
     * @return true se o objeto tiver sido removido, false caso ele no tenha
     *         sido encontrado na lista.
     */
    public boolean removeListener(TreeFilterPanelListener l) {
      return listeners.remove(l);
    }

    /**
     * <p>
     * Esconde ou exibe o painel, de acordo com o parmetro especificado.
     * </p>
     * <p>
     * Quando exibe, transfere o foco para a caixa de texto. Quando esconde,
     * remove o filtro.
     * </p>
     * <p>
     * Notifica os observadores sobre a mudana no seu estado.
     * </p>
     */
    @Override
    public void setVisible(boolean visible) {
      super.setVisible(visible);
      if (visible) {
        filterTextField.requestFocusInWindow();
      }
      else {
        /*
         * como estamos escondendo o painel, devemos desativar o filtro
         * (retornando-o para ALL_FILES caso esteja com outro valor)
         */
        String filterText = filterTextField.getText();
        filterTextField.setText(null);
        if (!filterText.isEmpty()
          || !typeComboBox.isAllItemSelected()) {
          typeComboBox.selectAllItem();
        }
      }
      for (int i = 0; i < listeners.size(); i++) {
        listeners.get(i).visibilityChanged(isVisible());
      }
    }

  }

}
