/*
 * $Id$
 */

package csbase.client.applications.projectsmanager.models;

import javax.swing.ImageIcon;

import csbase.client.applications.projectsmanager.ProjectsManagerUI;
import csbase.logic.ProjectAllocationState;

/**
 * Status de alocacao de projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public enum ProjectSpaceAllocation {

  /**
   * Sem alocao (projeto comum)
   */
  NO_ALLOCATION(ProjectAllocationState.NO_ALLOCATION,
                ProjectsManagerUI.NO_ALLOCATION),

  /**
   * Aguardando alocao de rea.
   */
  WAITING_ALLOCATION(ProjectAllocationState.WAITING_AREA_ALLOCATION,
                     ProjectsManagerUI.ALLOCATED_BLOCKED),

  /**
   * Projeto removido, aguardando desalocao de rea.
   */
  WAITING_DEALLOCATION(ProjectAllocationState.WAITING_AREA_FREE,
                       ProjectsManagerUI.WAITING_DEALLOCATION),

  /**
   * rea alocada.
   */
  ALLOCATED(ProjectAllocationState.UNLOCKED_WITH_AREA_ALLOCATED,
            ProjectsManagerUI.ALLOCATED_FREE);

  /**
   * Estado de alocao do projeto.
   */
  final private ProjectAllocationState state;

  /**
   * cone referente ao status do projeto.
   */
  final private ImageIcon icon;

  /**
   * Construtor
   * 
   * @param state Estado de alocao do projeto.
   * @param icon icone
   */
  private ProjectSpaceAllocation(final ProjectAllocationState state,
    final ImageIcon icon) {
    this.state = state;
    this.icon = icon;
  }

  /**
   * Retorna o item da enumeracao correspondente ao espaco informado.
   * 
   * @param space espaco possivelmente alocado.
   * @return O item de enumeracao referente ao espaco informado.
   */
  final static public ProjectSpaceAllocation getItemFromSpace(final long space) {
    if (space == 0) {
      return NO_ALLOCATION;
    }

    if (space == -1) {
      return WAITING_ALLOCATION;
    }

    return ALLOCATED;
  }

  /**
   * Retorna item de status de alocao do projeto em funo de estado do
   * projeto representado por enumerao ProjectAllocationState.
   * 
   * TODO: Unificar ProjectSpaceAllocation e ProjectAllocationState? Os cones
   * que representam os estados so a nica coisa que diferencia as duas
   * enumeraes (ProjectSpaceAllocation inclusive referencia item de
   * ProjectAllocationState).
   * 
   * @param projectAllocationState Item de status de alocao.
   * @return Estado do projeto representado por enumerao
   *         ProjectAllocationState.
   */
  final static public ProjectSpaceAllocation getItemFromAllocationState(
    final ProjectAllocationState projectAllocationState) {
    for (ProjectSpaceAllocation psa : values()) {
      if (psa.getState() == projectAllocationState) {
        return psa;
      }
    }
    return null;
  }

  /**
   * Imagem do escopo.
   * 
   * @return a imagem
   */
  final public ImageIcon getIcon() {
    return icon;
  }

  /**
   * Mtodo de busca de uma string de internacionalizao dentro do escopo da
   * ao conforme o padro "<code>NomeDaClasse.TagDesejada</code>"
   * 
   * @param tag tag desejada
   * @return o texto de internacionalizao
   */
  final private String getClassString(final String tag) {
    return ProjectsManagerUI.getClassString(this.getClass(), tag);
  }

  /**
   * Mtodo de teste de uma string de internacionalizao dentro do escopo da
   * ao conforme o padro "<code>NomeDaClasse.TagDesejada</code>"
   * 
   * @param tag tag desejada
   * @return o texto de internacionalizao
   */
  final private boolean hasString(final String tag) {
    return ProjectsManagerUI.hasClassString(this.getClass(), tag);
  }

  /**
   * @return O identificador deste campo da enumerao.
   */
  public String getDescription() {
    final String tag = this.name() + ".description";
    if (hasString(tag))
      return getClassString(tag);
    return null;
  }

  /**
   * Consulta ao ttulo
   * 
   * @return o ttulo
   */
  final public String getTitle() {
    final String tag = this.name() + ".title";
    if (hasString(tag))
      return getClassString(tag);
    return null;
  }

  /**
   * Indica se o projeto pode ser aberto. Projetos que esto aguardando alocao
   * / desalocao de espao no podem ser abertos.
   * 
   * @return Boolean indicando se o projeto pode ser aberto.
   */
  public boolean isOpenable() {
    if (this == NO_ALLOCATION || this == ALLOCATED) {
      return true;
    }

    return false;
  }

  /**
   * Retorna o estado de alocao do projeto.
   * 
   * @return Estado de alocao do projeto.
   */
  public ProjectAllocationState getState() {
    return state;
  }

}
