/*
 * RoleManagerPanel.java
 * 
 * $Author: oikawa $ $Revision: 150777 $ - $Date: 2008-04-18 09:06:37 -0300
 * (Fri, 18 Apr 2008) $
 */
package csbase.client.ias;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.table.TableModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.remote.srvproxies.RoleProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.SwingObserverWrapper;
import csbase.logic.Role;

/**
 * A classe <code>RoleManagerPanel</code> representa o painel de administrao
 * dos perfis dos usurios. Mostra uma tabela com o nome e descrio de cada
 * perfil. O administrador pode inclui e remover um perfil, alm de modificar os
 * dados de um perfil qualquer.
 * 
 * @author Marcus Leal
 * @version $Revision: 150777 $
 */
public class RoleManagerPanel extends ManagerPanel {
  /** A janela onde esse painel est sendo usado. */
  private JFrame parent;
  private JButton associateButton;

  // Constante utilizada para testes de condies de erro.
  // 0: funcionamento normal
  // 1: exibe a mensagem de comunicao com o servidor e dispara o tratamento 
  //    para perfil no encontrado
  private static final int debug = 0;

  /**
   * Constri um painel de gerncia de perfis cadastrados no sistema
   * 
   * @param parent A janela onde esse painel est sendo usado.
   */
  public RoleManagerPanel(JFrame parent) {
    this.parent = parent;
    Vector<Role> roles =
      RoleProxy.getAllRoles(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ADMIN_DATA"));
    if (roles == null) {
      roles = new Vector<Role>();
    }
    Collections.sort(roles, Role.getNameComparator());
    ObjectTableProvider provider = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        Role role = (Role) item;
        if (role == null) {
          return null;
        }
        return new Object[] { role.getId(), role.getName(),
            role.getDescription() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { null, LNG.get("IAS_ROLE_NAME"),
            LNG.get("IAS_ROLE_DESCRIPTION") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class<?>[] { Object.class, String.class, String.class };
      }
    };
    TableModel model = new ObjectTableModel<Role>(roles, provider);
    Comparator<?>[] comparators =
      { null, ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase() };
    make(model, comparators, false);
    Role.addObserver(new SwingObserverWrapper(this));
  }

  /**
   * Remove o observador associado  classe <code>Role</code>.
   */
  @Override
  public void beforeClose() {
    Role.deleteObserver(new SwingObserverWrapper(this));
  }

  /**
   * Exibe a janela de criao de perfil.
   */
  @Override
  public void add() {
    new RoleDefinitionDialog(parent).showCreateDialog();
  }

  /**
   * Exibe a janela de modificao de perfil.
   * 
   * @param id identificador do perfil a ser modificado.
   */
  @Override
  public void modify(Object id) {
    Role role = getRole(id);
    if (role == null) {
      return;
    }
    new RoleDefinitionDialog(parent).showUpdateDialog(role);
  }

  /**
   * Obtm o perfil identificado pelo parmetro especificado. Caso o perfil no
   * seja encontrado, exibe mensagem de erro adequada.
   * 
   * @param id identificador do perfil.
   * 
   * @return perfil ou <code>null</code> caso este no seja encontrado.
   */
  private Role getRole(Object id) {
    Role role =
      RoleProxy.getRole(parent, parent.getTitle(), LNG.get("IAS_WAITING_ROLE"),
        id);
    if (debug == 1) { //@debug
      displayMessage(parent.getTitle(), LNG.get("IAS_WAITING_ROLE"));
      role = null; // @debug
    } //@debug    
    if (role == null) {
      displayError(LNG.get("IAS_ROLEMGR_ROLE_NOT_FOUND"));
    }
    return role;
  }

  /**
   * Exibe a janela de remoo de perfil.
   * 
   * @param id identificador do perfil a ser removido.
   */
  @Override
  public void delete(Object id) {
    Role role = getRole(id);
    if (role == null) {
      return;
    }
    String roleName = role.getName();
    int res =
      StandardDialogs.showYesNoDialog(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_ROLEMGR_ROLE_REMOVAL_CONFIRMATION"),
          new Object[] { roleName }));
    if (res == JOptionPane.YES_OPTION) {
      RoleProxy.deleteRole(parent, parent.getTitle(), MessageFormat.format(
        LNG.get("IAS_ROLEMGR_WAITING_REMOVE_ROLE"), new Object[] { roleName }),
        role.getId());
    }
  }

  /**
   * Mostra uma janela com uma mensagem.
   * 
   * @param title ttulo da janela de mensagem
   * @param message mensagem a ser exibida
   */
  protected void displayMessage(String title, String message) {
    StandardDialogs.showInfoDialog(parent, title, message);
  }

  /**
   * Mostra uma janela com uma mensagem de erro.
   * 
   * @param message mensagem de erro a ser exibida.
   */
  protected void displayError(String message) {
    StandardErrorDialogs.showErrorDialog(parent, parent.getTitle(), message);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel makeButtons() {
    JPanel panel = super.makeButtons();
    panel.add(new JPanel());
    this.createAssociateButton(panel, false);
    return panel;
  }

  private void createAssociateButton(JPanel panel, boolean enabled) {
    this.associateButton = new JButton(LNG.get("IAS_ROLE_USERS_BUTTON"));
    this.associateButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int row = getTable().getSelectedRow();
        Object id = getRowId(row);
        associate(id);
      }
    });
    this.associateButton.setEnabled(enabled);
    panel.add(this.associateButton);
    this.addButtonToSelectionListener(this.associateButton);
  }

  protected void associate(Object id) {
    Role role = getRole(id);
    if (role == null) {
      return;
    }
    new RoleUserDefinitionDialog(parent).showDialog(role);
  }
}
