/*
 * MultiSelectionFileTypeChangeDialog.java
 * 
 * $Author: fpina $ $Revision: 175310 $ - $Date: 2007-10-10 15:45:37 -0300
 * (Wed, 10 Oct 2007) $
 */
package csbase.client.project.action;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.project.ProjectFileTypeComboBox;
import csbase.client.project.ProjectFileTypeComboBox.Mode;
import csbase.client.util.ClientUtilities;
import csbase.exception.BugException;
import csbase.logic.ClientProjectFile;

/**
 * A classe <code>FileTypeChangeDialog</code> modela uma janela que permite ao
 * usurio modificar o tipo de um <code>ProjectFile</code>.
 */
public class MultiSelectionFileTypeChangeDialog {

  /**
   * Dilogo principal.
   */
  private DesktopComponentDialog dialog = null;

  /**
   * Janela a partir do qual esse dilogo foi chamado.
   */
  private Window owner = null;

  /**
   * O boto default que fecha o dilogo.
   */
  private JButton defaultButton;

  /**
   * O arquivo cujo tipo vai ser modificado.
   */
  private ClientProjectFile[] files;

  /**
   * Nomes dos arquivos.
   */
  private String[] fileNames;

  /**
   * Campo de tipos de arquivo.
   */
  private ProjectFileTypeComboBox fileTypeList;

  /**
   * Indica se o dilogo ir operar sobre arquivos ou diretrios.
   */
  private boolean isDirectory;

  /**
   * Mostra a janela que permite ao usurio modificar o tipo do arquivo.
   * 
   * @param owner A janela a partir do qual esse dilogo foi chamado.
   * @param files O arquivo cujo tipo vai ser alterado.
   */
  public static void show(Window owner, ClientProjectFile[] files) {
    new MultiSelectionFileTypeChangeDialog(owner, files);
  }

  /**
   * Cria o dilogo.
   * 
   * @return .
   */
  private DesktopComponentDialog makeDialog() {
    /* Cria um dilogo no modal */
    DesktopComponentDialog d = new DesktopComponentDialog(owner, getTitle());
    d.setModal(false);

    /* Cria o painel principal do dilogo */
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(makePropertiesPanel(d.getBackground()));
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);
    d.getContentPane().add(mainPanel);
    d.pack();
    d.center(owner);
    d.getRootPane().setDefaultButton(defaultButton);
    return d;
  }

  /**
   * Cria o painel com as propriedades do arquivo.
   * 
   * @param background A cor de fundo usada pelos componentes.
   * 
   * @return .
   */
  private JPanel makePropertiesPanel(Color background) {
    /* Campo com o nome do arquivo */
    JList selectedFilesList = new JList(fileNames);
    JScrollPane selectedfiles = new JScrollPane(selectedFilesList);
    selectedFilesList.setEnabled(false);

    Mode mode = isDirectory ? Mode.DIRECTORY_ONLY : Mode.FILE_ONLY;
    /* Tipos de arquivos */
    fileTypeList = new ProjectFileTypeComboBox(mode, false);
    JPanel panel = new JPanel(new GridBagLayout());
    GridBagConstraints constraints = new GridBagConstraints();
    constraints.insets = new Insets(10, 10, 10, 10);
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 1;
    constraints.weighty = 0;
    constraints.gridx = 0;
    constraints.gridy = 0;
    panel.add(new JLabel(LNG.get("PRJ_FILE_NAME")), constraints);
    constraints.fill = GridBagConstraints.BOTH;
    constraints.weightx = 1;
    constraints.weighty = 1;
    constraints.gridx = 1;
    constraints.gridy = 0;
    panel.add(selectedfiles, constraints);
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 1;
    constraints.weighty = 0;
    constraints.gridx = 0;
    constraints.gridy = 1;
    panel.add(new JLabel(LNG.get("PRJ_FILE_NEW_TYPE")), constraints);
    constraints.fill = GridBagConstraints.HORIZONTAL;
    constraints.weightx = 1;
    constraints.weighty = 0;
    constraints.gridx = 1;
    constraints.gridy = 1;
    panel.add(fileTypeList, constraints);
    return panel;
  }

  /**
   * Cria o painel com os botes.
   * 
   * @return .
   */
  private JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    defaultButton = new JButton(LNG.get("PRJ_FILE_TYPE_CHANGE_CONFIRMATION"));
    panel.add(defaultButton);
    defaultButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        changeType();
      }
    });

    JButton cancelButton = new JButton(LNG.get("PRJ_CANCEL"));
    panel.add(cancelButton);
    cancelButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        dialog.close();
      }
    });

    JButton[] allButtons = { defaultButton, cancelButton };
    ClientUtilities.adjustEqualSizes(allButtons);
    return panel;
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de exibio.
   * 
   * @return O ttulo da janela.
   */
  protected String getTitle() {
    return LNG.get("PRJ_PROJECT_FILE_CHANGE_TYPE_TITLE");
  }

  /**
   * Recupera as propriedades do arquivo.
   */
  private void retrieveFileProperties() {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        for (int i = 0; i < files.length; i++) {
          fileNames[i] = files[i].getName();
        }
      }
    };
    task.execute(owner, getTitle(),
      LNG.get("PRJ_WAITING_OBTAIN_FILE_PROPERTIES"));
  }

  /**
   * Consulta aos dados digitados pelo usurio.
   * 
   * @return Novo tipo de arquivo.
   */
  private String getNewType() {
    return fileTypeList.getSelectedTypeCode();
  }

  /**
   * Muda o tipo do arquivo.
   */
  private void changeType() {
    final String newType = getNewType();
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        for (int i = 0; i < files.length; i++) {
          if (!newType.equals(files[i].getType())) {
            files[i].changeType(newType);
          }
        }
      }
    };
    if (task.execute(owner, getTitle(),
      LNG.get("PRJ_WAITING_CHANGE_FILES_TYPE"))) {
      dialog.close();
    }
  }

  /**
   * Constri o dilogo.
   * 
   * @param owner A janela pai a partir da qual esse dilogo  chamado.
   * @param files O arquivo cujo tipo vai ser alterado.
   */
  private MultiSelectionFileTypeChangeDialog(Window owner,
    ClientProjectFile[] files) {
    Boolean isDirectory = null;
    for (ClientProjectFile file : files) {
      if (isDirectory == null) {
        isDirectory = file.isDirectory();
      }
      else if (!isDirectory.equals(file.isDirectory())) {
        throw new BugException("No  permitido misturar diretrios e arquivos.");
      }
    }
    this.isDirectory = isDirectory;
    this.owner = owner;
    this.files = files;
    this.fileNames = new String[files.length];
    retrieveFileProperties();
    dialog = makeDialog();
    dialog.setVisible(true);
  }
}
