package csbase.client.applications.projectsmanager.panels;

import java.awt.GridBagLayout;
import java.util.List;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;

/**
 * Painel que apresenta informaes sobre uma seleo de mltiplas linhas na
 * tabela de projetos.
 * 
 * @author jnlopes
 * 
 */
class MultipleSelectionPanel extends AbstractProjectManagerInfoPanel {

  /**
   * Array com todos os projetos selecionados.
   */
  private List<ProjectsManagerData> projectsManagerData;

  /**
   * Nmero de projetos selecionados
   */
  private JTextField numberOfProjectsText;

  /**
   * Espao alocado pelos projetos selecionados.
   */
  private JTextField projectsSizeText;

  /**
   * Espao usado pelos projetos selecionados.
   */
  private JTextField projectsSpaceText;

  /**
   * Construtor.
   * 
   * @param projectsManager A aplicao.
   */
  MultipleSelectionPanel(final ProjectsManager projectsManager) {
    super(projectsManager);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void initComponents() {
    numberOfProjectsText = new JTextField();
    projectsSizeText = new JTextField();
    projectsSpaceText = new JTextField();
    numberOfProjectsText.setEditable(false);
    numberOfProjectsText.setHorizontalAlignment(SwingConstants.RIGHT);
    projectsSizeText.setEditable(false);
    projectsSizeText.setHorizontalAlignment(SwingConstants.RIGHT);
    projectsSpaceText.setEditable(false);
    projectsSpaceText.setHorizontalAlignment(SwingConstants.RIGHT);
  }

  /**
   * Monta o painel.
   */
  @Override
  void buildPanel() {
    final int T = 12;
    final int TI = 6;
    final int L = 12;
    final int B = 17;
    final int R = 11;
    this.setLayout(new GridBagLayout());

    final JPanel panel = new JPanel(new GridBagLayout());
    final JLabel numOfProjectsLabel =
      new JLabel(
        getString("MultipleSelectionPanel.project.number.of.projects.label"));
    final JLabel projectsSizeLabel =
      new JLabel(
        getString("MultipleSelectionPanel.project.allocated.space.label"));
    final JLabel projectsSpaceLabel =
      new JLabel(
        getString("MultipleSelectionPanel.project.occupied.space.label"));

    panel.add(numOfProjectsLabel, new GBC(0, 0).west().insets(T, L, 0, R));
    panel.add(numberOfProjectsText, new GBC(1, 0).west().insets(T, L, 0, R)
      .horizontal());

    panel.add(projectsSizeLabel, new GBC(0, 1).west().insets(TI, L, 0, R));
    panel.add(projectsSizeText, new GBC(1, 1).west().insets(TI, L, 0, R)
      .horizontal());

    panel.add(projectsSpaceLabel, new GBC(0, 2).west().insets(TI, L, B, R));
    panel.add(projectsSpaceText, new GBC(1, 2).west().insets(TI, L, B, R)
      .horizontal());

    BorderUtil.setTitledBorder(this,
      getString("MultipleSelectionPanel.project.info.title"));

    add(panel, new GBC(0, 0).horizontal(1));
    add(new JPanel(), new GBC(1, 0).horizontal(3));

    validate();
    repaint();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void updateFields() {
    numberOfProjectsText.setText(String.valueOf(projectsManagerData.size()));
    projectsSizeText.setText(String.valueOf(calculateSummedUpAllocatedSpace())
      + " " + getString("MultipleSelectionPanel.GBytes"));

    double occupation = calculateSummedUpOccupiedSpace();
    final String unit;
    if (occupation > 1024) {
      occupation = occupation / 1024;
      unit = getString("MultipleSelectionPanel.GBytes");
    }
    else {
      unit = getString("MultipleSelectionPanel.MBytes");
    }

    occupation = Math.round(occupation * 1000) / 1000.0;
    projectsSpaceText.setText(String.valueOf(occupation) + " " + unit);

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clearFields() {
    numberOfProjectsText.setText(null);
    projectsSizeText.setText(null);
    projectsSpaceText.setText(null);
  }

  /**
   * Calcula o espaco alocado pelos projetos selecionados na tabela.
   * 
   * @return espaco total alocado pelos projetos selecionados na tabela.
   */
  private double calculateSummedUpAllocatedSpace() {
    double total = 0;

    for (ProjectsManagerData pmd : this.projectsManagerData) {
      final double aux = pmd.getAllocatedSpace();
      total = total + aux;
    }

    return total;
  }

  /**
   * Calcula o espaco ocupado pelos projetos selecionados na tabela.
   * 
   * @return espaco total ocupado pelos projetos selecionados na tabela.
   */
  private double calculateSummedUpOccupiedSpace() {

    double total = 0;
    for (ProjectsManagerData pmd : this.projectsManagerData) {
      final double aux = pmd.getOccupiedSpace();
      if (aux > 0) {
        total = total + aux;
      }
    }
    return total;
  }

  /**
   * Define array de projetos selecionados e solicita atualizao dos
   * componentes.
   * 
   * @param projectsManagerData Array de projetos selecionados.
   */
  public void setProjectsManagerData(
    final List<ProjectsManagerData> projectsManagerData) {
    this.projectsManagerData = projectsManagerData;
    updateFields();
  }
}
