package csbase.client.project;

import java.awt.Window;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.MessageFormat;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.LocalTask;
import csbase.client.util.StandardErrorDialogs;

/**
 * Tarefa responsvel por exportar o contedo de um String para para um arquivo
 * do sistema de arquivos do cliente. O encoding utilizado  o padro da
 * plataforma.
 * 
 * @see String#getBytes()
 */
public class ExportStringTask extends LocalTask<Void> {

  /**
   * Ttulo do dilogo de exportao
   */
  private static final String EXPORT_TITLE = "PRJ_PROJECT_FILE_EXPORT_TITLE";

  /**
   * Prefixo para ttulos de dilogos de erros
   */
  private static final String ERROR_PREFIX = "cabase.title.error";

  /**
   * Mensagem em caso de erro de I/O
   */
  private static final String EXPORT_IO_ERROR =
    "PRJ_PROJECT_FILE_EXPORT_IO_ERROR";

  /**
   * Tamanho do buffer de I/O.
   */
  private static final int BUFFER_SIZE = 48 * 1024;

  /**
   * Mensagem para passo da transferncia em execuo
   */
  private static final String EXPORT_STEP = "PRJ_PROJECT_FILE_EXPORT_STEP";

  /**
   * janela onde sero lanadas as dialogs de erro
   */
  private Window window;

  /**
   * Contedo a ser exportado.
   */
  private String content;

  /**
   * Diretrio de destino para os arquivos/diretrios a serem exportados. O
   * prefixo "root" tem a mesma semntica do campo <code>rootFile</code>:
   * trata-se do diretrio que servir de destino para a transferncia como um
   * todo.
   */
  private File rootTargetFile;

  /**
   * Volume total ocupado pelos arquivos a serem exportados.
   */
  private long totalSize;

  /**
   * Cria a tarefa.
   * 
   * @param targetFile arquivo/diretrio de destino no cliente.
   * @param content dados a serem gravados no arquivo de destino
   * @param win janela onde sero lanadas as dialogs de erro.
   */
  public ExportStringTask(File targetFile, String content, Window win) {
    this.window = win;
    this.content = content;
    this.rootTargetFile = targetFile;
  }

  /**
   * Especifica como  feito o tratamento de erros. Trata somente
   * <code>IOException</code>, outros erros so repassados para o mtodo pai na
   * hierarquia.
   */
  @Override
  protected void handleError(Exception error) {
    if (error instanceof IOException) {
      String msg =
        MessageFormat.format(LNG.get(EXPORT_IO_ERROR),
          new Object[] { rootTargetFile.getName() });
      StandardErrorDialogs.showErrorDialog(window, LNG.get(ERROR_PREFIX)
        + " - " + LNG.get(EXPORT_TITLE), msg);
      return;
    }
    super.handleError(error);
  }

  /**
   * Executa a tarefa. Transfere o contedo do arquivo ou diretrio para o
   * sistema de arquivos do cliente.
   */
  @Override
  protected void performTask() throws Exception {
    transfer(rootTargetFile);
  }

  /**
   * Transfere o arquivo ou diretrio selecionado para o sistema de arquivos do
   * cliente.
   * 
   * @param targetFile arquivo/diretrio de destino no cliente.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private void transfer(File targetFile) throws IOException {
    totalSize += content.length();
    transferFile(targetFile);
  }

  /**
   * Transfere o contedo de um arquivo para o sistema de arquivos do cliente.
   * 
   * @param targetFile arquivo de destino no cliente.
   * 
   * @throws IOException em caso de erro de I/O.
   */
  private void transferFile(File targetFile) throws IOException {
    String msg =
      MessageFormat.format(LNG.get(EXPORT_STEP),
        new Object[] { targetFile.getName() });
    setStepText(msg);
    targetFile.createNewFile();
    final BufferedOutputStream out =
      new BufferedOutputStream(new FileOutputStream(targetFile));

    byte[] buf = content.getBytes();
    final ByteArrayInputStream inputStream = new ByteArrayInputStream(buf);
    try {
      byte[] buffer = new byte[BUFFER_SIZE];
      int readBytes = 0;
      while ((readBytes = inputStream.read(buffer)) > 0) {
        out.write(buffer, 0, readBytes);
      }
      out.flush();
    }
    finally {
      out.close();
      inputStream.close();
    }
  }

  /**
   * Retorna o volume total transferido.
   * 
   * @return o volume total transferido.
   */
  public long getTotalSize() {
    return totalSize;
  }
}
