package csbase.client.algorithms.commands.view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.AbstractAction;
import javax.swing.JFileChooser;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.Timer;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.algorithms.commands.cache.CommandsCache;
import csbase.client.algorithms.commands.cache.events.AbstractCommandUpdatedEventListener;
import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent.Type;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.LocalTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommandInfo;
import csbase.logic.CommandStatus;

/**
 * Representa uma aba de log consolidado na viso de um comando. Permite que
 * vrios arquivos de log possam ser mostrados numa mesma aba.
 */
public abstract class AbstractConsolidatedLogTab extends Tab {
  /**
   * rea de texto aonde  exibido o log.
   */
  protected JTextArea textArea;

  /**
   * Reloader do log.
   */
  private Timer reloader;

  /**
   * Indicativo de no seleo.
   */
  private AtomicBoolean wasNeverSelected;

  /**
   * Boto para reload automtico.
   */
  private JToggleButton reloaderBtn;

  /**
   * Boto para exportao de log consolidado.
   */
  private JToggleButton exportBtn;

  /**
   * Informaes sobre o comando cujo log deve ser apresentado.
   */
  protected CommandInfo command;

  /**
   * Construtor
   * 
   * @param commandInfo Informaes sobre o comando cujo log deve ser
   *        apresentado.
   */
  public AbstractConsolidatedLogTab(CommandInfo commandInfo) {
    this.command = commandInfo;
    setLayout(new BorderLayout());

    textArea = new JTextArea();
    textArea.setFont(new Font("monospaced", Font.PLAIN, 12));
    textArea.setEditable(false);
    textArea.setBackground(new Color(220, 220, 220));

    reloader = createReloader();
    wasNeverSelected = new AtomicBoolean(true);

    JToolBar toolBar = new JToolBar();
    toolBar.setFloatable(false);
    reloaderBtn = new JToggleButton();
    reloaderBtn.setSelected(reloader.isRunning());
    reloaderBtn.setAction(new ReloaderAction(reloaderBtn, reloader));
    toolBar.add(reloaderBtn);

    toolBar.add(new JToolBar.Separator());

    exportBtn = new JToggleButton();
    exportBtn.setSelected(false);
    exportBtn.setAction(new ExportAction(textArea));
    toolBar.add(exportBtn);

    add(toolBar, BorderLayout.NORTH);

    final JScrollPane scrollpane = new JScrollPane(textArea);
    scrollpane.setPreferredSize(new Dimension(600, 300));
    add(scrollpane, BorderLayout.CENTER);
    CommandsCache.getInstance().addEventListener(
      new AbstractCommandUpdatedEventListener(command.getProjectId(), command
        .getId()) {
        @SuppressWarnings("incomplete-switch")
        @Override
        protected void eventFired(Type type, CommandInfo commandInfo) {
          command = commandInfo;
          /*
           * Caso o commando tenha terminado sua execuo (independente do
           * porqu),  preciso parar de recarregar o arquivo de log
           */
          switch (type) {
            case end:
            case success:
            case error:
            case failed:
            case killed:
            case lost:
            case no_code:
            case removed:
              setSelected(false);
          }
        }
      });
  }

  /**
   * Retorna as informaes sobre o comando cujo log deve ser apresentado.
   * 
   * @return command Informaes sobre o comando cujo log deve ser apresentado.
   */
  public CommandInfo getCommand() {
    return command;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void setSelected(boolean selected) {
    super.setSelected(selected);

    /*
     * Se a tab de log est sendo selecionada pela primeira vez e o comando
     * ainda no terminou, a recarga de log deve estar ligada. Se este no for o
     * caso, a recarga deve ficar desligada se esta tab estiver sendo
     * desselecionada ou deve ficar no ltimo estado em que o usurio a deixou.
     */

    if (wasNeverSelected.compareAndSet(true, !selected)) {
      if (command.getStatus().equals(CommandStatus.FINISHED)) {
        /*
         * Carrega o texto para a visualizao, mas no liga a recarga
         * automtica.
         */
        loadLog();
      }
      else {
        reloader.start();
        reloaderBtn.setSelected(true);
      }
      return;
    }
    else if (!reloaderBtn.isSelected()) {
      return;
    }

    if (selected) {
      reloader.start();
      reloaderBtn.setSelected(true);
    }
    else {
      reloader.stop();
      reloaderBtn.setSelected(false);
    }
  }

  /**
   * Cria o {@link Timer} responsvel pela recarga do log.
   * 
   * @return um novo {@link Timer} responsvel pela recarga do log.
   */
  private Timer createReloader() {
    Timer reloadTimer = new Timer(3000, new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        loadLog();
      }
    });
    reloadTimer.setInitialDelay(0);

    return reloadTimer;
  }

  /**
   * Recarga do log na rea de texto.
   */
  protected abstract void loadLog();

}

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Ao para gerenciar o {@link Timer} de recarga do log e o estado de
 *         seleo do boto que detm esta ao.
 */
class ReloaderAction extends AbstractAction {

  /** Boto que detm esta ao. */
  private JToggleButton parent;
  /** Objeto responsvel pela recarga do log. */
  private Timer reloader;

  /**
   * Construtor.
   * 
   * @param parent Boto que detm esta ao
   * @param reloader Objeto responsvel pela recarga do log
   */
  public ReloaderAction(JToggleButton parent, Timer reloader) {
    super(null, ApplicationImages.ICON_REFRESH_16);
    putValue(SHORT_DESCRIPTION,
      LNG.get("FlowAlgorithmCommandView.tab.log.button.reload.tooltip"));

    this.parent = parent;
    this.reloader = reloader;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {
    if (reloader.isRunning()) {
      reloader.stop();
      parent.setSelected(false);
    }
    else {
      reloader.start();
      parent.setSelected(true);
    }
  }
}

/**
 * Ao para exportar o log consolidado.
 */
class ExportAction extends AbstractAction {

  /** rea de texto que contm o log consolidado. */
  private final JTextArea textArea;

  /**
   * File chooser nico para a exportao para manter o path do ltimo diretrio
   * escolhido.
   */
  private JFileChooser exportationFileChooser;

  /**
   * Construtor.
   * 
   * @param textArea rea de texto que contm o log consolidado
   */
  public ExportAction(final JTextArea textArea) {
    super(null, ApplicationImages.ICON_EXPORT_16);
    putValue(SHORT_DESCRIPTION,
      LNG.get("FlowAlgorithmCommandView.tab.log.button.export.tooltip"));
    this.textArea = textArea;
    this.exportationFileChooser = createCustomChooser();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(ActionEvent e) {

    // Obtm o arquivo local que conter o arquivo de log.
    final File localFile = getLocalFile();
    if (localFile == null) {
      return;
    }

    // Exporta o arquivo de log para o sistema de arquivos local.
    LocalTask<Void> task = new LocalTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        exportAsStream(localFile, textArea);
      }
    };

    // Executa a task para exportar o arquivo de log.
    Window window = DesktopFrame.getInstance().getDesktopFrame();
    String title =
      LNG.get("FlowAlgorithmCommandView.tab.log.button.export.title");
    String message =
      LNG.get("FlowAlgorithmCommandView.tab.log.button.export.log");
    boolean result = task.execute(window, title, message);
    if (result) {
      message =
        LNG.get("FlowAlgorithmCommandView.tab.log.button.export.log.success",
          new Object[] { localFile.getAbsolutePath() });
      StandardDialogs.showInfoDialog(window, title, message);
    }
    else {
      StandardErrorDialogs.showErrorDialog(window, title, task.getError());
    }
  }

  /**
   * Obtm o arquivo local que conter o arquivo de log.
   * 
   * @return Arquivo local que conter o arquivo de log.
   */
  private File getLocalFile() {
    final int ret =
      exportationFileChooser
        .showDialog(
          textArea.getParent(),
          LNG
            .get("FlowAlgorithmCommandView.tab.log.button.export.chooser.approve.button"));
    // Dilogo para a seleo do diretrio/arquivo de destino a ser exportado o
    // log.
    if (ret == JFileChooser.CANCEL_OPTION) {
      return null;
    }
    File localFile = exportationFileChooser.getSelectedFile();
    if (localFile.isDirectory()) {
      // O usurio selecionou um diretrio. O log ser gravado nesse diretrio
      // com o mesmo nome do arquivo de log remoto.
      String nameLocalFile = localFile.getAbsolutePath() + "/out.log";
      localFile = new File(nameLocalFile);
    }
    // Verifica se o arquivo de log j existe.
    if (localFile.exists()) {
      final String title =
        LNG.get("FlowAlgorithmCommandView.tab.log.button.export.title");
      final String[] args =
        new String[] { localFile.getName(),
            localFile.getParentFile().getAbsolutePath() };
      final String msg =
        LNG.get("FlowAlgorithmCommandView.tab.log.button.export.overwrite.msg",
          args);
      final int option = StandardDialogs.showYesNoDialog(textArea, title, msg);
      if (option == 1) {
        return null;
      }
    }
    // Para manter o path do ltimo diretrio escolhido.
    exportationFileChooser.setCurrentDirectory(localFile);
    return localFile;

  }

  /**
   * Exporta o arquivo de log como stream.
   * 
   * @param localFile Arquivo local que conter o arquivo de log.
   * @param textArea Log consolidado como string
   * @throws Exception Exceo lanada
   */
  private void exportAsStream(final File localFile, final JTextArea textArea)
    throws Exception {
    final int bufferSize = 48 * 1024;
    String consolidatedLog = textArea.getText();
    final BufferedInputStream in =
      new BufferedInputStream(new ByteArrayInputStream(
        consolidatedLog.getBytes()));
    final FileOutputStream fileStream = new FileOutputStream(localFile);
    BufferedOutputStream out = new BufferedOutputStream(fileStream);
    byte[] buffer = new byte[bufferSize];
    int len = 0;
    while ((len = in.read(buffer)) > 0) {
      out.write(buffer, 0, len);
    }
    out.flush();
    out.close();
    in.close();
  }

  /**
   * Instncia um file chooser personalizado para escolher um arquivo para
   * exportao.
   * 
   * @return O JFileChooser.
   */
  private JFileChooser createCustomChooser() {
    exportationFileChooser = new JFileChooser();
    exportationFileChooser.setMultiSelectionEnabled(false);
    exportationFileChooser
      .setApproveButtonText(LNG
        .get("FlowAlgorithmCommandView.tab.log.button.export.chooser.approve.button"));
    exportationFileChooser
      .setApproveButtonToolTipText(LNG
        .get("FlowAlgorithmCommandView.tab.log.button.export.chooser.approve.button.tooltip"));
    exportationFileChooser.setDialogTitle(LNG
      .get("FlowAlgorithmCommandView.tab.log.button.export.chooser.title"));
    exportationFileChooser
      .setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
    return exportationFileChooser;
  }

}