package csbase.client.applications.algorithmsmanager.versiontree;

/**
 * $Id$
 */

import java.awt.Window;
import java.io.File;
import java.text.Collator;
import java.util.Collections;

import javax.swing.ImageIcon;
import javax.swing.JPopupMenu;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;

/**
 * Representao abstrata de um n de uma {@link VersionTree rvore de verses}
 * 
 * @author Tecgraf / PUC-Rio
 */
public abstract class AbstractVersionTreeNode extends DefaultMutableTreeNode
  implements Comparable<MutableTreeNode> {

  /**
   * Menu popup a ser exibido se o usurio selecionar o n com o boto direito
   * do mouse.
   */
  private static final JPopupMenu NULL_POPUP_MENU = new JPopupMenu() {
    @Override
    public void show() {
    }
  };

  /** rvore que detm este n */
  private VersionTree tree;

  /**
   * Cria um inicializado com o objeto passado. O mtodo
   * {@link Object#toString()} do objeto ir determinar o nome deste n. <br>
   * O n criado  inicialmente desprovido de pai ou filhos, porm que permite
   * receber filhos.
   * 
   * @param tree a rvore
   * @param userObject objeto que representa o n.
   */
  public AbstractVersionTreeNode(VersionTree tree, Object userObject) {
    super(userObject);

    this.tree = tree;
  }

  /**
   * Obtm a rvore que detm este n.
   * 
   * @return a rvore que detm este n.
   */
  public final VersionTree getTree() {
    return tree;
  }

  /**
   * Obtm a janela que criou a rvore.
   * 
   * @return a janela que criou a rvore.
   */
  public final Window getWindow() {
    return tree.getOwner();
  }

  /**
   * Cria um menu pop-up para este n. <br>
   * A princpio este mtodo retorna um menu pop-up cujo mtodo
   * {@link JPopupMenu#show()} foi sobrescrito para no mostrar nada. Caso
   * queira que este n tenha um menu pop-up, sobrescreva este mtodo.
   * 
   * @return um menu pop-up para este n.
   */
  public JPopupMenu createPopupMenu() {
    return NULL_POPUP_MENU;
  }

  /**
   * <p>
   * Indica se este n pode ser selecionado em conjunto com o n passado.
   * </p>
   * <p>
   * O padro  retornar <tt>false</tt>. Para possibilitar a seleo deste n em
   * conjunto com outros, este mtodo deve ser sobrescrito.
   * </p>
   * 
   * @param other n a ser selecionado junto com esse.
   * 
   * @return <tt>true</tt> se este n puder ser selecionado em conjunto com o n
   *         passado.
   */
  public boolean allowMultipleSelection(AbstractVersionTreeNode other) {
    return true;
  }

  /**
   * Adiciona um n-filho a este n. A insero  feita na posio adequada
   * visando manter-se uma lista em ordem crescente.
   */
  @Override
  public void add(final MutableTreeNode newChild) {
    if (newChild != null && newChild.getParent() == this) {
      remove(newChild);
    }
    int index;
    if (children == null) {
      index = 0;
    }
    else {
      index = Collections.binarySearch(children, newChild);
    }
    if (index < 0) {
      index = -index - 1;
    }
    insert(newChild, index);
  }

  /**
   * {@inheritDoc}
   */
  public String getName() {

    StringBuilder name = new StringBuilder();

    TreeNode[] path = getPath();
    for (TreeNode node : path) {
      name.append(File.separator).append(node.toString());
    }

    return name.toString();
  }

  /**
   * Mtodo para comparao de ns. Compara os nomes sendo exibidos, permitindo
   * uma ordenao alfabtica crescente.
   * 
   * @see java.lang.Comparable#compareTo(java.lang.Object)
   */
  @Override
  public int compareTo(MutableTreeNode node) {
    return Collator.getInstance().compare(this.toString(), node.toString());
  }

  /**
   * <p>
   * Obtm a imagem a ser utilizada para representar este n.
   * </p>
   * <p>
   * Caso este mtodo retorne <tt>null</tt>, o renderizador ir desenhar o n da
   * forma padro de acordo com o {@link java.awt.Toolkit}
   * </p>
   * 
   * @return a imagem a ser utilizada para representar este n.
   */
  public abstract ImageIcon getImageIcon();
}
