package csbase.client.applications.projectsmanager.proxy;

import java.util.List;

import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.proxy.core.ProjectsManagerTask;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectAdminInfo;
import csbase.logic.ProjectAllocationState;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;

/**
 * Task que recarrega todas as informaes dos projetos selecionados, com
 * exceo do espao ocupado.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ReloadSelectedProjectsTask extends
  ProjectsManagerTask<List<ProjectsManagerData>> {

  /**
   * Array de projetos a serem atualizados.
   */
  private List<ProjectsManagerData> projectsManagerData;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   * @param projectsManagerData Lista de projetos a serem atualizados.
   */
  public ReloadSelectedProjectsTask(ProjectsManager projectsManager,
    List<ProjectsManagerData> projectsManagerData) {
    super(projectsManager);
    this.projectsManagerData = projectsManagerData;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void performTask() throws Exception {
    getProjectsData();
    setResult(projectsManagerData);
  }

  /**
   * Obtm dados dos projetos.
   * 
   * @throws Exception em caso de falha.
   */
  private void getProjectsData() throws Exception {
    ProjectServiceInterface psi = ClientRemoteLocator.projectService;
    for (int i = 0; i < projectsManagerData.size(); i++) {
      ProjectsManagerData pmd = projectsManagerData.get(i);
      ProjectAdminInfo projectAdminInfo =
        psi.getProjectAdminInfo(pmd.getProjectId());
      if (projectAdminInfo != null) {
        ProjectAllocationState state = projectAdminInfo.getState();
        ProjectSpaceAllocation spaceAllocation =
          ProjectSpaceAllocation.getItemFromAllocationState(state);
        pmd.setProjectSpaceAllocation(spaceAllocation);
        if (!projectAdminInfo.isUnlockedWithAreaAllocated()) {
          pmd.clearData();
          continue;
        }
      }
      CommonClientProject ccp = psi.openProject(pmd.getProjectId(), false);
      long allocatedSpace = ccp.getLockingAreaSize();
      if (allocatedSpace == 0) {
        pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.NO_ALLOCATION);
      }
      else if (allocatedSpace > 0) {
        pmd.setProjectSpaceAllocation(ProjectSpaceAllocation.ALLOCATED);
      }
      String userName = User.getName(ccp.getUserId());
      pmd.setOwnerName(userName);
      pmd.setProjectDescription(ccp.getDescription());
      pmd.setCreationDate(ccp.getCreationDate());
      pmd.setServerOwnerName(ccp.getOwnerServerName());
      pmd.setLocationInServer(ccp.getLocationInServer());
      pmd.setModificationDate(ccp.getLastModificationDate());
      pmd.setSharingType(ccp.getSharingType());
      pmd.setUsers(ccp.getUsersRO(), ccp.getUsersRW());
      ccp.close(false);
    }
  }
}
