package csbase.client.csdk.v1_0.application;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;
import java.util.Map;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.csdk.v1_0.CSDKContextFactory;
import csbase.client.csdk.v1_0.core.CSDKEnvironment;
import csbase.logic.applicationservice.ApplicationRegistry;
import csdk.v1_0.api.application.IApplication;
import csdk.v1_0.api.application.IApplicationContext;
import csdk.v1_0.api.application.IMessage;
import csdk.v1_0.api.application.IMessageSender;
import csdk.v1_0.api.core.ICSDKEnvironment;
import csdk.v1_0.api.core.IContext;
import csdk.v1_0.helper.application.Message;
import csdk.v1_0.helper.application.MessageSender;

/**
 * Classe que encapsula uma aplicao baseada no CSDK em uma aplicao nativa do
 * sistema.
 */
public class CSDKApplication extends ApplicationType {

  /**
   * Aplicao baseada no CSDK.
   */
  private IApplication application;

  /**
   * A interface com o ambiente CSDK do CSBase.
   */
  private CSDKEnvironment csdkEnv;

  /**
   * Construtor.
   * 
   * @param registry registro da aplicao.
   * @param appClass classe principal da aplicao.
   * @throws ApplicationException em caso de erro na criao da aplicao.
   */
  public CSDKApplication(ApplicationRegistry registry,
    Class<? extends IApplication> appClass) throws ApplicationException {
    super(registry.getId());
    this.csdkEnv = createCSDKEnvironment(registry, appClass);
    this.application = createApplicationClassInstance(registry, appClass);
  }

  /**
   * Cria o ambiente de interface com o CSDK para a aplicao.
   * 
   * @param registry o registro da aplicao.
   * @param appClass classe principal da aplicao.
   * @return o ambiente.
   * @throws ApplicationException em caso de erro ao criar os contextos.
   */
  private CSDKEnvironment createCSDKEnvironment(ApplicationRegistry registry,
    Class<? extends IApplication> appClass) throws ApplicationException {
    Client client = Client.getInstance();
    CSDKContextFactory csdkContextFactory = client.getCSDKContextFactory();
    Map<Class<? extends IContext>, IContext> contexts =
      csdkContextFactory
        .createCSDKContexts(getInstanceId(), appClass, registry);
    if (contexts == null || contexts.isEmpty()
      || !contexts.containsKey(IApplicationContext.class)) {
      throw new ApplicationException(
        "A fbrica de contextos precisa pelo menos criar o contexto obrigatrio "
          + IApplicationContext.class.getSimpleName());
    }
    return new CSDKEnvironment(contexts);
  }

  /**
   * Instancia a aplicao CSDK.
   * 
   * @param registry o registro da aplicao.
   * @param appClass a classe principal da aplicao.
   * @return a instncia.
   * @throws ApplicationException em caso de erro na instanciao.
   */
  private IApplication createApplicationClassInstance(
    ApplicationRegistry registry, Class<? extends IApplication> appClass)
    throws ApplicationException {
    String id = registry.getId();
    try {
      Constructor<?> constructor =
        appClass.getConstructor(ICSDKEnvironment.class);
      Object appInstance = constructor.newInstance(csdkEnv);
      IApplication app = appClass.cast(appInstance);
      return app;
    }
    catch (final InvocationTargetException e) {
      final Throwable appException = e.getTargetException();
      final String message =
        MessageFormat.format(
          LNG.get(CSDKApplication.class.getSimpleName()
            + ".instatiation.exception"), new Object[] { id,
              appClass.getName(), appException.getLocalizedMessage() });
      throw new ApplicationException(message, appException);
    }
    catch (final NoSuchMethodException e) {
      final String message =
        MessageFormat
          .format(
            LNG.get(CSDKApplication.class.getSimpleName()
              + ".no.valid.constructor"),
            new Object[] { id, appClass.getName() });
      throw new ApplicationException(message, e);
    }
    catch (final InstantiationException e) {
      final String message =
        MessageFormat.format(
          LNG.get(CSDKApplication.class.getSimpleName() + ".no.instantiation"),
          new Object[] { id, appClass.getName() });
      throw new ApplicationException(message, e);
    }
    catch (final IllegalAccessException e) {
      final String message =
        MessageFormat.format(
          LNG.get(CSDKApplication.class.getSimpleName()
            + ".application.manager.bad.implementation"), new Object[] { id,
              appClass.getName() });
      throw new ApplicationException(message, e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean userCanKillApplication() {
    return application.canEndApplication();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void sendMessage(String name, Object value, String senderId) {
    IMessage message = new Message(name, value);
    IMessageSender sender = null;
    ApplicationManager manager = ApplicationManager.getInstance();
    ApplicationType senderApp = manager.getApplicationInstance(senderId);
    if (senderApp != null) {
      sender = new MessageSender(senderApp.getId(), senderId);
    }
    application.onMessageReceived(message, sender);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void killApplication() throws ApplicationException {
    try {
      application.onApplicationEnd();
    }
    catch (csdk.v1_0.api.application.ApplicationException e) {
      throw new ApplicationException(e);
    }
    finally {
      ApplicationManager manager = ApplicationManager.getInstance();
      manager.notifyDeath(this);
      this.csdkEnv.cleanup();
      this.csdkEnv = null;
      this.application = null;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startApplication() throws ApplicationException {
    try {
      application.onApplicationStart();
    }
    catch (csdk.v1_0.api.application.ApplicationException e) {
      throw new ApplicationException(e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void finishApplication() {
    // No faz nada
  }

  /**
   * Obtm a aplicao propriamente dita.
   * 
   * @return a aplicao.
   */
  public IApplication getApplication() {
    return application;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Class<?> getMainApplicationClass() {
    return application.getClass();
  }

}
