package csbase.client.remote.srvproxies.messageservice.consumers;

import java.rmi.RemoteException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;

import tecgraf.javautils.core.filter.IFilter;
import csbase.remote.IRemoteMessageListener;
import csbase.remote.MessageServiceInterface;
import csbase.util.messages.Message;
import csbase.util.rmi.PortReference;

/**
 * Implementa uma {@link IMessageConsumer estratgia de recebimento de
 * mensagens} que cadastra um ouvinte no {@link MessageServiceInterface servio
 * de mensagens}.
 * 
 * @author Tecgraf
 */
public class MessageListenerConsumer implements IMessageConsumer {

  /**
   * Intermedirio entre receptores e produtores de mensagens.
   */
  private MessageServiceInterface service;

  /**
   * Referncia para a porta em que ir se exportar o
   * {@link IRemoteMessageListener} utilizado para ouvir o
   * {@link MessageServiceInterface}.
   */
  private PortReference portRef;
  /**
   * Ouvinte que ir receber as mensagens vindas do servidor e redireciona-las
   * para o {@link IMessageConsumer.IListener}.
   */
  private IRemoteMessageListener messageListener;

  /**
   * Ouvinte que ir tratar as mensagens obtidas pelo timer.
   */
  private IListener listener;
  /**
   * Filtro que determina que mensagens sero consumidas do servio de
   * mensagens.
   */
  private IFilter<Message> filter;

  /**
   * <tt>True</tt> indica que o adapter est ouvindo o servio de mensagens.
   */
  private AtomicBoolean started;

  /**
   * Utilizado para:
   * <ul>
   * <li>
   * inserir e remover o {@link IRemoteMessageListener ouvinte de mensagens} do
   * {@link MessageServiceInterface servio de mensagens}</li>
   * <li>
   * repassar ao listener as mensagens recebidas pelo messageListener,
   * retornando rapidamente a chamada do servidor.</li>
   * </ul>
   */
  private ExecutorService executor;

  /**
   * Construtor.
   * 
   * @param service Servio de mensagens.
   * @param portRef porta em que ir se exportar o
   *        {@link IRemoteMessageListener ouvinte de mensagens} utilizado para
   *        ouvir o {@link MessageServiceInterface servio de mensagens}.
   */
  public MessageListenerConsumer(MessageServiceInterface service,
    PortReference portRef) {
    if (service == null) {
      throw new IllegalArgumentException("service == null");
    }
    if (!portRef.isBound()) {
      throw new IllegalArgumentException("no port reserved");
    }

    this.service = service;
    this.portRef = portRef;
    this.started = new AtomicBoolean(false);
    this.executor = Executors.newSingleThreadExecutor();
    this.messageListener = new IRemoteMessageListener() {
      @Override
      public void onMessagesReceived(final Message... messages)
        throws RemoteException {
        /*
         * Repassa para o ouvinte de mensagens a partir de uma nova thread,
         * liberando assim a chamada RMI.
         */
        final IListener receiver = listener;
        Runnable runnable = new Runnable() {
          @Override
          public void run() {
            receiver.onMessagesReceived(messages);
          }
        };
        executor.execute(runnable);
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void setListener(IListener listener,
    IFilter<Message> filter) {
    if (listener == null) {
      throw new IllegalArgumentException("listener == null");
    }
    if (filter == null) {
      throw new IllegalArgumentException("filter == null");
    }

    this.listener = listener;
    this.filter = filter;

    Runnable task = new Runnable() {
      /**
       * {@inheritDoc}
       */
      public void run() {
        try {
          IFilter<Message> filter = MessageListenerConsumer.this.filter;

          // Se o consumidor estiver parado, ...
          if (started.compareAndSet(false, true)) {
            /*
             * comea a ouvir por menagens no {@link MessageServiceInterface
             * servio de mensagens}.
             */
            portRef.exportObject(messageListener);
            service.setMessageListener(messageListener, filter);
          }
          else {
            /*
             * Caso contrrio, reseta o filtro do {@link MessageServiceInterface
             * servio de mensagens}.
             */
            service.clearMessageListener();
            service.setMessageListener(messageListener, filter);
          }
        }
        catch (Exception e) {
          MessageListenerConsumer.this.listener.onExceptionThrown(e);
        }
      }
    };
    executor.execute(task);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void clearListener() {
    Runnable task = new Runnable() {
      /**
       * {@inheritDoc}
       */
      public void run() {
        // Se o consumidor estiver funcionando, ..
        if (started.compareAndSet(true, false)) {
          /*
           * Para de ouvir por mensagens no {@link MessageServiceInterface
           * servio de mensagens}.
           */
          try {
            service.clearMessageListener();
            portRef.unexportObject(messageListener, true);
          }
          catch (Exception e) {
            listener.onExceptionThrown(e);
          }
        }
      }
    };
    executor.execute(task);
  }
}
