package csbase.client.project;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.RowMapper;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import csbase.logic.ClientProjectFile;
import csbase.logic.ProjectFileFilter;
import csbase.util.collections.NoDuplicatesCollection;

/**
 * Define um modelo de seleo de uma rvore de projetos.
 * 
 * @author Rodrigo Carneiro Henrique (rodrigoh)
 */
public final class ProjectTreeSelectionModel implements TreeSelectionModel {
  /**
   * Responsvel por gerenciar a lista de paths selecionados quando o modo de
   * seleo  SINGLE_TREE_SELECTION. Nesse modo apenas um item pode estar
   * selecionado.
   * 
   * @author Rodrigo Carneiro Henrique (rodrigoh)
   */
  private final class SelectionModeStateSingle implements
    SelectionModeStateInterface {
    /**
     * {@inheritDoc}
     */
    public void addSelectionPaths(
      NoDuplicatesCollection<TreePath> treePathCollection) {
      this.setSelectionPaths(treePathCollection);
    }

    /**
     * {@inheritDoc}
     */
    public void setSelectionPaths(
      NoDuplicatesCollection<TreePath> treePathCollection) {
      if (treePathCollection.isEmpty()) {
        return;
      }
      TreePath oldLeadSelectionPath =
        ProjectTreeSelectionModel.this.getLeadSelectionPath();
      ProjectTreeSelectionModel.this.selectedPaths.clear();
      ProjectTreeSelectionModel.this.selectedPaths.addAll(treePathCollection);
      ProjectTreeSelectionModel.this.resetRowSelection();
      TreePath newLeadSelectionPath =
        ProjectTreeSelectionModel.this.getLeadSelectionPath();
      TreePath[] paths;
      boolean[] areNew;
      if (oldLeadSelectionPath != null) {
        if (oldLeadSelectionPath.equals(newLeadSelectionPath)) {
          return;
        }
        else {
          paths = new TreePath[] { oldLeadSelectionPath, newLeadSelectionPath };
          areNew = new boolean[] { false, true };
        }
      }
      else {
        paths = new TreePath[] { newLeadSelectionPath };
        areNew = new boolean[] { true };
      }
      TreeSelectionEvent treeSelectionEvent =
        new TreeSelectionEvent(this, paths, areNew, oldLeadSelectionPath,
          newLeadSelectionPath);
      ProjectTreeSelectionModel.this.fireValueChanged(treeSelectionEvent);
    }

    /**
     * {@inheritDoc}
     */
    public void removeSelectionPaths(
      NoDuplicatesCollection<TreePath> treePathCollection) {
      if (treePathCollection.isEmpty()) {
        return;
      }
      TreePath oldSelectionPath =
        ProjectTreeSelectionModel.this.getLeadSelectionPath();
      TreePath[] oldSelectedPaths =
        ProjectTreeSelectionModel
          .getTreePathArrayFromList(ProjectTreeSelectionModel.this.selectedPaths);
      ProjectTreeSelectionModel.this.selectedPaths
        .removeAll(treePathCollection);
      ProjectTreeSelectionModel.this.resetRowSelection();
      boolean[] areNew = new boolean[oldSelectedPaths.length];
      for (int i = 0; i < areNew.length; i++) {
        areNew[i] = !treePathCollection.contains(oldSelectedPaths[i]);
      }
      TreeSelectionEvent treeSelectionEvent =
        new TreeSelectionEvent(this, oldSelectedPaths, areNew,
          oldSelectionPath, ProjectTreeSelectionModel.this
            .getLeadSelectionPath());
      ProjectTreeSelectionModel.this.fireValueChanged(treeSelectionEvent);
    }
  }

  /**
   * Responsvel por gerenciar a lista de paths selecionados quando o modo de
   * seleo  DISCONTIGUOUS_TREE_SELECTION. Nesse modo mltiplos itens podem
   * ser selecionados, independente de sua posio. Essa seleo normalmente 
   * feita utilizando-se a tecla CTRL.
   * 
   * @author Rodrigo Carneiro Henrique (rodrigoh)
   */
  private final class SelectionModeStateDiscontiguous implements
    SelectionModeStateInterface {
    /**
     * {@inheritDoc}
     */
    public void addSelectionPaths(
      NoDuplicatesCollection<TreePath> treePathCollection) {
      if (treePathCollection.isEmpty()) {
        return;
      }
      TreePath oldLeadSelectionPath =
        ProjectTreeSelectionModel.this.getLeadSelectionPath();
      ProjectTreeSelectionModel.this.selectedPaths.addAll(treePathCollection);
      ProjectTreeSelectionModel.this.resetRowSelection();
      TreePath[] paths =
        ProjectTreeSelectionModel
          .getTreePathArrayFromList(ProjectTreeSelectionModel.this.selectedPaths);
      boolean[] areNew = new boolean[paths.length];
      Arrays.fill(areNew, true);
      TreeSelectionEvent treeSelectionEvent =
        new TreeSelectionEvent(this, paths, areNew, oldLeadSelectionPath,
          ProjectTreeSelectionModel.this.getLeadSelectionPath());
      ProjectTreeSelectionModel.this.fireValueChanged(treeSelectionEvent);
    }

    /**
     * {@inheritDoc}
     */
    public void setSelectionPaths(
      NoDuplicatesCollection<TreePath> treePathCollection) {
      if (treePathCollection.isEmpty()) {
        return;
      }
      TreePath oldLeadSelectionPath =
        ProjectTreeSelectionModel.this.getLeadSelectionPath();
      NoDuplicatesCollection<TreePath> allPaths =
        new NoDuplicatesCollection<TreePath>(
          ProjectTreeSelectionModel.this.selectedPaths);
      allPaths.addAll(treePathCollection);
      ProjectTreeSelectionModel.this.selectedPaths.clear();
      ProjectTreeSelectionModel.this.selectedPaths.addAll(treePathCollection);
      ProjectTreeSelectionModel.this.resetRowSelection();
      TreePath[] paths = getTreePathArrayFromList(allPaths);
      boolean[] areNew = new boolean[paths.length];
      for (int i = 0; i < areNew.length; i++) {
        areNew[i] = treePathCollection.contains(paths[i]);
      }
      TreeSelectionEvent treeSelectionEvent =
        new TreeSelectionEvent(this, paths, areNew, oldLeadSelectionPath,
          ProjectTreeSelectionModel.this.getLeadSelectionPath());
      ProjectTreeSelectionModel.this.fireValueChanged(treeSelectionEvent);
    }

    /**
     * {@inheritDoc}
     */
    public void removeSelectionPaths(
      NoDuplicatesCollection<TreePath> treePathCollection) {
      if (treePathCollection.isEmpty()) {
        return;
      }
      TreePath oldSelectionPath =
        ProjectTreeSelectionModel.this.getLeadSelectionPath();
      TreePath[] oldSelectedPaths =
        ProjectTreeSelectionModel
          .getTreePathArrayFromList(ProjectTreeSelectionModel.this.selectedPaths);
      ProjectTreeSelectionModel.this.selectedPaths
        .removeAll(treePathCollection);
      ProjectTreeSelectionModel.this.resetRowSelection();
      boolean[] areNew = new boolean[oldSelectedPaths.length];
      for (int i = 0; i < areNew.length; i++) {
        areNew[i] = !treePathCollection.contains(oldSelectedPaths[i]);
      }
      TreeSelectionEvent treeSelectionEvent =
        new TreeSelectionEvent(this, oldSelectedPaths, areNew,
          oldSelectionPath, ProjectTreeSelectionModel.this
            .getLeadSelectionPath());
      ProjectTreeSelectionModel.this.fireValueChanged(treeSelectionEvent);
    }
  }

  /**
   * Filtro usado para avaliar se determinado arquivo de projeto pode ser
   * selecionado.
   */
  private ProjectFileFilter filter;

  /** Coleo de ouvintes de eventos de seleo do modelo. */
  private Set<TreeSelectionListener> treeSelectionListenerSet;

  /** Coleo de paths que esto atualmente selecionados. */
  private NoDuplicatesCollection<TreePath> selectedPaths;

  /**
   * Corresponde  linha de cada path que est atualmente selecionado.
   * {@link RowMapper#getRowsForPaths(javax.swing.tree.TreePath[])}
   */
  private int[] selectedRows;
  private RowMapper rowMapper;

  /**  o responsvel por gerenciar a lista de paths selecionados. */
  private SelectionModeStateInterface selectionModeState;
  private final SelectionModeStateSingle SINGLE_TREE_SELECTION_STATE =
    new SelectionModeStateSingle();
  private final SelectionModeStateDiscontiguous DISCONTIGUOUS_TREE_SELECTION_STATE =
    new SelectionModeStateDiscontiguous();

  /** O modo de seleo do modelo. */
  private int selectionMode;

  /**
   * Coleo de ouvintes de eventos de mudana em propriedades do modelo. Esses
   * ouvintes esto interessados na propriedade {@link #selectionMode}.
   */
  private Set<PropertyChangeListener> propertyChangeListenerSet;
  private static final String SELECTION_MODE_PROPERTY = "selectionMode";

  /**
   * Cria um ProjectTreeSelectionModel que aceitar a seleo de qualquer tipo
   * de arquivo de projeto.
   */
  public ProjectTreeSelectionModel() {
    this(null);
  }

  /**
   * Cria um ProjectTreeSelectionModel que permitir a seleo apenas de
   * arquivos aceitos pelo filtro recebido.
   * 
   * @param filter O filtro de seleo.
   */
  public ProjectTreeSelectionModel(ProjectFileFilter filter) {
    this.treeSelectionListenerSet = new HashSet<TreeSelectionListener>();
    this.propertyChangeListenerSet = new HashSet<PropertyChangeListener>();
    this.selectedPaths = new NoDuplicatesCollection<TreePath>();
    this.setSelectionMode(DISCONTIGUOUS_TREE_SELECTION);
    this.setFilter(filter);
  }

  ProjectFileFilter getFilter() {
    return this.filter;
  }

  /**
   * Atualiza o filtro de seleo de arquivos de projeto.
   * 
   * @param filter O novo filtro de seleo.
   */
  void setFilter(ProjectFileFilter filter) {
    this.filter = filter;
    this.updateSelection();
  }

  private void updateSelection() {
    NoDuplicatesCollection<TreePath> newSelection =
      this.filterPaths(getTreePathArrayFromList(this.selectedPaths));
    if (newSelection.isEmpty()) {
      this.selectionModeState.removeSelectionPaths(this.selectedPaths.clone());
    }
    else {
      this.selectionModeState.setSelectionPaths(newSelection);
    }
    this.resetRowSelection();
  }

  /**
   * Obtm o ndice do ltimo path adicionado na lista.
   * 
   * @return O ndice do ltimo path adicionado na lista.
   */
  private int getLeadSelectionIndex() {
    return (this.selectedPaths.size() - 1);
  }

  /**
   * {@inheritDoc}
   */
  public int getLeadSelectionRow() {
    if (this.rowMapper == null) {
      return -1;
    }
    return this.selectedRows[this.getLeadSelectionIndex()];
  }

  /**
   * {@inheritDoc}
   */
  public int getMaxSelectionRow() {
    if (this.rowMapper == null) {
      return -1;
    }
    if (this.selectedRows.length == 0) {
      return -1;
    }
    int max = this.selectedRows[0];
    for (int i = 1; i < this.selectedRows.length; i++) {
      max = Math.max(max, this.selectedRows[i]);
    }
    return max;
  }

  /**
   * {@inheritDoc}
   */
  public int getMinSelectionRow() {
    if (this.rowMapper == null) {
      return -1;
    }
    if (this.selectedRows.length == 0) {
      return -1;
    }
    int min = this.selectedRows[0];
    for (int i = 1; i < this.selectedRows.length; i++) {
      min = Math.min(min, this.selectedRows[i]);
    }
    return min;
  }

  /**
   * {@inheritDoc}
   */
  public int getSelectionCount() {
    return this.selectedPaths.size();
  }

  /**
   * {@inheritDoc}
   */
  public int getSelectionMode() {
    return this.selectionMode;
  }

  /**
   * {@inheritDoc}
   */
  public void clearSelection() {
    TreePath[] treePathsSelected = getTreePathArrayFromList(this.selectedPaths);
    if (treePathsSelected.length == 0) {
      return;
    }
    boolean[] areNew = new boolean[treePathsSelected.length];
    for (int i = 0; i < areNew.length; i++) {
      areNew[i] = false;
    }
    TreeSelectionEvent treeSelectionEvent =
      new TreeSelectionEvent(this, treePathsSelected, areNew, this
        .getLeadSelectionPath(), null);
    this.fireValueChanged(treeSelectionEvent);
    this.selectedPaths.clear();
  }

  /**
   * Envia o evento de mudana na seleo para todos os ouvintes cadastrados.
   * 
   * @param treeSelectionEvent O evento que ser enviado aos ouvintes.
   */
  private void fireValueChanged(TreeSelectionEvent treeSelectionEvent) {
    for (TreeSelectionListener listener : treeSelectionListenerSet) {
      listener.valueChanged(treeSelectionEvent);
    }
  }

  /**
   * {@inheritDoc}
   */
  public void resetRowSelection() {
    if (this.rowMapper != null) {
      this.selectedRows =
        this.rowMapper
          .getRowsForPaths(getTreePathArrayFromList(this.selectedPaths));
    }
    else {
      this.selectedRows = null;
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean isSelectionEmpty() {
    return this.selectedPaths.isEmpty();
  }

  /**
   * {@inheritDoc}
   */
  public int[] getSelectionRows() {
    return this.selectedRows;
  }

  /**
   * {@inheritDoc}
   */
  public void setSelectionMode(int mode) {
    if (mode == this.selectionMode) {
      return;
    }
    switch (mode) {
      case TreeSelectionModel.SINGLE_TREE_SELECTION:
        this.selectionModeState = SINGLE_TREE_SELECTION_STATE;
        break;
      case TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION:
        this.selectionModeState = DISCONTIGUOUS_TREE_SELECTION_STATE;
        break;
      default:
        throw new IllegalArgumentException("Modo de seleo invlido. Use: "
          + TreeSelectionModel.SINGLE_TREE_SELECTION + " ou "
          + TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION + ".");
    }
    int oldMode = this.selectionMode;
    this.selectionMode = mode;
    PropertyChangeEvent event =
      new PropertyChangeEvent(this, SELECTION_MODE_PROPERTY, new Integer(
        oldMode), new Integer(mode));
    this.firePropertyChange(event);
    this.updateSelection();
  }

  /**
   * Envia o evento de mudana no valor da propriedade de nome
   * {@link #SELECTION_MODE_PROPERTY} para todos os ouvintes cadastrados.
   * 
   * @param event O evento que ser enviado aos ouvintes.
   */
  private void firePropertyChange(PropertyChangeEvent event) {
    for (PropertyChangeListener listener : propertyChangeListenerSet) {
      listener.propertyChange(event);
    }
  }

  /**
   * {@inheritDoc}
   */
  public boolean isRowSelected(int row) {
    if (this.selectedRows == null) {
      return false;
    }
    for (int i = 0; i < this.selectedRows.length; i++) {
      if (this.selectedRows[i] == row) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  public void addPropertyChangeListener(PropertyChangeListener listener) {
    this.propertyChangeListenerSet.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  public void removePropertyChangeListener(PropertyChangeListener listener) {
    this.propertyChangeListenerSet.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  public void addTreeSelectionListener(TreeSelectionListener listener) {
    this.treeSelectionListenerSet.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  public void removeTreeSelectionListener(TreeSelectionListener listener) {
    this.treeSelectionListenerSet.remove(listener);
  }

  /**
   * {@inheritDoc}
   */
  public RowMapper getRowMapper() {
    return this.rowMapper;
  }

  /**
   * {@inheritDoc}
   */
  public void setRowMapper(RowMapper newMapper) {
    this.rowMapper = newMapper;
    this.resetRowSelection();
  }

  /**
   * {@inheritDoc}
   */
  public TreePath getLeadSelectionPath() {
    if (this.selectedPaths.isEmpty()) {
      return null;
    }
    return (TreePath) this.selectedPaths.getLast();
  }

  /**
   * {@inheritDoc}
   */
  public TreePath getSelectionPath() {
    if (this.selectedPaths.isEmpty()) {
      return null;
    }
    return (TreePath) this.selectedPaths.getFirst();
  }

  /**
   * {@inheritDoc}
   */
  public TreePath[] getSelectionPaths() {
    return getTreePathArrayFromList(this.selectedPaths);
  }

  /**
   * {@inheritDoc}
   */
  public void addSelectionPath(TreePath path) {
    if (path == null) {
      return;
    }
    TreePath[] paths = new TreePath[] { path };
    this.addSelectionPaths(paths);
  }

  /**
   * {@inheritDoc}
   */
  public void removeSelectionPath(TreePath path) {
    if (path == null) {
      return;
    }
    TreePath[] paths = new TreePath[] { path };
    this.removeSelectionPaths(paths);
  }

  /**
   * {@inheritDoc}
   */
  public void setSelectionPath(TreePath path) {
    if (path == null) {
      return;
    }
    TreePath[] paths = new TreePath[] { path };
    this.setSelectionPaths(paths);
  }

  /**
   * {@inheritDoc}
   */
  public boolean isPathSelected(TreePath path) {
    return this.selectedPaths.contains(path);
  }

  /**
   * {@inheritDoc}
   */
  public void addSelectionPaths(TreePath[] paths) {
    this.selectionModeState.addSelectionPaths(this.filterPaths(paths));
  }

  /**
   * {@inheritDoc}
   */
  public void removeSelectionPaths(TreePath[] paths) {
    this.selectionModeState.removeSelectionPaths(this.filterPaths(paths));
  }

  /**
   * {@inheritDoc}
   */
  public void setSelectionPaths(TreePath[] paths) {
    this.selectionModeState.setSelectionPaths(this.filterPaths(paths));
  }

  /**
   * Filtra os paths recebidos para que sejam selecionados apenas arquivos
   * aceitos pelo filtro do modelo. Alm disso, remove elementos duplicados.
   * 
   * @param paths Os paths a serem enxugados. Se paths for null, retorna uma
   *        lista vazia.
   * 
   * @return Uma lista com os elementos aceitos pelo filtro, sem elementos
   *         duplicados. Obs: Nunca retorna null, apenas lista vazia.
   */
  private NoDuplicatesCollection<TreePath> filterPaths(TreePath[] paths) {
    NoDuplicatesCollection<TreePath> filteredPathsList =
      new NoDuplicatesCollection<TreePath>();
    if (paths != null) {
      for (TreePath path : paths) {
        if (path != null) {
          ProjectTreeNode treeNode =
            (ProjectTreeNode) path.getLastPathComponent();
          ClientProjectFile projectFile = treeNode.getClientProjectFile();
          if (filter == null || filter.accept(projectFile)) {
            filteredPathsList.add(path);
          }
        }
      }
    }
    return filteredPathsList;
  }

  /**
   * Criar um array de paths a partir de uma lista de elementos no-duplicados.
   * 
   * @param list A lista com os paths que comporo o array gerado.
   * 
   * @return O array de elementos no-duplicados.
   */
  private static TreePath[] getTreePathArrayFromList(
    NoDuplicatesCollection<TreePath> list) {
    return list.toArray(new TreePath[list.size()]);
  }
}
