package csbase.client.algorithms.parameters;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.text.MessageFormat;

import csbase.client.algorithms.view.simple.SimpleAlgorithmConfiguratorPanel;
import csbase.exception.OperationFailureException;
import csbase.logic.algorithms.parameters.FileParameter;
import csbase.logic.algorithms.parameters.FileParameterPipeAcceptance;
import csbase.logic.algorithms.parameters.Parameter;
import csbase.logic.algorithms.parameters.ParameterGroup;

/**
 * Fbrica de vises dos parmetros dos algoritmos.
 */
public class DefaultParameterViewFactory implements ParameterViewFactory {

  /**
   * {@inheritDoc}
   */
  @Override
  public ParameterView<?> create(
    SimpleAlgorithmConfiguratorPanel configurationPanel,
    Parameter<?> parameter, ParameterView.Mode mode) {
    try {
      return createCustomParameterView(parameter, mode, "logic", "client");
    }
    catch (Exception e) {
      if (parameter instanceof ParameterGroup) {
        ParameterGroup parameterGroup = (ParameterGroup) parameter;
        return new ParameterGroupView(configurationPanel, parameterGroup, mode);
      }
      if (parameter instanceof FileParameter) {
        FileParameter fileParameter = (FileParameter) parameter;
        if (fileParameter.hasLink()
          || fileParameter.usesPipe() == FileParameterPipeAcceptance.ALWAYS) {
          return null;
        }
      }
      throw new IllegalArgumentException(
        "No foi possvel criar a viso do parmetro: "
          + parameter.getClass().getName() + ". Erro: " + e + " .", e
          .getCause());
    }
  }

  /**
   * Cria a viso de um parmetro por reflexo. Para ser criada corretamente, a
   * viso precisa estender a classe {@link ParameterView} e deve estar no
   * pacote cliente anlogo ao pacote da lgica que contm o parmetro. Alm
   * disso, o nome da classe da viso deve ser o mesmo da classe da lgica, com
   * sufixo "View".
   * 
   * @param parameter Parmetro.
   * @param mode Modo da viso.
   * @param logicPackageName Parte do nome do pacote do parmetro que indica que
   *         uma classe de lgica. Exemplo: Se pacote  x.logic.y, esse
   *        parmetro deve ter como valor "logic".
   * @param clientPackageName Parte do nome do pacote da viso do parmetro que
   *        indica que  uma classe de viso. Exemplo: Se pacote  x.client.y,
   *        esse parmetro deve ter como valor "client".
   * @return A viso do parmetro
   * @throws OperationFailureException Caso no seja possvel criar a viso do
   *         parmetro.
   */
  protected ParameterView<?> createCustomParameterView(Parameter<?> parameter,
    ParameterView.Mode mode, String logicPackageName, String clientPackageName)
    throws OperationFailureException {

    String parameterClassName = parameter.getClass().getName();
    String parameterViewClassName = parameterClassName + "View";

    if (!parameterViewClassName.contains("." + logicPackageName + ".")) {
      throw new OperationFailureException(
        "A classe {0} precisa estar no pacote {1}", parameterViewClassName,
        logicPackageName);
    }

    String replacedParameterViewClassName =
      parameterViewClassName.replaceFirst("\\." + logicPackageName + "\\.", "."
        + clientPackageName + ".");

    Class<?> viewClass;
    try {
      viewClass = Class.forName(replacedParameterViewClassName);
    }
    catch (ClassNotFoundException e) {
      throw new OperationFailureException("A classe {0} no foi encontrada",
        replacedParameterViewClassName);
    }
    if (!ParameterView.class.isAssignableFrom(viewClass)) {
      throw new OperationFailureException(
        "A classe {0} no implementa a interface necessria {1}",
        replacedParameterViewClassName, ParameterView.class.getName());
    }
    Constructor<?> viewConstructor;
    try {
      viewConstructor =
        viewClass
          .getConstructor(parameter.getClass(), ParameterView.Mode.class);
    }
    catch (NoSuchMethodException e) {
      throw new OperationFailureException(
        "No existe um construtor pblico com parmetros dos tipos {0} e {1} em {2}",
        parameter.getClass().getSimpleName(), ParameterView.Mode.class
          .getSimpleName(), replacedParameterViewClassName);
    }
    ParameterView<?> view;
    try {
      view = (ParameterView<?>) viewConstructor.newInstance(parameter, mode);
    }
    catch (InstantiationException e) {
      throw new OperationFailureException(MessageFormat.format(
        "No foi possvel construir a fbrica {0}",
        replacedParameterViewClassName), e);
    }
    catch (IllegalAccessException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Construtor da fbrica {0} no aceita os parmetros usados",
        replacedParameterViewClassName), e);
    }
    catch (InvocationTargetException e) {
      throw new OperationFailureException(MessageFormat.format(
        "Erro ao criar a fbrica de parmetros {0}",
        replacedParameterViewClassName), e.getTargetException());
    }
    return view;
  }
}
