/**
 * $Id: DefaultSwingRenderer.java 141063 2013-05-16 15:46:28Z fpina $
 */

package csbase.client.applications.serverdiagnostic.monitor;

import java.awt.Color;
import java.awt.Component;
import java.awt.GridBagLayout;
import java.text.SimpleDateFormat;
import java.util.List;

import javax.swing.DefaultListModel;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.table.DefaultTableCellRenderer;

import tecgraf.javautils.gui.GBC;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.serverdiagnostic.ServerDiagnostic;
import csbase.logic.diagnosticservice.ResourceProperty;
import csbase.logic.diagnosticservice.Status;
import csbase.logic.diagnosticservice.StatusCode;
import csbase.logic.diagnosticservice.ValidationStatus;

/**
 * Renderer padro de um estado que formata as informaes em componentes Swing
 * 
 * @author Tecgraf PUC-Rio
 */
public class DefaultSwingRenderer implements SwingRenderer {
  /**
   * Formato de data.
   */
  private SimpleDateFormat dateFormat = new SimpleDateFormat("dd/MM/yyyy "
    + "HH:mm:ss");

  /**
   * {@inheritDoc}
   */
  @Override
  public JComponent getHeadline(Status status) {
    JLabel headline = new JLabel(status.getResourceName());
    headline.setIcon(getIcon(status.getCode()));
    headline.setToolTipText("[" + dateFormat.format(status.getTimestamp())
      + "] " + status);

    return headline;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JComponent getDetails(Status status) {
    JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    JLabel paramTableLabel =
      new JLabel(getString("DefaultSwingRenderer.parameters.label"));
    panel.add(paramTableLabel,
      new GBC(0, 0).horizontal().northwest().insets(0, 0, 5, 0));
    panel.add(new JScrollPane((getParamTable(status))), new GBC(0, 1).both()
      .northwest().insets(0, 0, 5, 0));
    JLabel validationStatusListLabel =
      new JLabel(getString("DefaultSwingRenderer.validations.label"));
    paramTableLabel.setHorizontalTextPosition(SwingConstants.LEFT);
    panel.add(validationStatusListLabel, new GBC(0, 2).none().northwest()
      .insets(5, 0, 5, 0));
    panel.add(new JScrollPane(getValidationStatusList(status)), new GBC(0, 3)
      .both().northwest().insets(0, 0, 5, 0));
    return panel;
  }

  /**
   * Obtm a lista de estados das validaes realizadas para avaliar o recurso
   * monitorado.
   * 
   * @param status o estado
   * 
   * @return a lista de estados da validaes
   */
  protected JComponent getValidationStatusList(Status status) {
    final DefaultListModel validationStatusListModel = new DefaultListModel();
    for (ValidationStatus validationStatus : status.getValidationStatusList()) {
      validationStatusListModel.addElement(validationStatus);
    }
    final JList validationStatusList = new JList(validationStatusListModel);
    validationStatusList
      .setCellRenderer(new ValidationStatusListCellRenderer());
    return validationStatusList;
  }

  /**
   * Obtm o cone correspondente a um
   * {@link csbase.logic.diagnosticservice.StatusCode}.
   * 
   * @param code o {@link csbase.logic.diagnosticservice.StatusCode}
   * 
   * @return o cone correspondente
   */
  protected Icon getIcon(StatusCode code) {
    switch (code) {
      case OK:
        return ApplicationImages.GREEN_BALL_16;
      case WARNING:
        return ApplicationImages.YELLOW_BALL_16;
      case ERROR:
      case UNAVAILABLE:
      case INITIATING:
        return ApplicationImages.RED_BALL_16;
      default:
        return null;
    }
  }

  /**
   * Obtm uma tabela com os parmetros do monitor usados para monitorar o
   * recurso.
   * 
   * @param status o estado
   * 
   * @return a tabela com os parmetros
   */
  private JComponent getParamTable(Status status) {
    Object[][] data = getParameters(status);
    JTable paramTable =
      new JTable(data, new String[] {
          getString("DefaultSwingRenderer.name.label"),
          getString("DefaultSwingRenderer.value.label"),
          getString("DefaultSwingRenderer.source.label") });
    paramTable.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
    paramTable.setEnabled(false);
    paramTable.getColumnModel().getColumn(1)
      .setCellRenderer(new DefaultTableCellRenderer() {
        @Override
        public Component getTableCellRendererComponent(JTable table,
          Object value, boolean isSelected, boolean hasFocus, int row,
          int column) {
          final JTextArea jtext = new JTextArea();
          jtext.setText((String) value);
          jtext.setWrapStyleWord(true);
          jtext.setLineWrap(true);
          if (isSelected) {
            jtext.setBackground((Color) UIManager
              .get("Table.selectionBackground"));
          }

          int lines = Utils.countLines((String) value);
          if (lines > 1) {
            int defaultHeight = getPreferredSize().height * 8;
            table.setRowHeight(row, defaultHeight * lines);
          }
          return jtext;
        }
      });

    return paramTable;
  }

  /**
   * Obtm os parmetros de um estado com um array de array
   * 
   * @param status o estado
   * 
   * @return os parmetros do estado
   */
  private Object[][] getParameters(Status status) {
    List<ResourceProperty> props = status.getProperties();
    if (props == null) {
      return new Object[0][0];
    }
    Object[][] data = new Object[props.size()][];
    int i = 0;
    for (ResourceProperty p : props) {
      data[i++] = new Object[] { p.name, p.value, p.source };
    }
    return data;
  }

  /**
   * Obtm a mensagem para uma dada chave.
   * 
   * @param key a chave.
   * 
   * @return a mensagem.
   */
  protected String getString(String key) {
    return ApplicationManager.getInstance()
      .getApplicationRegistry(ServerDiagnostic.ID).getString(key);
  }

}
