package csbase.client.ias;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.swing.AbstractListModel;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.logic.AttributesPermission;
import csbase.logic.Permission;

/**
 * Constri o grupo de componentes de atributos, somente visveis quando uma
 * classe filha da classe genrica de atributos estiver selecionada.
 */
class AttributesPanel extends AbstractPermissionPanel {

  /** Label para entrada de novos atributos (opcional) */
  private JLabel newAttributeL;

  /** Campo para entrada de novos atributos (opcional) */
  private JTextField newAttribute;

  /** Modelo para a lista de atributos (opcional) */
  private SortedListModel attribModel;

  /** Componente para scroll da lista de atributos (opcional) */
  private JScrollPane scrollList;

  /** Boto de incluso de nova permisso */
  private JButton includeButton;

  /** Boto de remoo de permisso */
  private JButton removeButton;

  /**
   * Construtor.
   * 
   * @param dialog Dilogo no qual este painel est contido.
   */
  AttributesPanel(PermissionInfoDialog dialog) {
    super(new GridBagLayout(), dialog);

    newAttributeL = new JLabel(LNG.get("IAS_PERMISSION_NEW_ATTRIBUTE") + ":");
    JLabel attributesL = new JLabel(LNG.get("IAS_PERMISSION_ATTRIBUTES") + ":");
    newAttribute = new JTextField(20);
    attribModel = new SortedListModel();
    final JList attributesList = new JList(attribModel);
    scrollList = new JScrollPane(attributesList);
    includeButton = new JButton(LNG.get("IAS_ADD"));
    includeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        String text = newAttribute.getText().trim();
        if (!text.equals("")) {
          attribModel.add(text);
        }
        newAttribute.setText("");
      }
    });
    removeButton = new JButton(LNG.get("IAS_REMOVE"));
    removeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int[] indexes = attributesList.getSelectedIndices();
        if ((indexes != null) && (indexes.length != 0)) {
          for (int i = indexes.length - 1; i >= 0; i--) {
            attribModel.remove(indexes[i]);
          }
        }
      }
    });

    // Label atributos
    GBC gbc = new GBC(0, 0).northwest().insets(5, 10, 10, 5);
    add(newAttributeL, gbc);

    // Campo atributo
    gbc = new GBC(1, 0).horizontal().northwest().insets(0, 0, 5, 0).fillxy();
    add(newAttribute, gbc);

    // Boto incluir
    gbc = new GBC(2, 0).northwest().insets(0, 5, 5, 10).fillx();
    add(includeButton, gbc);

    // Label lista de atributos
    gbc = new GBC(0, 1).northwest().insets(0, 10, 10, 5);
    add(attributesL, gbc);

    // JTextArea lista de atributos
    gbc =
      new GBC(1, 1).horizontal().northwest().insets(0, 0, 5, 0).fillxy().both();
    add(scrollList, gbc);

    // Boto remover
    gbc = new GBC(2, 1).northwest().insets(0, 5, 5, 10);
    add(removeButton, gbc);

  }

  /**
   * {@inheritDoc}
   */
  @Override
  void importFrom(Permission permission) {
    AttributesPermission attribPermission = (AttributesPermission) permission;
    String[] attributes = attribPermission.getAttributes();
    for (int i = 0; i < attributes.length; i++) {
      attribModel.add(attributes[i]);
    }
    attribModel.addListDataListener(new ListDataListener() {
      @Override
      public void intervalAdded(ListDataEvent e) {
        dialog.checkDataChange();
      }

      @Override
      public void intervalRemoved(ListDataEvent e) {
        dialog.checkDataChange();
      }

      @Override
      public void contentsChanged(ListDataEvent e) {
      }
    });

  }

  /**
   * {@inheritDoc}
   */
  @Override
  void exportTo(Permission permission) {
    AttributesPermission attribPermission = (AttributesPermission) permission;
    attribPermission.setAttributes(getAttributes());
  }

  /**
   * Classe que representa um modelo em ordem alfabtica para os atributos de
   * uma permisso.
   */
  private class SortedListModel extends AbstractListModel {
    /** Itens do modelo */
    List<String> items;

    /**
     * Instancia a lista de itens.
     */
    public SortedListModel() {
      items = new ArrayList<String>();
    }

    /**
     * Obtm o tamanho da lista de itens.
     * 
     * @return tamanho da lista
     */
    @Override
    public int getSize() {
      return items.size();
    }

    /**
     * Obtm o item especificado da lista.
     * 
     * @param index ndice do item desejado
     * 
     * @return item indicado pelo ndice
     */
    @Override
    public Object getElementAt(int index) {
      return items.get(index);
    }

    /**
     * Obtm array contendo todos os itens da lista.
     * 
     * @return array contendo todos os itens da lista.
     */
    public Object[] getAll() {
      return items.toArray();
    }

    /**
     * Adiciona um item  lista, procurando o ponto exato de insero para
     * manter a ordem alfabtica. Permite a insero de duplicatas.
     * 
     * @param item item a ser inserido
     */
    public void add(String item) {
      if (item == null) {
        return;
      }
      int index = Collections.binarySearch(items, item);
      if (index < 0) {
        index = -index - 1; // Ajuste para ponto de insero
        // ver javadoc de Collections.binarySearch
      }
      items.add(index, item);
      fireIntervalAdded(this, index, index);
    }

    /**
     * Remove o item especificado da lista.
     * 
     * @param index do item a ser removido
     * 
     * @return item recm-removido
     */
    public String remove(int index) {
      if (index == -1) {
        return null;
      }
      if (items.size() == 0) {
        return null;
      }
      if ((index < 0) || (index > (items.size() - 1))) {
        return null;
      }
      String item = items.remove(index);
      fireIntervalRemoved(this, index, index);
      return item;
    }

    /**
     * Limpa a lista.
     */
    public void clear() {
      items.clear();
      fireIntervalRemoved(this, 0, 0);
    }
  }

  /**
   * @return Array com os atributos inseridos.
   */
  String[] getAttributes() {
    Object[] attributesObj = attribModel.getAll();
    String[] screenAttributes = new String[attributesObj.length];
    System.arraycopy(attributesObj, 0, screenAttributes, 0,
      attributesObj.length);
    return screenAttributes;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  void clearFields() {
    newAttribute.setText("");
    attribModel.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  boolean hasChanged(Permission permission) {
    AttributesPermission attribPermission = (AttributesPermission) permission;
    String[] screenAttributes = getAttributes();
    String[] currentAttributes = attribPermission.getAttributes();
    Arrays.sort(currentAttributes);
    if (!Arrays.equals(screenAttributes, currentAttributes)) {
      return true;
    }
    return false;
  }

}
