package csbase.client.project.tasks;

import java.awt.Window;
import java.rmi.RemoteException;

import csbase.exception.CSBaseRuntimeException;
import csbase.logic.CommonClientProject;
import csbase.remote.ClientRemoteLocator;

/**
 * A classe <code>CreateDirectoryPathTask</code> implementa uma tarefa que cria
 * um caminho de diretrios em um projeto.
 * 
 * O tratamento de RemoteException  delegado  superclasse. O resultado da
 * execuo pode ser verificado atravs do mtodo {@link #getErrorCode()}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class CreateDirectoryPathTask extends CoreTask<Void> {
  /**
   * Cdigos de erro.
   */
  public static enum ErrorCode {
    /** Tarefa no executada */
    NOT_EXECUTED,
    /** Execuo sem erro */
    NO_ERROR,
    /** Exceo unchecked */
    RUNTIME_EXCEPTION,
    /** Falha de RMI */
    RMI_FAILURE,
    /** Erro desconhecido */
    UNKNOWN,
  }

  /**
   * Projeto do cliente.
   */
  final private CommonClientProject project;

  /**
   * Caminho de diretrios a ser criado no projeto
   */
  final private String[] dirPath;

  /**
   * Cdigo de erro associado  execuo
   */
  private ErrorCode errorCode = ErrorCode.NOT_EXECUTED;

  /**
   * Construtor.
   * 
   * @param project projeto do cliente
   * @param dirPath caminho de diretrios a ser criado no projeto
   */
  public CreateDirectoryPathTask(final CommonClientProject project,
    final String[] dirPath) {
    this.project = project;
    this.dirPath = dirPath;
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.desktop.Task#performTask()
   */
  @Override
  protected void performTask() throws Exception {
    if (ClientRemoteLocator.projectService.existsFile(project.getId(), dirPath)) {
      return;
    }
    // Cria o caminho de diretrios caso um deles ainda no exista
    project.createAndWaitForDirectoryPath(project.getRoot(), dirPath);
    errorCode = ErrorCode.NO_ERROR;
  }

  /**
   * (non-Javadoc)
   * 
   * @see csbase.client.desktop.RemoteTask#handleError(java.lang.Exception)
   */
  @Override
  protected void handleError(Exception error) {
    if (error instanceof RemoteException) {
      super.handleError(error);
      errorCode = ErrorCode.RMI_FAILURE;
    }
    else if (error instanceof CSBaseRuntimeException) {
      super.handleError(error);
      errorCode = ErrorCode.RUNTIME_EXCEPTION;
    }
    else {
      errorCode = ErrorCode.UNKNOWN;
    }
    setResult(null);
  }

  /**
   * Retorna o cdigo de erro associado  execuo.
   * 
   * @return cdigo de erro associado  execuo
   */
  public ErrorCode getErrorCode() {
    return errorCode;
  }

  /**
   * Mtodo de convenincia para criao e execuo da tarefa com base na janela
   * de desktop e com texto ttulo padronizado.
   * 
   * @param window janela-me da task.
   * @param project projeto do cliente
   * @param dirPath caminho de diretrios a ser criado no projeto
   * @return retorna true, se a tarefa foi executada com sucesso, caso
   *         contrrio, retorna false
   */
  static final public boolean runTask(Window window,
    final CommonClientProject project, final String[] dirPath) {
    final CreateDirectoryPathTask task =
      new CreateDirectoryPathTask(project, dirPath);
    return task.execute(window, new String[] { project.getName() });
  }
}