/**
 * $Id: CommonClientProject.java 152540 2014-05-16 17:54:33Z analodi $
 */

package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;

import csbase.logic.ProjectPermissions.SharingType;
import csbase.remote.ClientRemoteLocator;

/**
 * Modela a viso cliente de um projeto.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class CommonClientProject extends Observable implements Serializable {

  /**
   * Identificador do usurio.
   */
  protected Object userId;

  /**
   * Nome do projeto.
   */
  protected String name;

  /**
   * Identificador do projeto.
   */
  protected Object projectId;

  /**
   * Informaes do projeto.
   */
  protected CommonProjectInfo info;

  /**
   * rvore do projeto.
   */
  protected ClientProjectFile tree;

  /**
   * Caminho de onde esse projeto est instalado, a partir da raiz de todos os
   * projetos.
   */
  protected String[] path;

  /**
   * Observador desse projeto no servidor.
   */
  protected ProjectObserver observer;

  /**
   * Informa se o servidor tem a capacidade de escrever no projeto. O Valor 
   * true se o servidor for o criador do projeto.
   */
  protected boolean serverCanWriteProject;

  /**
   * Define a capacidade do servidor de escrever no projeto.
   * 
   * @param serverCanWriteProject indicador da capacidade do servidor de
   *        escrever no projeto.
   */
  public void setServerCanWriteProject(boolean serverCanWriteProject) {
    this.serverCanWriteProject = serverCanWriteProject;
  }

  /**
   * Retorna o identificador do projeto.
   * 
   * @return O identificador do projeto.
   */
  public Object getId() {
    return projectId;
  }

  /**
   * Obtm um conjunto com todos os usurios que possuem acesso ao projeto. O
   * conjunto retornado no inclui o dono do projeto e nem o admin, que tem
   * acesso irrestrito a todos os projetos.
   * 
   * @return Conjunto com todos os usurios com os quais o projeto est
   *         compartilhado, seja RO ou RW.
   */
  public Set<Object> getUserIdsSet() {
    return ProjectPermissions.getAllUsers(info);
  }

  /**
   * Obtm o conjunto de usurios que tm acesso RO ao projeto no modo de
   * compartilhamento seletivo.
   * <p>
   * <b>ATENO:</b>
   * <ul>
   * <li>retorna o prprio conjunto, portanto alteraes devem ser feitas em uma
   * cpia
   * <li>o conjunto (vazio ou no) ser retornado mesmo que o tipo de
   * compartilhamento seja privado ou pblico.
   * </ul>
   * 
   * @return conjunto de usurios que tm acesso RO ao projeto
   */
  public Set<Object> getUsersRO() {
    return ProjectPermissions.getUsersRO(info);
  }

  /**
   * Obtm o conjunto de usurios que tm acesso RW ao projeto no modo de
   * compartilhamento seletivo.
   * <p>
   * <b>ATENO:</b>
   * <ul>
   * <li>retorna o prprio conjunto, portanto alteraes devem ser feitas em uma
   * cpia
   * <li>o conjunto (vazio ou no) ser retornado mesmo que o tipo de
   * compartilhamento seja privado ou pblico.
   * </ul>
   * 
   * @return conjunto de usurios que tm acesso RW ao projeto
   */
  public Set<Object> getUsersRW() {
    return ProjectPermissions.getUsersRW(info);
  }

  /**
   * Obtm o tipo de compartilhamento do projeto.
   * 
   * @return tipo de compartilhamento do projeto
   */
  public SharingType getSharingType() {
    return ProjectPermissions.getSharingType(info);
  }

  /**
   * Define o tipo de compartilhamento do projeto.
   * 
   * @param sharingType - tipo de compartilhamento do projeto
   */
  public void setSharingType(SharingType sharingType) {
    ProjectPermissions.setSharingType(info, sharingType);
  }

  /**
   * Verifica se um projeto  pblico.
   * 
   * @return true, caso o projeto seja pblico, ou false, caso contrrio.
   */
  public boolean isPublic() {
    return ProjectPermissions.isPublic(info);
  }

  /**
   * Retorna o nome do servidor que criou o projeto.
   * 
   * @return nome do servidor
   */
  public String getOwnerServerName() {
    String serverName =
      (String) info
        .getAttribute(ProjectAttribute.SERVER_NAME.getAttributeKey());
    if (serverName == null) {
      return "";
    }
    return serverName;
  }

  /**
   * Retorna a data de criao do projeto.
   * 
   * @return a data
   */
  public long getCreationDate() {
    Long date =
      (Long) info
        .getAttribute(ProjectAttribute.CREATION_DATE.getAttributeKey());
    if (date == null) {
      return 0;
    }
    return date.longValue();
  }

  /**
   * Retorna a data de ltima modificao do projeto.
   * 
   * @return a data
   */
  public long getLastModificationDate() {
    return tree.getModificationDate();
  }

  /**
   * Verifica se o projeto  privativo.
   * 
   * @return true, caso o projeto seja privativo, ou false, caso contrrio.
   */
  public boolean isPrivate() {
    return ProjectPermissions.isPrivate(info);
  }

  /**
   * Verifica se o projeto est sendo compartilhado.
   * 
   * @return true, caso o projeto seja compartilhado, ou false, caso contrrio.
   */
  public boolean isShared() {
    return ProjectPermissions.isShared(info);
  }

  /**
   * Verifica se o usurio tem qualquer tipo de acesso RW ao projeto, ou seja,
   * se uma das seguintes condies  satisfeita:
   * <ul>
   * <li>usurio  o admin ou o dono do projeto
   * <li>o projeto  pblico para leitura e escrita
   * <li>o compartilhamento  seletivo e o usurio possui acesso RW
   * </ul>
   * 
   * @param userID - identificador do usurio
   * 
   * @return true se o usurio tem acesso RW ao projeto
   */
  public boolean userHasAccessRW(Object userID) {
    return ProjectPermissions.userHasAccessRW(info, userID)
      && isWritableFromServer();
  }

  /**
   * Verifica se o usurio tem qualquer tipo de acesso RO ao projeto, ou seja,
   * se uma das seguintes condies  satisfeita:
   * <ul>
   * <li>o projeto  pblico apenas para leitura
   * <li>o compartilhamento  seletivo e o usurio possui acesso RO
   * </ul>
   * 
   * @param userID - identificador do usurio
   * 
   * @return true se o usurio tem acesso RO ao projeto
   */
  public boolean userHasAccessRO(Object userID) {
    return ProjectPermissions.userHasAccessRO(info, userID)
      && !isWritableFromServer();
  }

  /**
   * <p>
   * Verifica se quem chamou este mtodo tem permisso de escrita neste projeto.
   * </p>
   * 
   * @return <code>true</code> se quem chamou este mtodo tem permisso de
   *         escrita neste projeto e se o servidor possuir permisso para
   *         escrever no projeto pois foi o seu criador.
   * 
   * @see #userHasAccessRW(Object)
   */
  public boolean isWritable() {
    User user = User.getLoggedUser();
    return user != null && userHasAccessRW(user.getId())
      && isWritableFromServer();
  }

  /**
   * Verifica se o projeto pode ser alterado pelo determinado servidor. Quando o
   * projeto  criado, o servidor salva nas configuraes do projeto o seu
   * identificador indicando ao projeto qual  o seu servidor dono. Quando o
   * cliente tenta escrever no projeto, verifica esse parmetro.
   * 
   * @return <code>true</code> se o servidor pode escrever no projeto pois  o
   *         seu dono.
   */
  private boolean isWritableFromServer() {
    return serverCanWriteProject;
  }

  /**
   * Retorna um atributo da classe info dado a sua chave.
   * 
   * @param key Chave do atributo.
   * @return Valor do atributo.
   */
  public Object getAttribute(String key) {
    return info.getAttribute(key);
  }

  /**
   * Retorna a hash de atributos do projeto.
   * 
   * @return Hash de atributos do projeto.
   */
  public Hashtable<String, Object> getAttributes() {
    return info.getAttributes();
  }

  /**
   * Adiciona um atributo ao projeto.
   * 
   * @param key Chave do atributo.
   * @param attribute Valor do atributo.
   */
  public void setAttribute(String key, Object attribute) {
    info.setAttribute(key, attribute);
  }

  /**
   * Retorna todos os projetos de um usurio. Retorna null caso o usurio no
   * tenha um "home".
   * <p>
   * <b>IMPORTANTE:</b> este mtodo faz uma chamada remota, e portanto deve ser
   * executado em uma <code>RemoteTask</code>.
   * 
   * @param userId Identificador do usurio.
   * @return Lista de UserProjectInfo contendo informaes sobre todos os
   *         projetos do usurio. Retorna null caso o usurio no tenha
   *         projetos.
   * @throws RemoteException
   */
  public static List<UserProjectInfo> getAllProjects(Object userId)
    throws RemoteException {
    return ClientRemoteLocator.projectService.getProjectsFromUser(userId);
  }

  /**
   * Verifica se o usurio possui algum projeto prprio.
   * 
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio possui algum projeto prprio
   * @throws RemoteException
   * 
   * @see #userParticipatesOnSharedProjects(Object)
   * @see #getAllProjects(Object)
   */
  public static boolean userHasHisOwnProjects(Object userId)
    throws RemoteException {
    return ClientRemoteLocator.projectService.userHasHisOwnProjects(userId);
  }

  /**
   * Obtm todos os projetos que um usurio participa, mas no  o dono.
   * 
   * @param userId Identificador do usurio.
   * @return Lista de UserProjectInfo contendo informaes sobre os projetos de
   *         outros usurios aos quais esse usurio tem acesso.
   * @throws RemoteException
   */
  public static List<UserProjectInfo> getAllUserProjectsFromOthers(Object userId)
    throws RemoteException {
    return ClientRemoteLocator.projectService.getProjectsSharedWithUser(userId);
  }

  /**
   * Verifica se um usurio participa de projetos de outros usurios.
   * 
   * @param userId identificador do usurio
   * @return <code>true</code> se o usurio participa de projetos de outros
   *         usurios
   * @throws RemoteException
   * 
   * @see #userHasHisOwnProjects(Object)
   * @see #getAllUserProjectsFromOthers(Object)
   */
  public static boolean userParticipatesOnSharedProjects(Object userId)
    throws RemoteException {
    return ClientRemoteLocator.projectService
      .userParticipatesOnSharedProjects(userId);
  }

  /**
   * Obtm um conjunto com informaes sobre todos os projetos de um usurio
   * (inclusive os projetos compartilhados por outros usurios).
   * 
   * @param userId O identificador do usurio.
   * @return Um conjunto com informaes sobre todos os projetos de um usurio.
   *         Caso o usurio no tenha acesso a nenhum projeto, o conjunto ter
   *         tamanho zero.
   * @throws RemoteException
   */
  public static List<UserProjectInfo> getAllProjectsInfo(Object userId)
    throws RemoteException {
    List<UserProjectInfo> allProjects = new ArrayList<UserProjectInfo>();
    List<UserProjectInfo> ownProjects = getAllProjects(userId);
    if (ownProjects != null) {
      allProjects.addAll(ownProjects);
    }
    List<UserProjectInfo> otherProjects = getAllUserProjectsFromOthers(userId);
    if (otherProjects != null) {
      allProjects.addAll(otherProjects);
    }
    return allProjects;
  }

  /**
   * Define que o projeto  pblico (RO ou RW). As listas de acesso no so
   * alteradas.
   * 
   * @param readOnly - flag que indica se o projeto  RO (<code>true</code>) ou
   *        RW (<code>false</code>)
   * @throws RemoteException
   * @see #updateUsersRO(Set)
   * @see #updateUsersRW(Set)
   * @see #updateUsers(Set, Set)
   * @see #makePrivate()
   * @see #makePublic(boolean)
   */
  public void makePublic(boolean readOnly) throws RemoteException {
    if (readOnly) {
      updateUsers(SharingType.ALL_RO, null, null);
    }
    else {
      updateUsers(SharingType.ALL_RW, null, null);
    }
  }

  /**
   * Define que o projeto  privado. As listas de acesso no so alteradas.
   * 
   * @throws RemoteException
   * 
   * @see #updateUsersRO(Set)
   * @see #updateUsersRW(Set)
   * @see #updateUsers(Set, Set)
   * @see #makePrivate()
   * @see #makePublic(boolean)
   */
  public void makePrivate() throws RemoteException {
    updateUsers(SharingType.PRIVATE, null, null);
  }

  /**
   * Atualiza as informaes de compartilhamento de um projeto.
   * 
   * @param sharingType - tipo de compartilhamento
   * @param usersRO - usurios com acesso RO. Este conjunto pode estar
   *        preenchido independente do projeto ser compartilhado ou no
   * @param usersRW - usurios com acesso RW. Este conjunto pode estar
   *        preenchido independente do projeto ser compartilhado ou no
   * @throws RemoteException
   * 
   * @see #updateUsersRO(Set)
   * @see #updateUsersRW(Set)
   * @see #updateUsers(Set, Set)
   * @see #makePrivate()
   * @see #makePublic(boolean)
   */
  public void updateUsers(SharingType sharingType, Set<Object> usersRO,
    Set<Object> usersRW) throws RemoteException {
    ProjectPermissions.setSharingType(info, sharingType);
    ProjectPermissions.setUsersRO(info, usersRO);
    ProjectPermissions.setUsersRW(info, usersRW);
    ClientRemoteLocator.projectService.updateUsers(getId(), sharingType,
      usersRO, usersRW);
  }

  /**
   * Atualiza ambas as listas de acesso (RO e RW) ao projeto. No altera o tipo
   * de compartilhamento.
   * 
   * @param usersRO - usurios com acesso RO
   * @param usersRW - usurios com acesso RW
   * @throws RemoteException
   * 
   * @see #updateUsersRO(Set)
   * @see #updateUsersRW(Set)
   * @see #makePrivate()
   * @see #makePublic(boolean)
   */
  public void updateUsers(Set<Object> usersRO, Set<Object> usersRW)
    throws RemoteException {
    updateUsers(ProjectPermissions.getSharingType(info), usersRO, usersRW);
  }

  /**
   * Atualiza o conjunto de usurios com acesso RO ao projeto. No altera o tipo
   * de compartilhamento.
   * 
   * @param usersRO - usurios com acesso RO. Este conjunto pode estar
   *        preenchido independente do projeto ser compartilhado ou no
   * @throws RemoteException
   * 
   * @see #updateUsersRW(Set)
   * @see #updateUsers(Set, Set)
   * @see #makePrivate()
   * @see #makePublic(boolean)
   */
  public void updateUsersRO(Set<Object> usersRO) throws RemoteException {
    updateUsers(ProjectPermissions.getSharingType(info), usersRO, null);
  }

  /**
   * Atualiza o conjunto de usurios com acesso RW ao projeto. No altera o tipo
   * de compartilhamento.
   * 
   * @param usersRW - usurios com acesso RW. Este conjunto pode estar
   *        preenchido independente do projeto ser compartilhado ou no
   * @throws RemoteException
   * 
   * @see #updateUsersRO(Set)
   * @see #updateUsers(Set, Set)
   * @see #makePrivate()
   * @see #makePublic(boolean)
   */
  public void updateUsersRW(Set<Object> usersRW) throws RemoteException {
    updateUsers(ProjectPermissions.getSharingType(info), null, usersRW);
  }

  /**
   * Remove um usurio das listas de acesso ao projeto.
   * 
   * @param userID - identificador do usurio
   * @return flag <code>true</code> indicando se o usurio foi removido,
   *         <code>false</code> se ele no tinha acesso ao projeto
   * @throws RemoteException
   */
  public boolean removeUser(Object userID) throws RemoteException {
    return ClientRemoteLocator.projectService.removeUser(getId(), userID);
  }

  /**
   * Remove Todos os projetos de um usurio. Remove tambm a pasta do usurio.
   * 
   * @param userID - identificador do usurio
   * @return flag <code>true</code> indicando se a remoo foi realizada com
   *         sucesso, <code>false</code> se houve algum erro
   * @throws RemoteException
   */
  public static boolean removeUserProjects(Object userID)
    throws RemoteException {
    return ClientRemoteLocator.projectService.removeUserProjects(userID);
  }

  /**
   * Abre um projeto de um usurio. Cria um observador para esse projeto e
   * registra esse observador no provedor para que ele seja notificado de
   * alteraes ocorridas nesse projeto.
   * 
   * @param projectId Identificador do projeto a ser aberto.
   * @param notify Indicativo de notificao aos clientes de projeto aberto.
   * @return Retorna o projeto criado ou null, caso no seja possvel cri-lo.
   * @throws RemoteException
   */
  public static CommonClientProject openProject(Object projectId, boolean notify)
    throws RemoteException {
    CommonClientProject cp =
      ClientRemoteLocator.projectService.openProject(projectId, notify);
    if (cp == null) {
      return null;
    }
    return cp;
  }

  /**
   * Abre novamente o projeto. Um projeto aberto pode ser fechado pelo servidor
   * arbitrariamente. Isso acontece, por exemplo, caso o servidor termine e
   * volte  operao enquanto o cliente permanece sendo utilizado.
   * 
   * @throws RemoteException
   */
  public void reopen() throws RemoteException {
    CommonClientProject cp =
      ClientRemoteLocator.projectService.openProject(projectId, true);
    info = cp.info;
    tree = cp.tree;
    path = cp.path;
  }

  /**
   * Cria um arquivo dentro de um diretrio desse projeto e espera que o arquivo
   * seja devolvido pelo servidor.
   * 
   * @param dir Diretrio onde o arquivo vai ser criado.
   * @param fileName Nome do arquivo a ser criado.
   * @param type Tipo do arquivo a ser criado.
   * @return O arquivo criado ou <code>null</code> caso a operao falhe.
   * @throws RemoteException
   */
  public ClientProjectFile createAndWaitForFile(ClientProjectFile dir,
    String fileName, String type) throws RemoteException {
    return createFile(dir, fileName, type);
  }

  /**
   * Cria vrios arquivos simultaneamente dentro de um diretrio desse projeto e
   * espera que o arquivos sejam devolvidos pelo servidor.
   * 
   * @param dir Diretrio onde os arquivos vo ser criados.
   * @param fileInfoList lista de dados dos arquivos a serem criados.
   * @return Os arquivos criados.
   * @throws RemoteException
   */
  public ClientProjectFile[] createAndWaitForFiles(ClientProjectFile dir,
    List<ProjectFileInfo> fileInfoList) throws RemoteException {
    return createFiles(dir, fileInfoList);
  }

  /**
   * Cria um diretrio dentro do projeto e espera que o diretrio seja devolvido
   * pelo servidor.
   * 
   * @param dir Raiz do projeto ou um sub-diretrio.
   * @param dirName Nome do diretrio a ser criado.
   * @return O diretrio criado ou <code>null</code> caso a operao falhe.
   * @throws RemoteException
   */
  public ClientProjectFile createAndWaitForDirectory(
    final ClientProjectFile dir, final String dirName) throws RemoteException {
    return createFile(dir, dirName, ProjectFileType.DIRECTORY_TYPE);
  }

  /**
   * Cria o caminho de diretrios especificado dentro do projeto, e espera at
   * que todo o caminho de diretrios esteja criado no servidor.
   * 
   * @param root raiz do projeto
   * @param dirPath caminho do diretrio a ser criado, que j deve incluir o
   *        nome do diretrio raiz do projeto
   * @throws RemoteException erro de rmi
   */
  public void createAndWaitForDirectoryPath(final ClientProjectFile root,
    String[] dirPath) throws RemoteException {
    NewDirPathObserver newDirPathObserver =
      new NewDirPathObserver(root, dirPath);
    addObserver(newDirPathObserver);

    ClientRemoteLocator.projectService.createDirectory(root.getProjectId(),
      dirPath);

    newDirPathObserver.verifyAndWaitUntilDirPathCreated();
    deleteObserver(newDirPathObserver);
  }

  /**
   * <p>
   * Solicita a criao de um arquivo ou diretrio no projeto e bloqueia at
   * receber notificao do servidor.
   * </p>
   * O mtodo instala um observador no objeto <code>CommonClientProject</code>,
   * que por sua vez  um observador de um objeto <code>ProjectObserver</code>,
   * o qual finalmente  observador remoto do projeto. Quando o servidor termina
   * a criao do arquivo, notifica o <code>ProjectObserver</code>, que notifica
   * o <code>CommonClientProject</code>, que por sua vez notifica o observador
   * criado neste mtodo. O observador "acorda" a thread principal (posta para
   * "dormir" logo aps a chamada ao mtodo remoto), para que ela possa
   * continuar o processamento, j dispondo do arquivo solicitado.
   * 
   * @param parentDir diretrio-pai para o arquivo a ser criado.
   * @param fileName nome do arquivo ou diretrio a ser criado.
   * @param type tipo do arquivo a ser criado, ou DIRECTORY_TYPE no caso de
   *        criao de diretrio.
   * @return O arquivo ou diretrio criado.
   * @throws RemoteException
   */
  private ClientProjectFile createFile(ClientProjectFile parentDir,
    String fileName, String type) throws RemoteException {
    NewFileObserver newFileObserver =
      new NewFileObserver(parentDir, new String[] { fileName });
    addObserver(newFileObserver);
    if (type.equals(ProjectFileType.DIRECTORY_TYPE)) {
      parentDir.createDirectory(fileName);
    }
    else {
      parentDir.createFile(fileName, type);
    }
    ClientProjectFile[] newFiles = newFileObserver.getNewFiles();
    deleteObserver(newFileObserver);
    if (newFiles.length < 1) {
      throw new RemoteException("newFiles < 1");
    }
    ClientProjectFile newFile = newFiles[0];
    return newFile;
  }

  /**
   * <p>
   * Solicita a criao de vrios arquivos ou diretrios simultaneamente no
   * projeto e bloqueia at receber notificao do servidor.
   * </p>
   * O mtodo instala um observador no objeto <code>CommonClientProject</code>,
   * que por sua vez  um observador de um objeto <code>ProjectObserver</code>,
   * o qual finalmente  observador remoto do projeto. Quando o servidor termina
   * a criao do arquivo, notifica o <code>ProjectObserver</code>, que notifica
   * o <code>CommonClientProject</code>, que por sua vez notifica o observador
   * criado neste mtodo. O observador "acorda" a thread principal (posta para
   * "dormir" logo aps a chamada ao mtodo remoto), para que ela possa
   * continuar o processamento, j dispondo do arquivo solicitado.
   * 
   * @param parentDir diretrio-base para os arquivos a serem criados (os
   *        caminhos so relativos a este diretrio).
   * @param fileInfoList lista de dados dos arquivos a serem criados.
   * @return os arquivos ou diretrios criados.
   * @throws RemoteException
   */
  private ClientProjectFile[] createFiles(ClientProjectFile parentDir,
    List<ProjectFileInfo> fileInfoList) throws RemoteException {
    if (fileInfoList.isEmpty()) {
      return new ClientProjectFile[0];
    }

    String[] topNames = getExpectedNames(fileInfoList);

    //TODO Esse Observador  mesmo necessrio?
    NewFileObserver newFileObserver = new NewFileObserver(parentDir, topNames);
    addObserver(newFileObserver);
    parentDir.createFiles(fileInfoList);
    ClientProjectFile[] newFiles = newFileObserver.getNewFiles();
    deleteObserver(newFileObserver);
    return newFiles;
  }

  /**
   * Obtm todos os nomes de arquivos e/ou diretrios esperados pelo observador.
   *  necessrio que se observe todos, ao invs de observar apenas o 'diretrio
   * pai', pois algum diretrio pode j existir no servidor.
   * 
   * @param pathList - lista de caminhos.
   * @return todos os nomes de arquivos e/ou diretrios, para que possam ser
   *         observados.
   */
  private String[] getExpectedNames(List<ProjectFileInfo> pathList) {
    Set<String> topNames = new HashSet<String>();
    for (ProjectFileInfo projectFileInfo : pathList) {
      String[] filePath = projectFileInfo.getPath();
      topNames.add(filePath[filePath.length - 1]);
    }
    return topNames.toArray(new String[topNames.size()]);
  }

  /**
   * Fornece o identificador do usurio ao qual o projeto pertence.
   * 
   * @return Identificador do usurio.
   */
  public Object getUserId() {
    return info.userId;
  }

  /**
   * Fornece o nome do projeto.
   * 
   * @return Nome do projeto.
   */
  public String getName() {
    return info.name;
  }

  /**
   * Fornece a dscrio do projeto.
   * 
   * @return A descrio do projeto.
   */
  public String getDescription() {
    return info.description;
  }

  /**
   * Fornece o {@link CommonProjectInfo} do projeto.
   * 
   * @return ProjectInfo
   */
  public CommonProjectInfo getInfo() {
    return info;
  }

  /**
   * Altera a descrio do projeto.
   * 
   * @param description descrio.
   */
  public void setDescription(String description) {
    info.description = description;
  }

  /**
   * Fornece o tamanho da rea reservada para o projeto.
   * 
   * @return o tamanho em GB da rea reservada ou zero caso no exista reserva
   *         de rea.
   */
  public long getLockingAreaSize() {
    final Long value =
      (Long) info.getAttribute(ProjectAttribute.LOCKING_AREA_SIZE
        .getAttributeKey());
    if (value == null) {
      return 0;
    }
    return value.longValue();
  }

  /**
   * Altera o tamanho da rea reservada para o projeto.
   * 
   * @param size o tamanho em GB da rea reservada ou zero caso no exista
   *        reserva de rea.
   */
  public void setLockingAreaSize(long size) {
    info.setAttribute(ProjectAttribute.LOCKING_AREA_SIZE.getAttributeKey(),
      Long.valueOf(size));
  }

  /**
   * Altera a configurao dos frames por usurio.
   * 
   * @param userId Identificador do usurio logado.
   * @param groupsFrameInfo Configurao do projeto do usurio logado.
   * 
   *        FIXME - A remoo das warnings deste mtodo exigiria modificaes em
   *        ApplicationPanel.
   */
  public void setGroupsFrameInfo(Object userId, Hashtable<?, ?> groupsFrameInfo) {
    Hashtable<Object, Hashtable<?, ?>> usersGrpsFrameInfo =
      (Hashtable<Object, Hashtable<?, ?>>) info
        .getAttribute(ProjectAttribute.FRAME_CONFIG.getAttributeKey());
    if (usersGrpsFrameInfo == null) {
      usersGrpsFrameInfo = new Hashtable<Object, Hashtable<?, ?>>();
    }
    usersGrpsFrameInfo.put(userId, groupsFrameInfo);
    info.setAttribute(ProjectAttribute.FRAME_CONFIG.getAttributeKey(),
      usersGrpsFrameInfo);
  }

  /**
   * Obtm as configuraes do frames de um usurio
   * 
   * @param uId Identificador dos usurios que se deseja a configurao.
   * @return groupsFrameInfo Configuraes dos frames do usurio.
   * 
   *         FIXME - A remoo das warnings deste mtodo exigiria modificaes
   *         em ApplicationPanel.
   */
  public Hashtable<?, ?> getGroupsFrameConfig(Object uId) {
    Hashtable<?, ?> usersGrpsFrameInfo =
      (Hashtable<?, ?>) info.getAttribute(ProjectAttribute.FRAME_CONFIG
        .getAttributeKey());
    if (usersGrpsFrameInfo == null) {
      return null;
    }
    Hashtable<?, ?> config = (Hashtable<?, ?>) usersGrpsFrameInfo.get(uId);
    if (config != null) {
      return config;
    }
    /* IMPORTANTE - NO REMOVER ESTE CDIGO */
    /* VERIFICAO FEITA PARA MANTER ESTE CDIGO DE ACORDO COM O QUE EXISTIA */
    /*
     * Se o usurio for o administrador do projeto, a configurao era guardada
     * direto no atributo FRAME_CONFIG
     */
    /*
     * Esta condio s ser testada da primeira vez que o projeto for aberto
     * pelo admin do mesmo
     */
    if (uId.equals(getUserId())) {
      Hashtable<?, ?> oldConfig = (Hashtable<?, ?>) usersGrpsFrameInfo.clone();
      /* Remove a configurao antiga associada ao FRAME_CONFIG */
      for (Enumeration<?> e = usersGrpsFrameInfo.keys(); e.hasMoreElements();) {
        String appId = (String) e.nextElement();
        usersGrpsFrameInfo.remove(appId);
      }
      return oldConfig;
    }
    return null;
  }

  /**
   * Retorna o arquivo associado  raiz do projeto.
   * <p>
   * IMPORTANTE: no se deve assumir que a rvore est preenchida, s 
   * garantido que o n-raiz esteja definido
   * 
   * @return arquivo associado  raiz do projeto. Seus filhos podem no ter sido
   *         ainda definidos
   * 
   */
  public ClientProjectFile getRoot() {
    return tree;
  }

  /**
   * Persiste as modificaes feitas nas informaes do projeto. Caso alguma
   * informao no possa ser alterada, seu valor original ser reestabelecido.
   * 
   * @throws RemoteException
   */
  public void modify() throws RemoteException {
    info = ClientRemoteLocator.projectService.modifyProject(projectId, info);
  }

  /**
   * Reconstri e rvore do projeto a partir do sistema de arquivos do servidor.
   * 
   * @throws RemoteException
   */
  public void refreshTree() throws RemoteException {
    ClientRemoteLocator.projectService.refreshTree(projectId);
  }

  /**
   * Atualiza o contedo do diretrio especificado no cliente, a partir da sua
   * contraparte no servidor.
   * 
   * @param dirPath caminho para o diretrio a ser atualizado
   * @throws RemoteException em caso de erro na execuo remota
   */
  public void refreshDir(String[] dirPath) throws RemoteException {
    ClientRemoteLocator.projectService.refreshDir(projectId, dirPath);
  }

  /**
   * Atualiza o contedo do diretrio especificado no cliente, a partir da sua
   * contraparte no servidor.
   * 
   * @param dir diretrio a ser atualizado
   * @throws RemoteException em caso de erro na execuo remota
   */
  public void refreshDir(ClientProjectFile dir) throws RemoteException {
    refreshDir(dir.getPath());
  }

  /**
   * Remove o projeto. Caso este mtodo retorne com sucesso, o objeto que
   * representa o projeto no poder mais ser utilizado.
   * 
   * @param projectId Identificador do projeto.
   * @throws RemoteException
   */
  public static void remove(Object projectId) throws RemoteException {
    ClientRemoteLocator.projectService.removeProject(projectId);
  }

  /**
   * Fecha um projeto. Aps chamado este mtodo, o objeto que representa o
   * projeto no poder mais ser utilizado.
   * 
   * @param notify indicativo de notificao de fechamento aos clientes
   * @throws RemoteException
   */
  public void close(boolean notify) throws RemoteException {
    uninstallObserver();
    ClientRemoteLocator.projectService.closeProject(projectId, notify);
  }

  /**
   * Indica se a rea do projeto est sob um link simblico.
   * 
   * @return indicativo
   * @throws RemoteException
   * @throws Exception se houver falhano acesso ao diretrio.
   */
  public String getLocationInServer() throws RemoteException, Exception {
    return ClientRemoteLocator.projectService
      .getProjectLocationInServer(projectId);
  }

  /**
   * Remove os arquivos especificados pelo <code>paths</code>.
   * 
   * @param paths caminhos dos arquivos (incluindo o nome do projeto)
   * @throws RemoteException
   */
  public void removeFiles(String[][] paths) throws RemoteException {
    ClientRemoteLocator.projectService.removeFiles(projectId, paths);
  }

  /**
   * Remove uma lista de arquivos. (OBS.: no remove a raiz do projeto)
   * 
   * @param files lista de arquivos a serem removidos
   * @throws RemoteException em caso de erro na execuo remota
   */
  public void removeFiles(List<ClientProjectFile> files) throws RemoteException {
    String[][] paths = new String[files.size()][];
    int j = 0;
    for (ClientProjectFile file : files) {
      paths[j++] = file.isRoot() ? null : file.getPath();
    }
    ClientRemoteLocator.projectService.removeFiles(projectId, paths);
  }

  /**
   * Retorna o caminho onde o projeto est instalado no servidor, a partir da
   * raiz (ROOT) de todos os projetos.
   * 
   * @return Caminho a partir da raiz dos projetos at este projeto.
   */
  public String[] getPath() {
    return path;
  }

  /**
   * Trata eventos de modificao do projeto, como consequncia de alguma ao
   * ocorrida no servidor. Eventos possveis: - mudana nos dados do projeto -
   * remoo do projeto - mudanas na rvore do projeto - incluso de um arquivo
   * - remoo de um arquivo - mudana de nome de um arquivo - mudana da
   * localizao de um arquivo - mudana do estado do arquivo (global,
   * publicado, construo, normal)
   * 
   * @param action Objeto que descreve o evento ocorrido e as informaes
   *        necessrias para trat-lo.
   * @throws IllegalArgumentException
   */
  public synchronized void update(ProjectEvent action) {
    if (action == null) {
      throw new IllegalArgumentException("action == null");
    }

    try {
      switch (action.event) {
        case ProjectEvent.INFO_MODIFIED:
          handleEvent((ProjectInfoModifiedEvent) action);
          break;
        case ProjectEvent.PROJECT_DELETED:
          handleEvent((ProjectDeletedEvent) action);
          break;
        case ProjectEvent.NEW_FILE:
          handleEvent((NewProjectFileEvent) action);
          break;
        case ProjectEvent.NEW_FILES:
          handleEvent((NewProjectFilesEvent) action);
          break;
        case ProjectEvent.FILE_DELETED:
          handleEvent((ProjectFileDeletedEvent) action);
          break;
        case ProjectEvent.FILES_DELETED:
          handleEvent((ProjectFilesDeletedEvent) action);
          break;
        case ProjectEvent.NEW_FILE_NAME:
          handleEvent((ProjectFileRenamedEvent) action);
          break;
        case ProjectEvent.FILE_MOVED:
          break;
        case ProjectEvent.FILE_STATE_CHANGED:
          handleEvent((ProjectFileStateChangedEvent) action);
          break;
        case ProjectEvent.TREE_CHANGED:
          handleEvent((TreeChangedEvent) action);
          break;
        case ProjectEvent.DIR_REFRESHED:
          handleEvent((DirRefreshedEvent) action);
          break;
        default:
          throw new IllegalArgumentException("Evento desconhecido. Nmero: "
            + action.event + " - Classe: " + action.getClass());
      }
    }
    catch (RemoteException e) {
      e.printStackTrace();
    }
  }

  /**
   * Trata o evento de modificao de informaes de um projeto.
   * 
   * @param event O evento a ser tratado.
   */
  private void handleEvent(ProjectInfoModifiedEvent event) {
    this.info = event.getInfo();
  }

  /**
   * Trata o evento de remoo de projeto.
   * 
   * @param event O evento a ser tratado.
   */
  private void handleEvent(ProjectDeletedEvent event) {
    tree = null;
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de incluso de um arquivo na rvore deste projeto.
   * 
   * @param event O evento a ser tratado.
   * 
   * @throws RemoteException
   */
  private void handleEvent(NewProjectFileEvent event) throws RemoteException {
    addFileFromServer(event.getPath(), event.getFile());
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de incluso simultnea de um ou mais arquivos na rvore
   * deste projeto.
   * 
   * @param event O evento a ser tratado.
   * 
   * @throws RemoteException
   */
  private void handleEvent(NewProjectFilesEvent event) throws RemoteException {
    String[][] paths = event.getPaths();
    if (paths == null) {
      throw new IllegalArgumentException("paths == null");
    }
    ClientProjectFile[] files = event.getFiles();
    if (files == null) {
      throw new IllegalArgumentException("files == null");
    }
    if (paths.length != files.length) {
      throw new IllegalArgumentException("paths.length != files.length");
    }
    for (int inx = 0; inx < files.length; inx++) {
      addFileFromServer(paths[inx], files[inx]);
    }
    setChanged();
    notifyObservers(event);
  }

  /**
   * Adiciona um novo arquivo obtido do servidor.
   * 
   * @param filePath caminho at o diretrio aonde o arquivo ser inserido.
   * @param file arquivo a ser inserido no projeto.
   * @throws RemoteException
   */
  private void addFileFromServer(String[] filePath, ClientProjectFile file)
    throws RemoteException {
    /*
     * Percorre a rvore at achar o pai.
     */
    ClientProjectFile dir = tree;
    for (int i = 0; i < filePath.length; i++) {
      dir = dir.getChild(filePath[i]);
      if (dir == null) {
        return;
      }
    }
    dir.addChild(true, true, file);
  }

  /**
   * Trata o evento de remover um arquivo da rvore deste projeto.
   * 
   * @param event O evento a ser tratado.
   * @throws RemoteException
   */
  private void handleEvent(ProjectFileDeletedEvent event)
    throws RemoteException {
    /* Obtm o caminho completo do arquivo a ser removido. */
    String[] filePath = event.getPath();
    ClientProjectFile file = tree;
    for (int i = 0; i < filePath.length; i++) {
      file = file.getChild(filePath[i]);
      if (file == null) {
        return;
      }
    }
    ClientProjectFile cParentFile = file.getParent();
    if (cParentFile == null) {
      return;
    }
    int index = cParentFile.removeChild(file);
    event.setRemovedFile(file);
    event.setRemovedFileIndex(index);
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de remover vrios arquivos da rvore deste projeto.
   * 
   * @param event O evento a ser tratado.
   * @throws RemoteException
   */
  private void handleEvent(ProjectFilesDeletedEvent event)
    throws RemoteException {
    /* Obtm o caminho completo do arquivos que foram removidos. */
    String[][] paths = event.getPaths();
    if (paths == null) {
      return;
    }
    List<ClientProjectFile> filesToRemove = new LinkedList<ClientProjectFile>();
    for (int i = 0; i < paths.length; i++) {
      ClientProjectFile file = tree.getChild(paths[i]);
      if (file == null) {
        continue;
      }
      ClientProjectFile parent = file.getParent();
      if (parent == null) {
        return;
      }
      parent.removeChild(file);
      filesToRemove.add(file);
    }
    event.setRemovedFiles(filesToRemove.toArray(new ClientProjectFile[0]));
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de renomear um arquivo na rvore deste projeto.
   * 
   * @param event O evento a ser tratado.
   * @throws RemoteException
   */
  private void handleEvent(ProjectFileRenamedEvent event)
    throws RemoteException {
    /* Obtm o caminho do arquivo */
    String[] filePath = event.getPath();
    /* Obtm o novo nome do arquivo */
    String newName = event.getNewName();
    /* Obtm o novo tipo do arquivo */
    String newType = event.getNewType();
    ClientProjectFile file = tree;
    for (int i = 0; i < filePath.length; i++) {
      file = file.getChild(filePath[i]);
    }
    if (file == null) {
      return;
    }
    file.setName(newName);
    file.setType(newType);

    /* Atualiza o path dos descendentes */
    file.updatePath();

    event.setFile(file);
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de atualizao do estado do arquivo da rvore do projeto.
   * 
   * @param event O evento a ser tratado.
   * @throws RemoteException
   */
  private void handleEvent(ProjectFileStateChangedEvent event)
    throws RemoteException {
    /* Obtm o caminho do arquivo cujo estado mudou */
    String[] filePath = event.getPath();
    /* Obtm o estado modificado */
    boolean isUnderConstruction = event.isUnderConstruction();
    ClientProjectFile file = tree;
    for (int i = 0; i < filePath.length; i++) {
      file = file.getChild(filePath[i]);
      // FIXME O evento de mudana do arquivo sob construo, s vezes, est
      // ocorrendo antes do evento que indica que o arquivo foi criado.
      // A soluo definitiva para este problema  o servidor deve garantir a
      // ordem das notificaes dos eventos remotos. Como a soluo afetaria
      // vrias partes do sistema, ela ser feita em um momento mais oportuno.
      // Por enquanto, se o problema ocorrer, este evento  ignorado.
      if (file == null) {
        return;
      }
    }
    file.setName(event.getName());
    file.setUnderConstruction(isUnderConstruction);
    file.setLocked(event.isLocked());
    file.setSize(event.getSize());
    file.setModificationDate(event.getModificationDate());
    event.setFile(file);
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de atualizao da rvore do projeto.
   * 
   * @param event O evento a ser tratado.
   */
  private void handleEvent(TreeChangedEvent event) {
    /* Obtm a nova rvore do projeto */
    tree = event.getNewTree();
    setChanged();
    notifyObservers(event);
  }

  /**
   * Trata o evento de reconstruo de um diretrio do cliente, a partir das
   * informaes recuperadas do servidor. Substitui o objeto que representa o
   * diretrio no cliente por uma verso atualizada com os dados do servidor.
   * 
   * @param event informaes sobre o evento.
   * @throws RemoteException
   */
  private void handleEvent(DirRefreshedEvent event) throws RemoteException {
    String[] dirPath = event.getPath();
    ClientProjectFile dir = tree;
    for (int i = 0; i < dirPath.length; i++) {
      dir = dir.getChild(dirPath[i]);
    }
    ClientProjectFile parent = dir.getParent();
    if (parent == null) {
      return;
    }
    parent.removeChild(dir);
    parent.addChild(true, true, event.getDir());
    setChanged();
    notifyObservers(event);
  }

  /**
   * Instala um observador desse projeto no servidor.
   * 
   * @param projObserver .
   */
  public void installObserver(ProjectObserver projObserver) {
    this.observer = projObserver;
  }

  /**
   * Desinstala um observador desse projeto no servidor.
   * 
   * @throws RemoteException
   */
  public void uninstallObserver() throws RemoteException {
    if (observer != null) {
      observer.uninstall();
      observer = null;
    }
  }

  /**
   * Mtodo utilitrio para converso de um array de strings que representa um
   * <i>path</i> em um arquivo no projeto corrente; caso no exista o arquivo,
   * retorna-se <code>null</code>. O nome do projeto no deve estar no
   * <i>path</i>.
   * 
   * @param filePath o array de String com o <i>path</i>.
   * @return o objeto que representa o arquivo ou <code>null</code>.
   * @see ClientProjectFile
   * @throws RemoteException caso {@link ClientProjectFile#getChild(String)}
   *         aplicado  raiz do projeto falhe.
   */
  public final ClientProjectFile getFile(final String[] filePath)
    throws RemoteException {
    if (filePath == null || (filePath.length == 0)) {
      throw new InvalidParameterException("Erro interno.\n"
        + "Parmetro invlido em ApplicationProject.stringArrayToFile");
    }
    // Obtm a raiz do projeto
    ClientProjectFile root = getRoot();

    // Insere o nome do projeto no path
    String[] rootPath = root.getPath();
    String[] fullPath = new String[rootPath.length + filePath.length];
    System.arraycopy(rootPath, 0, fullPath, 0, rootPath.length);
    System.arraycopy(filePath, 0, fullPath, rootPath.length, filePath.length);

    ClientProjectFile file = root.getChild(fullPath);
    return file;
  }

  /**
   * Indica que o projeto pode estar desatualizado.
   */
  public void setOutOfDate() {
    setUpdatedTree(tree, false);
  }

  /**
   * Percorre uma sub-rvore, a partir de sua raz, indicando se cada arquivo
   * pode estar desatualizado.
   * 
   * @param root a raz da sub-rvore que se deseja percorrer.
   * @param updated true se a sub-rvore est atualizada.
   */
  private void setUpdatedTree(ClientProjectFile root, boolean updated) {
    root.setUpdated(updated);
    if (!root.isDirectory()) {
      return;
    }
    /**
     * No queremos buscar ns da rvore que ainda no foram carregados, mas
     * apenas marcar os que j foram carregados no cliente.
     */
    ClientProjectFile[] localChildren = root.getLocalChildren();
    if (localChildren != null) {
      for (ClientProjectFile child : localChildren) {
        setUpdatedTree(child, updated);
      }
    }
  }

  /**
   * Cria a viso cliente de um projeto.
   * 
   * @param id Identificador do projeto.
   * @param info Informaes do projeto.
   * @param tree Viso cliente da rvore do projeto.
   * @param path Caminho de onde esse projeto est instalado, a partir da raiz
   *        de todos os projetos.
   */
  public CommonClientProject(Object id, CommonProjectInfo info,
    ClientProjectFile tree, String[] path) {
    this.projectId = id;
    this.info = info;
    this.tree = tree;
    this.path = path;
  }
}

/**
 * Observador do projeto, responsvel por aguardar notificaes do servidor
 * relativas  criao de arquivos no projeto.
 */
class NewFileObserver implements Observer {
  /**
   * Referncia para os arquivos/diretrios a serem criados no servidor.
   */
  private ClientProjectFile[] newFiles;

  /**
   * Diretrio-pai para os arquivos/diretrios aguardados.
   */
  private final ClientProjectFile parentDir;

  /**
   * Nomes do arquivos/diretrios aguardados.
   */
  private final Collection<String> names;

  /**
   * Cria um observador para arquivos/diretrios recm-criados.
   * 
   * @param parentDir diretrio-pai para o arquivo/diretrio aguardado.
   * @param names nomes do arquivos/diretrios aguardados.
   */
  public NewFileObserver(ClientProjectFile parentDir, String[] names) {
    this.parentDir = parentDir;
    this.names = Arrays.asList(names);
  }

  /**
   * <p>
   * Aguarda notificaes de evento "arquivos/diretrios recm-criados" no
   * servidor. Armazena a referncia para o arquivos/diretrios e acorda a
   * thread principal para continuar o processamento.
   * </p>
   * Caso eventos no-aguardados sejam recebidos, o mtodo simplesmente retorna.
   * 
   * @param observable .
   * @param arg .
   */
  @Override
  public void update(Observable observable, Object arg) {
    ProjectEvent action = (ProjectEvent) arg;
    if (action == null) {
      return;
    }
    switch (action.event) {
      case ProjectEvent.NEW_FILE: {
        NewProjectFileEvent event = (NewProjectFileEvent) action;
        ClientProjectFile file = event.getFile();
        if (file == null) {
          return;
        }
        String[] path = event.getPath();
        if (path == null) {
          return;
        }
        if (!pathEquals(path, parentDir.getPath())
          || !names.contains(file.getName())) {
          return;
        }
        setNewFiles(new ClientProjectFile[] { file });
        break;
      }
      case ProjectEvent.NEW_FILES: {
        NewProjectFilesEvent event = (NewProjectFilesEvent) action;
        ClientProjectFile[] files = event.getFiles();
        if (files == null || files.length == 0) {
          return;
        }
        String[][] paths = event.getPaths();
        if (paths.length != files.length) {
          return;
        }

        /*
         * caso corrigido: se j existe o diretrio escolhido numa importao de
         * arquivos locais para o projeto, apenas arquivos internos a ele sero
         * retornados pelo servidor [como 'novos arquivos']. o problema  que o
         * cliente ainda espera aquele diretrio escolhido, e a notificao no
         *  consumida. Para evitar esse tipo de erro, vamos apenas verificar se
         * o caminho 'inicia' com o diretrio escolhido, e vamos incluir todos
         * os nomes de arquivos escolhidos para a importao, pois o servidor
         * nem sempre retorna exatamente o que foi pedido.
         */
        for (int i = 0; i < files.length; i++) {
          if (!checkPath(paths[i], parentDir.getPath())
            || !names.contains(files[i].getName())) {
            return;
          }
        }
        setNewFiles(files);
        break;
      }
      default:
        return;
    }
  }

  /**
   * Verifica se um caminho  filho ou igual a um outro.
   * 
   * @param pathToCheck - caminho que ser analisado.
   * @param referencePath - caminho 'pai', de referncia.
   * @return TRUE se 'pathToCheck' for filho de 'referencePath'. Se forem o
   *         mesmo caminho, tambm retorna TRUE.
   */
  private boolean checkPath(String[] pathToCheck, String[] referencePath) {
    if (pathToCheck.length < referencePath.length) {
      return false;
    }
    for (int i = 0; i < referencePath.length; i++) {
      if (!pathToCheck[i].equals(referencePath[i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * Atualiza o observador com os arquivos recm-criados no servidor. O mtodo
   * notifica a thread responsvel por obter os arquivos de que os mesmos j se
   * encontram disponveis.
   * 
   * @param files arquivos recm-criados no servidor.
   */
  private synchronized void setNewFiles(ClientProjectFile[] files) {
    newFiles = files;
    notify();
  }

  /**
   * Verifica se os arquivos solicitados j foram criados. Se no estiverem
   * ainda disponveis, "dorme" at ser notificado por outra thread. Caso
   * contrrio, retorna os arquivos.
   * 
   * @return os arquivos recm-criados (assim que estiverem disponveis).
   */
  public synchronized ClientProjectFile[] getNewFiles() {
    while (newFiles == null) {
      try {
        wait();
      }
      catch (InterruptedException e) {
      }
    }
    return newFiles;
  }

  /**
   * Compara dois caminhos e retorna <code>true</code> se eles forem iguais.
   * 
   * @param path1 caminho 1.
   * @param path2 caminho 2.
   * @return <code>true</code> se os dois caminhos forem iguais.
   */
  private static final boolean pathEquals(String[] path1, String[] path2) {

    /*
     * Se ambos path1 e path2 so nulos, ento so iguais.
     */
    if (null == path1 && null == path2) {
      return true;
    }

    /*
     * Como a condio acima no foi atendida, sabemos que ambos path1 e path2
     * no so nulos, ento se um deles for, so diferentes.
     */
    if (path1 == null || path2 == null) {
      return false;
    }
    // Cdigo antigo, que confundia o eclipse e fazia aparecer um
    // warning desnecessrio, indicando que havia um potential null pointer
    // no if abaixo deste (de comparao de length).
    //    if ((null == path1 && null != path2) || (null != path1 && null == path2)) {
    //      return false;
    //    }

    if (path1.length != path2.length) {
      return false;
    }
    for (int inx = 0; inx < path1.length; inx++) {
      if (!path1[inx].equals(path2[inx])) {
        return false;
      }
    }
    return true;
  }
}

/**
 * Observador do projeto, responsvel por aguardar notificaes do servidor
 * relativas  criao de uma rvore de diretrios no projeto.
 */
class NewDirPathObserver implements Observer {
  /**
   * Referncia para os arquivos/diretrios a serem criados no servidor.
   */
  private List<ClientProjectFile> newDirList;

  /**
   * Diretrio-pai para os arquivos/diretrios aguardados.
   */
  private final ClientProjectFile parentDir;

  /**
   * Nomes do arquivos/diretrios aguardados.
   */
  private final Collection<String> names;

  /**
   * Cria um observador para arquivos/diretrios recm-criados.
   * 
   * @param parentDir diretrio-pai para o arquivo/diretrio aguardado.
   * @param names nomes do arquivos/diretrios aguardados.
   */
  public NewDirPathObserver(ClientProjectFile parentDir, String[] names) {
    this.parentDir = parentDir;
    this.names = Arrays.asList(names);
    newDirList = new ArrayList<ClientProjectFile>();
    newDirList.add(parentDir);
  }

  /**
   * <p>
   * Aguarda notificaes de evento "arquivos/diretrios recm-criados" no
   * servidor. Armazena a referncia para o arquivos/diretrios e acorda a
   * thread principal para continuar o processamento.
   * </p>
   * Caso eventos no-aguardados sejam recebidos, o mtodo simplesmente retorna.
   * 
   * @param observable .
   * @param arg .
   */
  @Override
  public void update(Observable observable, Object arg) {
    ProjectEvent action = (ProjectEvent) arg;
    if (action == null) {
      return;
    }
    switch (action.event) {
      case ProjectEvent.NEW_FILE: {
        NewProjectFileEvent event = (NewProjectFileEvent) action;
        ClientProjectFile file = event.getFile();
        if (file == null) {
          return;
        }
        if (!file.getParent().equals(parentDir)
          && !names.contains(file.getName())
          && !names.contains(file.getParent().getName())) {
          return;
        }
        addDirListAndNotify(file);
        break;
      }
      default:
        return;
    }
  }

  /**
   * Adiciona o diretrio criado na lista com os diretrios recm-criados no
   * servidor. O mtodo notifica a thread responsvel por informar que os
   * diretrios j se encontram disponveis.
   * 
   * @param dir - o diretrio criado no servidor.
   */
  private synchronized void addDirListAndNotify(ClientProjectFile dir) {
    if (!newDirList.contains(dir.getParent())) {
      newDirList.add(dir.getParent());
    }
    newDirList.add(dir);
    if (newDirList.size() == names.size()) {
      notify();
    }
  }

  /**
   * Verifica se os diretrios do caminho especificado j foram criados. Se no
   * estiverem ainda disponveis, "dorme" at ser notificado por outra thread.
   */
  public synchronized void verifyAndWaitUntilDirPathCreated() {
    while (newDirList.size() != names.size()) {
      try {
        wait();
      }
      catch (InterruptedException e) {
      }
    }
  }
}
