package csbase.logic;

import java.io.Serializable;
import java.security.DigestException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;

/**
 * Modela a chave que  usada para identificar a sesso de um usurio.
 */
public class SecureKey implements Serializable {
  private final int KEY_SIZE = 10;
  private byte[] value = new byte[KEY_SIZE];
  private int hashCode;

  /** O nome do algoritmo de gerao de nmeros pseudo-aleatrios. */
  private static final String PRNG_ALGORITHM = "SHA1PRNG";

  /**
   * Verifica a igualdade com outro objeto.
   * 
   * @param key A outra chave a ser comparada.
   * 
   * @return Verdadeiro, se os objetos forem da mesma classe e seus valores
   *         forem iguais, ou falso caso contrrio.
   */
  @Override
  public boolean equals(Object key) {
    if (key instanceof SecureKey) {
      SecureKey anotherKey = (SecureKey) key;
      return Arrays.equals(value, anotherKey.value);
    }
    else {
      return false;
    }
  }

  /**
   * Obtm o cdigo usado para indexao em HashTable.
   * 
   * @return Um cdigo hash.
   */
  @Override
  public int hashCode() {
    return hashCode;
  }

  /**
   * Gera um cdigo hash para esse objeto. O cdigo  calculado a partir dos
   * valores presentes na chave.
   * 
   * @return Um cdigo hash.
   */
  private int generateHashCode() {
    int hashCode = 1;
    for (int i = 0; i < KEY_SIZE; i++) {
      hashCode = (31 * hashCode) + (Byte.valueOf(value[i])).hashCode();
    }
    return hashCode;
  }

  /**
   * Obtm uma string que mostra os valores da chave.
   * 
   * @return Uma string com o valor da chave.
   */
  @Override
  public String toString() {
    StringBuffer str = new StringBuffer();
    for (int i = 0; i < KEY_SIZE; i++) {
      str.append(value[i]);
    }
    str.append('\n');
    return str.toString();
  }

  /**
   * Representao em hexadecimal do message digest da <code>SecureKey</code>.
   * 
   * @return Uma string com o digest.
   * 
   * @throws DigestException
   */
  public String digest() throws DigestException {
    return MDigest.getDigest(value);
  }

  /**
   * Constri uma nova chave usada para identificar a sesso de um usurio.
   */
  public SecureKey() {
    SecureRandom secureRandom = null;
    try {
      secureRandom = SecureRandom.getInstance(PRNG_ALGORITHM);
    }
    catch (NoSuchAlgorithmException e) {
      secureRandom = new SecureRandom();
    }
    secureRandom.nextBytes(value);
    hashCode = generateHashCode();
  }
}
