/*
 * $Id:$
 */
package csbase.logic;

import java.rmi.RemoteException;
import java.text.MessageFormat;
import java.util.Set;

import csbase.exception.CSBaseException;
import csbase.remote.RemoteObserver;
import csbase.remote.ServerEntryPoint;
import csbase.remote.ServiceInterface;

/**
 * O ServerManager  responsvel por manter, adicionar e remover um servidor
 * CSBase da monitorao. Ao menos um servidor estar sendo monitorado j que o
 * cliente est conectado no servidor inicial (default).
 * 
 * @author Tecgraf
 */
public class ServerManager {

  /**
   * Mantm os dados dos servidores gerenciados
   */
  private ServerManagerData serverManagerData;

  /**
   * Constri o gerente de servidores
   * 
   * @param sharedServerData Se true os dados gerenciados so compartilhados
   *        entre as instncias desta classe, false os dados so independentes
   */
  protected ServerManager(boolean sharedServerData) {
    this(sharedServerData, true);
  }

  /**
   * Constri o gerente de servidores
   * 
   * @param sharedServerData Se true os dados gerenciados so compartilhados
   *        entre as instncias desta classe, false os dados so independentes
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   */
  protected ServerManager(boolean sharedServerData, boolean verbose) {
    if (!sharedServerData) {
      this.serverManagerData = new ServerManagerData();
    }
    else {
      this.serverManagerData = ServerManagerData.getInstance();
    }
    serverManagerData.setVerbose(verbose);
  }

  /**
   * Constri o gerente de servidores compartilhandos os dados dos servidores
   * gerenciados entre todas as instncias desta classe
   */
  protected ServerManager() {
    this(true);
  }

  /**
   * Define se mensagens informativas devem ser exibidas.
   * 
   * @param verbose <code>true</code> se informaes devem ser exibidas,
   *        <code>false</code> se apenas erros e alertas devem ser exibidos
   */
  public void setVerbose(boolean verbose) {
    serverManagerData.setVerbose(verbose);
    getMonitor().setVerbose(verbose);
  }

  /**
   * Indica se o servidor est vivo.
   * 
   * @param serverURI A URI do servidor
   * @return true se o servidor estiver vivo, false caso contrrio
   */
  public final boolean isAlive(ServerURI serverURI) {
    return this.getMonitoredServer(serverURI).isAlive();
  }

  /**
   * Consulta o estado do servidor. O servidor  considerado vivo se est
   * alcanvel e o login foi bem sucedido.
   * 
   * @return true para servidor vivo, false caso contrrio.
   */
  public final boolean isAlive() {
    return this.getMonitoredServer(this.getDefaultURI()).isAlive();
  }

  /**
   * Adiciona ouvintes a todos os servidores gerenciados.
   * 
   * @param l A instncia do ouvinte.
   */
  public final void addCommonListener(MonitoredServerListener l) {
    this.serverManagerData.addCommonListener(l);
  }

  /**
   * Remove o ouvinte de todos o servidores gerenciados.
   * 
   * @param l A instncia do ouvinte
   */
  public final void deleteCommonListener(MonitoredServerListener l) {
    this.serverManagerData.deleteCommonListener(l);
  }

  /**
   * Adiciona um ouvinte do tipo {@link MonitoredServerListener} para um
   * servidor
   * 
   * @param serverURI A URI que identifica o servidor sendo monitorado
   * @param l A instncia do ouvinte a ser adicionada
   */
  public final void addListener(ServerURI serverURI, MonitoredServerListener l) {
    this.getMonitoredServer(serverURI).addListener(l);
  }

  /**
   * Remove o ouvinte do tipo {@link MonitoredServerListener} de um servidor
   * 
   * @param serverURI A URI que identifica o servidor sendo monitorado
   * @param l A instncia do ouvinte a ser removida
   */
  public final void deleteListener(ServerURI serverURI,
    MonitoredServerListener l) {
    this.getMonitoredServer(serverURI).deleteListener(l);
  }

  /**
   * Adiciona um ouvinte do tipo {@link MonitoredServerListener} ao servidor
   * default
   * 
   * @param l A instncia do ouvinte a ser adicionada
   */
  public final void addListener(MonitoredServerListener l) {
    this.getMonitoredServer(this.getDefaultURI()).addListener(l);
  }

  /**
   * Remove um ouvinte do tipo {@link MonitoredServerListener} do servidor
   * default
   * 
   * @param l A instncia do ouvinte a ser removida
   */
  public final void deleteListener(MonitoredServerListener l) {
    this.getMonitoredServer(this.getDefaultURI()).deleteListener(l);
  }

  /**
   * Adiciona um novo servidor para ser gerenciado. Os ouvintes comuns so
   * adicionados mas a monitorao no  iniciada neste momento. Se j houver
   * uma monitorao a mesma ser perdida e no ser possvel encerr-la, sendo
   * assim  importante que seja removida atravs de um logout antes da adio
   * da nova.
   * 
   * @param monitoredServer A Instncia do {@link MonitoredServer} do servidor a
   *        ser gerenciado
   */
  public void addServer(MonitoredServer monitoredServer) {
    this.serverManagerData.addServer(monitoredServer);
  }

  /**
   * Remove servidor do server-manager.
   * 
   * @param sURI URI
   */
  private void removeServer(ServerURI sURI) {
    this.serverManagerData.removeServer(sURI);
  }

  /**
   * Recupera a instncia de monitorao de um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return A instncia de monitorao ({@link MonitoredServer})
   */
  private MonitoredServer getMonitoredServer(ServerURI uri) {
    return this.serverManagerData.getMonitoredServer(uri);
  }

  /**
   * @param serverURI A URI que identifica o servidor
   * @return true se o servidor estiver sendo monitorado, false caso contrrio
   */
  public final boolean isMonitored(ServerURI serverURI) {
    return this.serverManagerData.isMonitored(serverURI);
  }

  /**
   * Recupera o monitor de um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return A instncia de {@link ServerMonitor}
   */
  public ServerMonitor getMonitor(ServerURI uri) {
    return this.getMonitoredServer(uri).monitor;
  }

  /**
   * Recupera o monitor do servidor default
   * 
   * @return A instncia de {@link ServerMonitor}
   */
  public ServerMonitor getMonitor() {
    return this.getMonitoredServer(this.getDefaultURI()).monitor;
  }

  /**
   * Retorna a referncia para o ponto de entrada de um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return A referncia do servidor
   */
  public ServerEntryPoint getServer(ServerURI uri) {
    return this.getMonitoredServer(uri).monitor.getServer();
  }

  /**
   * Retorna a referncia para o ponto de entrada do servidor default
   * 
   * @return A referncia do servidor
   */
  public ServerEntryPoint getServer() {
    return this.getMonitoredServer(this.getDefaultURI()).monitor.getServer();
  }

  /**
   * Obtm o conjunto de URI's que identifica os servidores sendo gerenciados
   * 
   * @return Um conjunto de {@link ServerURI}
   */
  public Set<ServerURI> getManagedServers() {
    return this.serverManagerData.getManagedServers();
  }

  /**
   * Executa o lookup de um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return true se servidor contactado com sucesso, false caso contrrio
   */
  public final boolean performLookup(ServerURI uri) {
    return this.getMonitoredServer(uri).monitor.lookup();
  }

  /**
   * Executa o lookup do servidor default
   * 
   * @return true se servidor contactado com sucesso, false caso contrrio
   */
  public final boolean performLookup() {
    return this.getMonitoredServer(this.getDefaultURI()).monitor.lookup();
  }

  /**
   * Executa o login e recupera a referncia para os servios de um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return true se validado com sucesso, false caso contrrio
   * @throws CSBaseException Em caso de erro durante a validao
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  protected boolean validate(ServerURI uri) throws CSBaseException,
    RemoteException {
    return this.getMonitoredServer(uri).validate();
  }

  /**
   * Executa o login e recupera a referncia para os servios do servidor
   * default
   * 
   * @return true se validado com sucesso, false caso contrrio
   * @throws CSBaseException Em caso de erro durante a validao
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  protected boolean validate() throws CSBaseException, RemoteException {
    return this.getMonitoredServer(this.getDefaultURI()).validate();
  }

  /**
   * Recupera a sesso do usurio em um servidor
   * 
   * @param uri A URI que identifica o servidor
   * @return A instncia da sesso
   */
  public Session getSession(ServerURI uri) {
    return this.getMonitoredServer(uri).session;
  }

  /**
   * Recupera a sesso do usurio no servidor default
   * 
   * @return A instncia da sesso
   */
  public Session getSession() {
    return this.getMonitoredServer(this.getDefaultURI()).session;
  }

  /**
   * Fora a verificao de estado do monitor de um servidor
   * 
   * @param serverURI A URI do servidor
   */
  public void invalidate(ServerURI serverURI) {
    this.getMonitoredServer(serverURI).invalidate();
  }

  /**
   * Fora a verificao de estado do monitor do servidor default
   */
  public void invalidate() {
    invalidate(getDefaultURI());
  }

  /**
   * Define o servidor default
   * 
   * @param serverURI A URI do servidor
   */
  public final void setDefaultServer(ServerURI serverURI) {
    if (serverURI == null) {
      throw new IllegalArgumentException("serverURI == null");
    }

    if (!this.isMonitored(serverURI)) {
      throw new IllegalStateException(MessageFormat.format(
        "Servidor {0} no est na monitorao", serverURI));
    }

    this.serverManagerData.setDefaultServer(serverURI);
  }

  /**
   * @return A URI do servidor default
   */
  public final ServerURI getDefaultURI() {
    return this.serverManagerData.getDefaultURI();
  }

  /**
   * Obtm o endereo para conexo com o servidor na forma host:port
   * 
   * @return A string do endereo para conexo com o servidor na forma host:port
   */
  public final String getServerPath() {
    return this.getDefaultURI().getHostAndPort();
  }

  /**
   * Define o nome do sistema a ser usado no servidor
   * 
   * @param sURI A URI do servidor
   * @param systemName O nome do sistema
   */
  protected void setSystemName(ServerURI sURI, String systemName) {
    this.getMonitoredServer(sURI).setSystemName(systemName);
  }

  /**
   * Testa se um determinado servidor  o default
   * 
   * @param sURI A URI do servidor
   * @return true se o servidor  o default, false caso contrrio
   */
  public final boolean isDefault(ServerURI sURI) {
    return this.getMonitoredServer(sURI).isDefault();
  }

  /**
   * Retorna a thread de lookup do monitor de um servidor
   * 
   * @param serverURI A URI do servidor
   * 
   * @return A thread de lookup
   */
  public Thread getServerLookupThread(ServerURI serverURI) {
    return this.getMonitor(serverURI).getServerLookupThread();
  }

  /**
   * @return A thread de lookup do monitor do servidor default
   */
  public Thread getServerLookupThread() {
    return this.getMonitor(this.getDefaultURI()).getServerLookupThread();
  }

  /**
   * Adiciona um observador.
   * 
   * @param serverURI A URI do servidor
   * @param serviceName O nome do servio
   * @param observer O observador
   * @param arg Os argumentos associados ao observador
   * @return true, caso tenha conseguido adicionar o observador, ou false, caso
   *         contrrio.
   */
  public final boolean addObserver(ServerURI serverURI, String serviceName,
    RemoteObserver observer, Object arg) {
    return this.getMonitoredServer(serverURI).addObserver(serviceName,
      observer, arg);
  }

  /**
   * Adiciona um observador ao servidor padro.
   * 
   * @param serviceName O nome do servio
   * @param observer O observador
   * @param arg Os argumentos associados ao observador
   * @return true, caso tenha conseguido adicionar o observador, ou false, caso
   *         contrrio.
   */
  public final boolean addObserver(String serviceName, RemoteObserver observer,
    Object arg) {
    return this.addObserver(this.getDefaultURI(), serviceName, observer, arg);
  }

  /**
   * Remove um observador.
   * 
   * @param serverURI A URI do servidor
   * @param serviceName O nome do servio
   * @param observer O observador
   * @param arg Os argumentos associados ao observador
   */
  public final synchronized void deleteObserver(ServerURI serverURI,
    String serviceName, RemoteObserver observer, Object arg) {
    this.getMonitoredServer(serverURI).deleteObserver(serviceName, observer,
      arg);
  }

  /**
   * Remove um observador do servidor padro.
   * 
   * @param serviceName O nome do servio
   * @param observer O observador
   * @param arg Os argumentos associados ao observador
   */
  public final synchronized void deleteObserver(String serviceName,
    RemoteObserver observer, Object arg) {
    this.deleteObserver(this.getDefaultURI(), serviceName, observer, arg);
  }

  /**
   * Remove todos os observadores cadastrados
   * 
   * @param serverURI A URI do servidor
   */
  protected final synchronized void removeObservers(ServerURI serverURI) {
    this.getMonitoredServer(serverURI).removeObservers();
  }

  /**
   * Remove todos os observadores do servidor padro
   */
  protected final void removeObservers() {
    this.removeObservers(this.getDefaultURI());
  }

  /**
   * Executa logout em um servidor, pra a monitorao, remove observadores e
   * ouvintes e por fim notifica os ouvintes do gerente do logout bem sucedido.
   * 
   * @param serverURI A URI do servidor
   * @param flush Se true remove o servidor do gerente
   */
  public final void logout(ServerURI serverURI, boolean flush) {
    if (flush) {
      this.removeServer(serverURI);
    }
    else {
      this.getMonitoredServer(serverURI).logout();
    }
  }

  /**
   * Executa logout do servidor default, pra a monitorao, remove observadores
   * e ouvintes e por fim remove o servidor do gerenciamento e notifica os
   * ouvintes do gerente do logout bem sucedido.
   * 
   * @param flush Se true remove o servidor do gerente
   */
  public final void logout(boolean flush) {
    this.logout(this.getDefaultURI(), flush);
  }

  /**
   * Finaliza a monitorao dos servidores.
   */
  public void shutdown() {
    this.serverManagerData.shutdown();
    User.registerLogout();
  }

  /**
   * Compara a verso do sistema com a do servidor
   * 
   * @param serverURI A URI do servidor
   * @return true se a verso  a mesma do servidor, false caso contrrio
   * 
   */
  public final boolean isSameVersion(ServerURI serverURI) {
    return this.getMonitoredServer(serverURI).isSameVersion();
  }

  /**
   * Compara a verso do sistema com a do servidor default
   * 
   * @return true se a verso  a mesma do servidor default, false caso
   *         contrrio
   */
  public final boolean isSameVersion() {
    return this.isSameVersion(this.getDefaultURI());
  }

  /**
   * Executa o login no servidor. O servidor precisa estar na monitorao. O
   * tipo de login ser executado de acordo com a monitorao para esse
   * servidor.
   * 
   * 
   * @param serverURI A URI do servidor
   * @return true se login bem sucedido, false caso contrrio
   * @throws CSBaseException Em caso de erro durante o processo de login
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final boolean login(ServerURI serverURI) throws CSBaseException,
    RemoteException {
    return this.getMonitoredServer(serverURI).login();
  }

  /**
   * Executa o login de um servidor monitorado pelo gerente.
   * 
   * @param serverURI A URI do servidor
   * @return true se login de sucesso ou false se login/senha invlido
   * @throws CSBaseException Se houver falha durante o login.
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  protected boolean doLogin(ServerURI serverURI) throws CSBaseException,
    RemoteException {
    MonitoredServer monitoredServer = this.getMonitoredServer(serverURI);
    boolean validated = false;

    try {
      validated = monitoredServer.login();
    }
    finally {
      if (!validated) {
        this.removeServer(serverURI);
      }
      else {
        monitoredServer.getMonitor().startMonitoring();
      }
    }

    return validated;
  }

  /**
   * 
   * Retorna a referncia para um servio de um servidor. Opcionalmente o
   * servidor pode ser adicionado sob demanda.
   * 
   * @param <T> O tipo do servio a ser retornado
   * @param serviceClass a classe do servio
   * @param serverURI A URI do servidor.
   * @return A instncia do servio ou null se no for encontrado a instncia do
   *         servio
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final <T> T getService(Class<T> serviceClass, ServerURI serverURI)
    throws RemoteException {

    MonitoredServer mServer = null;

    if (serverURI != null) {
      mServer = this.getMonitoredServer(serverURI);
    }
    else {
      throw new IllegalArgumentException("serverURI == null");
    }

    for (ServiceInterface s : mServer.services.values()) {
      if (serviceClass.isInstance(s)) {
        return serviceClass.cast(s);
      }
    }
    return null;
  }

  /**
   * Retorna a referncia para um servio do servidor default.
   * 
   * @param <T> O tipo do servio a ser retornado
   * @param serviceClass A classe do servio
   * @return A instncia do servio ou null se no for encontrado a instncia do
   *         servio
   * @throws RemoteException Em caso de falha na comunicao com o servidor
   */
  public final <T> T getService(Class<T> serviceClass) throws RemoteException {
    return this.getService(serviceClass, this.getDefaultURI());
  }

}
