/*
 * $Id$
 */
package csbase.logic;

import tecgraf.javautils.core.io.FileUtils;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.text.MessageFormat;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Representa uma verso do sistema.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class Version implements Comparable<Version>, Serializable {
  /**
   * Formato para apresentao da data da verso.
   */
  private static final String VERSION_DATE_FORMAT = "%s/%s/%s";

  /**
   * Data associada a um sistema sem verso.
   */
  public static final String NO_DATE_STR = String.format(VERSION_DATE_FORMAT,
    "??", "??", "????");

  /**
   * Formato para apresentao do identificador da verso.
   */
  private static final String VERSION_ID_FORMAT = "v%s.%s.%s";

  /**
   * String associada a um sistema sem verso.
   */
  public static final String BAD_VERSION_STR = String.format(VERSION_ID_FORMAT,
    "?", "??", "??");

  /**
   * Expresso regular para processamento do identificador da verso do sistema,
   * gravado no arquivo version.properties no formato PRJ_vXX_YY_ZZ_YYYY_MM_DD.
   */
  private static final String VERSION_ID_REGEX =
    "[^_]+_v(\\d)_(\\d{2})_(\\d{2})_(\\d{4})_(\\d{2})_(\\d{2})";

  /** O nome da propriedade que contm a verso do sistema. */
  private static final String PROPERTY_VERSION = "version";

  /** O caminho para o arquivo que contm a verso do sistema. */
  private static final String VERSION_FILE =
    "/csbase/resources/version.properties";

  /** A instncia nica da verso no sistema. */
  private static Version instance;

  /** O nome da verso. */
  private String name;
    
  /** O nome do projeto */
  private String projectName;

  /**
   * Identificador da verso do sistema no formato "vX.YY.ZZ".
   */
  protected String version;

  /**
   * Identificador da data da verso no formato "DD/MM/YYYY".
   */
  protected String releaseDate;

  /**
   * Major version number (o X em X.Y.Z).
   */
  protected int major;

  /**
   * Minor version number (o Y em X.Y.Z).
   */
  protected int minor;

  /**
   * Patch version number (o Z em X.Y.Z).
   */
  protected int patch;

  /**
   * Dia associado  verso.
   */
  protected int day;

  /**
   * Ms associado  verso.
   */
  protected int month;

  /**
   * Ano associado  verso.
   */
  protected int year;

  /**
   * Indica se a verso est com o formato vlido.
   */
  private boolean valid;

  /**
   * Cria uma verso do sistema.
   */
  protected Version() {
    Properties properties = new Properties();
    InputStream inputStream = null;
    try {
      inputStream = Version.class.getResourceAsStream(VERSION_FILE);
      if (inputStream == null) {
        throw new FileNotFoundException(MessageFormat.format(
          "O arquivo {0} no foi encontrado.", VERSION_FILE));
      }
      properties.load(inputStream);
      this.name = properties.getProperty(PROPERTY_VERSION);
    }
    catch (IOException e) {
      this.name = String.valueOf(System.currentTimeMillis());
      e.printStackTrace();
    }
    finally {
      FileUtils.close(inputStream);
    }
    if (!splitVersionId()) {
      version = null;
      releaseDate = null;
    }
  }

  /**
   * Construtor privado para testes.
   * 
   * @param prjVersionID - identificador da verso, no formato
   *        PRJ_vXX_YY_ZZ_YYYY_MM_DD
   */
  protected Version(String prjVersionID) {
    this.name = prjVersionID;
    if (!splitVersionId()) {
      version = null;
      releaseDate = null;
    }
  }

  /**
   * Processa o identificador da verso no formato PRJ_vX_YY_ZZ_YYYY_MM_DD,
   * extraindo o nmero da verso (X.YY.ZZ) e a data de lanamento no formato
   * DD/MM/YYYY.
   * 
   * @return true se os campos foram obtidos com sucesso
   */
  protected boolean splitVersionId() {
    if (name == null) {
      return false;
    }

      //PRJ_vX_YY_ZZ_YYYY_MM_DD


    Matcher matcher = Pattern.compile(VERSION_ID_REGEX).matcher(name);
    if (!matcher.matches()) {
      return false;
    }
      projectName = name.substring(0, name.indexOf("_"));

    int i = 1;
    String strMajor = matcher.group(i++);
    String strMinor = matcher.group(i++);
    String strPatch = matcher.group(i++);
    if (strMajor == null || strMinor == null || strPatch == null) {
      return false;
    }
    major = Integer.parseInt(strMajor);
    minor = Integer.parseInt(strMinor);
    patch = Integer.parseInt(strPatch);
    version = String.format(VERSION_ID_FORMAT, major, minor, patch);

    String strYear = matcher.group(i++);
    String strMonth = matcher.group(i++);
    String strDay = matcher.group(i++);
    if (strYear == null || strMonth == null || strDay == null) {
      return false;
    }
    day = Integer.parseInt(strDay);
    month = Integer.parseInt(strMonth);
    year = Integer.parseInt(strYear);
    releaseDate = String.format(VERSION_DATE_FORMAT, day, month, year);

    valid = true;
    return true;
  }

  /**
   * Obtm a instncia nica da verso no sistema.
   * 
   * @return A instncia nica da verso.
   */
  public static Version getInstance() {
    if (Version.instance == null) {
      Version.instance = new Version();
    }
    return Version.instance;
  }

  /**
   * Obtm o nome da verso.
   * 
   * @return O nome da verso.
   */
  public String getName() {
    return this.name;
  }

  /**
   * Obtm o nome do projeto
   * 
   * @return O nome do projeto.
   */
  public String getProjectName() {
    return projectName;
  }

  /**
   * Indica se a verso est com o formato vlido.
   * 
   * @return boolean
   */
  public boolean isValid() {
    return valid;
  }

    /**
   * Obtm o nmero da verso no formato (major.minor.patch).
   * 
   * @return O nome da verso.
   */
  final public String getVersion() {
    return version == null ? BAD_VERSION_STR : version;
  }

  /**
   * Retorna o nmero da <i>major version</i>.
   * 
   * @return nmero da major version, ou -1 caso a verso no tenha sido
   *         calculada
   */
  final public int getMajorVersion() {
    return version == null ? -1 : major;
  }

  /**
   * Retorna o nmero da <i>minor version</i>.
   * 
   * @return nmero da patch version, ou -1 caso a verso no tenha sido
   *         calculada
   */
  final public int getMinorVersion() {
    return version == null ? -1 : minor;
  }

  /**
   * Retorna o nmero da <i>patch version</i>.
   * 
   * @return nmero da patch version, ou -1 caso a verso no tenha sido
   *         calculada
   */
  final public int getPatchVersion() {
    return version == null ? -1 : patch;
  }

  /**
   * Obtm a data da verso no formato (dia/ms/ano).
   * 
   * @return a data.
   */
  final public String getReleaseDate() {
    return releaseDate == null ? NO_DATE_STR : releaseDate;
  }

  /**
   * Retorna o dia de lanamento da verso.
   * 
   * @return dia de lanamento da verso
   */
  final public int getReleaseDay() {
    return releaseDate == null ? -1 : day;
  }

  /**
   * Retorna o ms de lanamento da verso.
   * 
   * @return ms de lanamento da verso
   */
  final public int getReleaseMonth() {
    return releaseDate == null ? -1 : month;
  }

  /**
   * Retorna o ano de lanamento da verso.
   * 
   * @return ano de lanamento da verso
   */
  final public int getReleaseYear() {
    return releaseDate == null ? -1 : year;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!this.getClass().equals(obj.getClass())) {
      return false;
    }
    return compareTo((Version) obj) == 0;
  }

  /**
   * Indica se esta verso antecede outra.
   * 
   * @param other - verso para comparao
   * @return true se esta verso  anterior  outra. Equivale a
   *         <code>compareTo(other) == -1</code>.
   */
  public boolean precedes(Version other) {
    return compareTo(other) == -1;
  }

  /**
   * Indica se esta verso sucede outra.
   * 
   * @param other - verso para comparao
   * @return true se esta verso  posterior  outra. Equivale a
   *         <code>compareTo(other) == 1</code>.
   */
  public boolean succeeds(Version other) {
    return compareTo(other) == 1;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return this.name.hashCode();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.name;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(Version o) {
    /*
     * testamos se a verso foi obtida com sucesso. OBS.: assumimos que, se
     * version != null ento releaseDate != null, e vice-versa
     */
    if (version == null) {
      /*
       * se o objeto corrente no possui verso, considera-o igual ao outro se
       * este tambm no possui, seno considera-o por default _maior_ do que o
       * outro (desta forma o trunk sempre ser associado  ltima verso)
       */
      return o.version == null ? 0 : 1;
    }
    else if (o.version == null) {
      /*
       * analogamente, se este possui verso mas o outro no, considera esta
       * verso menor do que a outra
       */
      return -1;
    }

    /*
     * ok, ambos possuem verso, vamos compar-las. O critrio utilizado  o
     * tradicional: a ordem de prioridade  major > minor > patch i.e. s
     * precisamos testar o prximo nvel se o nvel atual  igual
     */
    if (major == o.major) {
      if (minor == o.minor) {
        if (patch == o.patch) {
          /*
           * tudo igual at agora, resta comparar as datas
           */
          return compareDate(o);
        }
        else {
          /*
           * major e minor iguais, mas patch  diferente
           */
          return patch > o.patch ? 1 : -1;
        }
      }
      else {
        /*
         * major igual, mas minor  diferente
         */
        return minor > o.minor ? 1 : -1;
      }
    }
    else {
      /*
       * major  diferente
       */
      return major > o.major ? 1 : -1;
    }
  }

  /**
   * Compara as datas de duas verses. Assume que existe uma data e que esta 
   * vlida.
   * 
   * @param o - objeto a ser comparado
   * @return -1 se a data desta verso  anterior  da outra, 0 se so iguais ou
   *         1 se a data desta verso  mais recente do que a da outra
   */
  public int compareDate(Version o) {
    if (year == o.year) {
      if (month == o.month) {
        if (day == o.day) {
          return 0;
        }
        else {
          return day > o.day ? 1 : -1;
        }
      }
      else {
        return month > o.month ? 1 : -1;
      }
    }
    else {
      return year > o.year ? 1 : -1;
    }
  }
}
