package csbase.logic.algorithms;

import java.io.Serializable;

import tecgraf.javautils.core.io.FileUtils;
import csbase.logic.ProjectFileType;

/**
 * Classe que representa um arquivo abstrato.
 */
public final class FileParameterValue implements
  Comparable<FileParameterValue>, Serializable, Cloneable {
  /**
   * Separador de arquivo utilizado nos caminhos dos arquivos.
   */
  private static final String FILE_SEPARATOR = "/";

  /** Caminho abstrato para o arquivo. */
  private String path;
  /** Tipo do arquivo. */
  private String type;

  /**
   * Cria um arquivo do tipo {@link ProjectFileType#UNKNOWN desconhecido} dado o
   * seu caminho.
   * 
   * @param path O caminho (No pode ser {@code null}).
   */
  public FileParameterValue(String path) {
    this(path, ProjectFileType.UNKNOWN);
  }

  /**
   * Cria um arquivo do tipo dado o seu caminho e o seu tipo.
   * 
   * @param path O caminho (No pode ser {@code null}).
   * @param type O tipo ({@code null} para tipo desconhecido).
   */
  public FileParameterValue(String path, String type) {
    setPath(path);
    setType(type);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(FileParameterValue file) {
    return getPath().compareToIgnoreCase(file.getPath());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    FileParameterValue file = (FileParameterValue) obj;
    if (!getPath().equals(file.getPath())) {
      return false;
    }
    return getType().equals(file.getType());
  }

  /**
   * Obtm o caminho para o arquivo.
   * 
   * @return O caminho.
   */
  public String getPath() {
    return path;
  }

  /**
   * Obtm o caminho para o arquivo com o separador especificado.
   * 
   * @param fileSeparator o separador a ser usado.
   * 
   * @return O caminho.
   */
  public String getPath(char fileSeparator) {
    return path.replace(FILE_SEPARATOR, String.valueOf(fileSeparator));
  }

  /**
   * Obtm um array representando o caminho para o arquivo.
   * 
   * @return um array representando o caminho para o arquivo.
   */
  public String[] getPathAsArray() {
    return FileUtils.splitPath(path, FILE_SEPARATOR);
  }

  /**
   * Obtm o tipo do arquivo.
   * 
   * @return O tipo.
   */
  public String getType() {
    return type;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return getPath().hashCode();
  }

  /**
   * Indica se  um diretrio.
   * 
   * @return .
   */
  public boolean isDirectory() {
    return getType().equals(ProjectFileType.DIRECTORY_TYPE);
  }

  /**
   * Indica se  um arquivo regular.
   * 
   * @return .
   */
  public boolean isRegularFile() {
    return !isDirectory();
  }

  /**
   * Indica se o tipo  conhecido.
   * 
   * @return .
   */
  public boolean isTypeKnown() {
    return getType().equals(ProjectFileType.UNKNOWN);
  }

  /**
   * Permite renomear o arquivo.
   * 
   * @param path String que representa o caminho do arquivo.
   */
  public void setPath(final String path) {
    if (path == null) {
      throw new IllegalArgumentException("O parmetro path est nulo.");
    }

    final String auxPath;
    if (path.length() == 0) {
      auxPath = ".";
    }
    else {
      /*
       * Substitui, o separador do windows - \ - pelo separador universal - /.
       * Desta forma, o caminho passa a ser independente do sistema operacional.
       */
      auxPath = path.replaceAll("\\\\", FILE_SEPARATOR);
    }
    this.path = auxPath;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    if (path.equals(".")) {
      return FILE_SEPARATOR;
    }
    return FILE_SEPARATOR + path;
  }

  /**
   * Atribui o tipo a este arquivo.
   * 
   * @param newType o tipo ({@code null} para tipo desconhecido).
   */
  private void setType(final String newType) {
    String auxType = newType;
    if (newType == null || newType.trim().isEmpty()) {
      auxType = ProjectFileType.UNKNOWN;
    }
    this.type = auxType;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public FileParameterValue clone() {
    return new FileParameterValue(path, type);
  }
}
