/*
 * $Author$ $Date$ $Revision$
 */
package csbase.logic.algorithms.parameters;

import java.io.Serializable;

/**
 * <p>
 * Item de enumerao
 * </p>
 * 
 * <p>
 * O item de enumerao  o elemento do parmetro enumerao (
 * {@link EnumerationParameter}). Um item de enumerao possui um identificador,
 * um rtulo e um valor. O identificador  utilizado para criar relacionamentos
 * com outros elementos como gatilhos e enumeraes. O identificador tambm
 * utilizado para verificar a igualdade entre 2 tens de enumerao, ou seja, se
 * 2 tens de enumerao possuem o mesmo identificador eles so considerados
 * iguais. O rtulo  texto que  exibido ao usurio. O valor  o texto que 
 * utilizado na linha de comando.
 * </p>
 * 
 * <p>
 * Um item de enumerao pode ser habilitado/desabilitado.
 * </p>
 * 
 * @author lmoreira
 */
public final class EnumerationItem implements Comparable<EnumerationItem>,
  Serializable {
  /**
   * A descrio deste item (Pode no existir).
   */
  private String description;

  /**
   * Indica se um item de enumerao est visvel ({@code true}) ou oculto (
   * {@code false}).
   */
  private boolean isVisible;

  /**
   * O identificador deste item.
   */
  private String id;

  /**
   * O rtulo deste item.
   */
  private String label;

  /**
   * O valor deste item.
   */
  private String value;

  /**
   * Cria um item de enumerao.
   * 
   * @param id O identificador do item (No aceita {@code null}).
   * @param name O nome do item (No aceita {@code null}).
   * @param value O valor do item (No aceita {@code null}).
   * @param description A descrio do item ((Aceita {@code null}).
   */
  public EnumerationItem(String id, String name, String value,
    String description) {
    this.description = description;
    setId(id);
    setLabel(name);
    setValue(value);
    setVisible(true);
  }

  /**
   * A ordem natural de um item de enumerao  a ordem crescente dos rtulos.
   * 
   * {@inheritDoc}
   */
  @Override
  public int compareTo(EnumerationItem item) {
    return getLabel().compareTo(item.getLabel());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object o) {
    if (o == null) {
      return false;
    }
    if (!getClass().equals(o.getClass())) {
      return false;
    }
    EnumerationItem item = (EnumerationItem) o;
    return getId().equals(item.getId());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return getId().hashCode();
  }

  /**
   * Obtm a descrio deste item.
   * 
   * @return A descrio ou {@code null} se ela no existir.
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Obtm o identificador deste item.
   * 
   * @return O identificador.
   */
  public String getId() {
    return this.id;
  }

  /**
   * Obtm o rtulo deste item.
   * 
   * @return O rtulo.
   */
  public String getLabel() {
    return this.label;
  }

  /**
   * Obtm o valor deste item.
   * 
   * @return O valor.
   */
  public String getValue() {
    return this.value;
  }

  /**
   * Indica se um item de enumerao est visvel ({@code true}) ou oculto (
   * {@code false}).
   * 
   * @return {@code true} se estiver visvel ou {@code false} se estiver oculto.
   */
  public boolean isVisible() {
    return this.isVisible;
  }

  /**
   * A representao com uma {@link java.lang.String} do item  o seu rtulo.
   * 
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return getLabel();
  }

  /**
   * <p>
   * Exibe/Oculta este item de enumerao.
   * </p>
   * 
   * @param isVisible {@code true} para exibir ou {@code false} para ocultar.
   * 
   * @return {@code true} se mudar o estado ou {@code false} caso contrrio.
   */
  boolean setVisible(boolean isVisible) {
    if (this.isVisible == isVisible) {
      return false;
    }
    this.isVisible = isVisible;
    return true;
  }

  /**
   * Atribui um identificador a este item.
   * 
   * @param id O identificador (No aceita {@code null}).
   */
  private void setId(String id) {
    if (id == null) {
      throw new IllegalArgumentException("O parmetro id est nulo.");
    }
    this.id = id;
  }

  /**
   * Atribui um rtulo a este item.
   * 
   * @param label O rtulo (No aceita {@code null}).
   */
  private void setLabel(String label) {
    if (label == null) {
      throw new IllegalArgumentException("O parmetro label est nulo.");
    }
    this.label = label;
  }

  /**
   * Atribui um valor a este item.
   * 
   * @param value O valor (No aceita {@code null}).
   */
  private void setValue(String value) {
    if (value == null) {
      throw new IllegalArgumentException("O parmetro value est nulo.");
    }
    this.value = value;
  }
}
