/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import tecgraf.javautils.parsers.exception.AutomatonException;
import tecgraf.openbus.algorithmservice.v1_0.parameters.TableParameterHelper;
import csbase.exception.ParseException;
import csbase.logic.algorithms.CommandLineContext;
import csbase.logic.algorithms.parameters.validators.TableParameterValidator;

/**
 * Parmetro do Tipo Tabela.
 * 
 * @author lmoreira
 */
public final class TableParameter extends SimpleParameter<List<RowValue>>
  implements RowValueListener {
  /**
   * As colunas.
   */
  private List<Column<?>> columns;

  /**
   * Indica se deve delimitar as linhas da tabela na linha de comando gerada.
   */
  private boolean delimitRows;

  /**
   * Indica se deve delimitar a tabela na linha de comando gerada.
   */
  private boolean delimitTable;

  /**
   * A quantidade de linhas da tabela, se esta tabela tiver quantidade de linhas
   * fixa.
   */
  private Integer fixedRowCount;

  /**
   * Os observadores desta tabela.
   */
  private transient List<TableParameterListener> listeners;

  /**
   * As linhas.
   */
  private List<RowValue> rowValues;

  /**
   * Indica o nmero de linhas da tabela que devem estar visveis (no caso do
   * nmero de linhas ser varivel).
   */
  private Integer visibleRowCount;

  /**
   * Indica o nmero mximo de linhas da tabela (no caso do nmero de linhas ser
   * varivel).
   */
  private Integer maxRowCount;

  /**
   * Indica o nmero mnimo de linhas da tabela (no caso do nmero de linhas ser
   * varivel).
   */
  private Integer minRowCount;

  /**
   * Cria um parmetro do tipo tabela.
   * 
   * @param name O nome (No aceita {@code null}).
   * @param label O rtulo (No aceita {@code null}).
   * @param description A descrio (No aceita {@code null}).
   * @param isOptional Indica se o parmetro  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param columns As colunas (No aceita {@code null} e no pode estar vazia).
   * @param delimitTable Indica se deve delimitar a tabela na linha de comando a
   *        ser gerada.
   * @param delimitRows Indica se deve delimitar as linhas da tabela na linha de
   *        comando a ser gerada.
   * @param fixedRowCount A quantidade de linhas da tabela (Aceita {@code null}
   *        - tem que ser um nmero positivo).
   * @param visibleRowCount O nmero de linhas da tabela que devem estar
   *        visveis.
   * @param maxRowCount nmero mximo de linhas.
   * @param minRowCount nmero mnimo de linhas.
   */
  public TableParameter(String name, String label, String description,
    boolean isOptional, boolean isVisible, String commandLinePattern,
    List<? extends Column<?>> columns, boolean delimitTable,
    boolean delimitRows, Integer fixedRowCount, Integer visibleRowCount,
    Integer minRowCount, Integer maxRowCount) {
    super(name, label, description, null, isOptional, isVisible,
      commandLinePattern);
    setColumns(columns);

    testIfPositive(visibleRowCount, "Nmero de linha visveis");
    testIfPositive(fixedRowCount, "Nmero de linhas (fixo)");
    testIfPositive(maxRowCount, "Nmero mximo de linhas");
    testIfPositive(minRowCount, "Nmero mnimo de linhas (fixo)");

    if (maxRowCount != null && minRowCount != null) {
      if (maxRowCount < minRowCount) {
        final String fmt =
          "Valores inconsistentes de min/max de linhas (%d e %d)";
        final String err = String.format(fmt, minRowCount, maxRowCount);
        throw new IllegalArgumentException(err);
      }
    }
    this.visibleRowCount = visibleRowCount;
    this.fixedRowCount = fixedRowCount;
    this.maxRowCount = maxRowCount;
    this.minRowCount = minRowCount;

    this.delimitTable = delimitTable;
    this.delimitRows = delimitRows;
    this.listeners = new LinkedList<TableParameterListener>();
    this.rowValues = new ArrayList<RowValue>();
    setValue(this.rowValues);
    createDefaultRows();
  }

  /**
   * Testa se o valor de uma entidade  positiva.
   * 
   * @param value o valor
   * @param name o nome da entidade
   */
  private void testIfPositive(Integer value, String name) {
    if (value != null) {
      if (value <= 0) {
        final String fmt = "%s no positivo. Valor = %d";
        final String err = String.format(fmt, name, value);
        throw new IllegalArgumentException(err);
      }
    }
  }

  /**
   * Adiciona um observador.
   * 
   * @param listener O observador (No aceita {@code null}).
   */
  public void addTableParameterListener(TableParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * Adio de linha
   * 
   * @param rowValue a linha
   */
  public void addRow(RowValue rowValue) {
    if (rowValue == null) {
      throw new IllegalArgumentException("O parmetro rowValue est nulo.");
    }
    rowValue.addRowValueListener(this);
    int rowIndex = this.rowValues.size();
    this.rowValues.add(rowValue);
    fireRowWasCreated(rowValue, rowIndex);
    fireValueWasChangedEvent();
  }

  /**
   * Cria uma linha.
   */
  public void createRow() {
    int rowIndex = this.rowValues.size();
    Object[] row = new Object[getColumnCount()];
    for (int columnIndex = 0; columnIndex < getColumnCount(); columnIndex++) {
      Column<?> column = this.columns.get(columnIndex);
      row[columnIndex] = column.getDefaultValue(rowIndex);
    }
    RowValue rowValue = new RowValue(row);
    addRow(rowValue);
  }

  /**
   * Indica se a quantidade de linhas  fixa.
   * 
   * @return indicativo
   */
  public boolean hasFixedRowCount() {
    return (this.fixedRowCount != null);
  }

  /**
   * Remove uma linha especfica.
   * 
   * @param rowIndex O ndice da linha (De 0 a quantidade de linhas - 1).
   */
  public void removeRow(int rowIndex) {
    if (rowIndex < 0 || rowIndex >= getRowCount()) {
      String message =
        MessageFormat.format(
          "O ndice da linha fornecido - {0} - fora da faixa suportada "
            + "(0 a {1}).", new Object[] { new Integer(rowIndex),
              new Integer(getRowCount()) });
      throw new IllegalArgumentException(message);
    }
    RowValue rowValue = this.rowValues.get(rowIndex);
    rowValue.removeRowValueListener(this);
    this.rowValues.remove(rowIndex);
    fireRowWasRemoved(rowValue, rowIndex);
    fireValueWasChangedEvent();
  }

  /**
   * Indica se deve delimitar as linhas da tabela na linha de comando gerada.
   * 
   * @return indicativo.
   */
  public boolean delimitRows() {
    return this.delimitRows;
  }

  /**
   * Indica se deve delimitar a tabela na linha de comando gerada.
   * 
   * @return indicativo
   */
  public boolean delimitTable() {
    return this.delimitTable;
  }

  /**
   * Obtm a quantidade de colunas.
   * 
   * @return a quantidade
   */
  public int getColumnCount() {
    return this.columns.size();
  }

  /**
   * Obtm a quantidade de linhas que deve estar visvel.
   * 
   * @return a quantidade.
   */
  public Integer getVisibleRowCount() {
    return visibleRowCount;
  }

  /**
   * Obtm a quantidade mxima de linhas.
   * 
   * @return a quantidade.
   */
  public Integer getMaxRowCount() {
    return maxRowCount;
  }

  /**
   * Obtm a quantidade mnima de linhas.
   * 
   * @return a quantidade.
   */
  public Integer getMinRowCount() {
    return minRowCount;
  }

  /**
   * <p>
   * Obtm a lista de colunas.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * @return .
   */
  public List<Column<?>> getColumns() {
    return Collections.unmodifiableList(this.columns);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  @Override
  public String getCommandValue(CommandLineContext context) {
    if (getRowCount() == 0) {
      return null;
    }
    if (getColumnCount() == 0) {
      return null;
    }
    String commandValue = "";
    if (delimitTable()) {
      commandValue += "{";
    }
    String rowSeparator = "";
    for (int i = 0; i < getRowCount(); i++) {
      commandValue += rowSeparator;
      if (delimitRows()) {
        commandValue += "{";
      }
      String cellSeparator = "";
      for (int j = 0; j < this.columns.size(); j++) {
        commandValue += cellSeparator;
        Object itemValue = getItemValue(i, j);
        Column column = this.columns.get(j);
        String itemCommandValue = column.getCommandValue(itemValue);
        commandValue += itemCommandValue;
        cellSeparator = ",";
      }
      if (delimitRows()) {
        commandValue += "}";
      }
      rowSeparator = ",";
    }
    if (delimitTable()) {
      commandValue += "}";
    }
    return commandValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return getName();
  }

  /**
   * Obtm o valor de uma clula.
   * 
   * @param rowIndex O ndice da linha (De 0 a quantidade de linhas - 1).
   * @param columnIndex O ndice da coluna (De 0 a quantidade de colunas - 1).
   * 
   * @return O valor ou {@code null} se a clula no estiver preenchida.
   */
  public Object getItemValue(int rowIndex, int columnIndex) {
    if (getRowCount() == 0) {
      String message =
        MessageFormat.format(
          "No  possvel obter o valor da clula ({0};{1}), "
            + "pois a tabela no possui linhas.", new Object[] {
              new Integer(rowIndex), new Integer(columnIndex) });
      throw new IllegalArgumentException(message);
    }
    if (getColumnCount() == 0) {
      String message =
        MessageFormat.format(
          "No  possvel obter o valor da clula ({0};{1}), "
            + "pois a tabela no possui colunas.", new Object[] {
              new Integer(rowIndex), new Integer(columnIndex) });
      throw new IllegalArgumentException(message);
    }
    if (rowIndex < 0 || rowIndex >= getRowCount()) {
      throw new IllegalArgumentException(MessageFormat.format(
        "O parmetro rowIndex est fora da faixa esperada.\n"
          + "Faixa esperada: 0 a {0}.\nndice da linha (rowIndex): {1}.",
        new Object[] { new Integer(getRowCount()), new Integer(rowIndex) }));
    }
    if (columnIndex < 0 || columnIndex >= getColumnCount()) {
      throw new IllegalArgumentException(MessageFormat
        .format("O parmetro columnIndex est fora da faixa esperada.\n"
          + "Faixa esperada: 0 a {0}.\nndice da coluna (columnIndex): {1}.",
          new Object[] { new Integer(getColumnCount()),
              new Integer(columnIndex) }));
    }
    RowValue rowValue = this.rowValues.get(rowIndex);
    return rowValue.getCellValue(columnIndex);
  }

  /**
   * Obtm a quantidade de linhas da tabela.
   * 
   * @return A quantidade de linhas.
   */
  public int getRowCount() {
    return this.rowValues.size();
  }

  /**
   * <p>
   * Obtm a linha de valores das linhas.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * @return A lista (Estar vazia se no houver linhas).
   */
  public List<RowValue> getRowValues() {
    return Collections.unmodifiableList(this.rowValues);
  }

  /**
   * <p>
   * Obtm os observadores.
   * </p>
   * 
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   * 
   * @return Os observador (A lista retornada est vazia se no houver
   *         observadores).
   */
  public List<TableParameterListener> getTableParameterListeners() {
    return Collections.unmodifiableList(this.listeners);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return "TABLE";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return TableParameterHelper.id();
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  @Override
  public String getValueAsText() {
    if (getRowCount() == 0) {
      return null;
    }
    if (getColumnCount() == 0) {
      return null;
    }
    String text = "";
    text += TableParameterParser.START_CHARACTER;
    String rowSeparator = "";
    for (int i = 0; i < getRowCount(); i++) {
      text += rowSeparator;
      text += TableParameterParser.START_CHARACTER;
      String cellSeparator = "";
      for (int j = 0; j < this.columns.size(); j++) {
        Object itemValue = getItemValue(i, j);
        Column column = this.columns.get(j);
        String itemText = column.getItemValueAsText(itemValue);
        if (itemText == null) {
          itemText = "";
        }
        text += cellSeparator;
        text += itemText;
        cellSeparator =
          Character.toString(TableParameterParser.SEPARATOR_CHARACTER);
      }
      text += TableParameterParser.END_CHARACTER;
      rowSeparator =
        Character.toString(TableParameterParser.SEPARATOR_CHARACTER);
    }
    text += TableParameterParser.END_CHARACTER;
    return text;
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  @Override
  public Map<String, Object> exportValue() {
    List<Object[]> newRowValues = new ArrayList<Object[]>(getRowCount());
    for (int i = 0; i < this.rowValues.size(); i++) {
      RowValue rowValue = this.rowValues.get(i);
      List<Object> cellValueList = rowValue.getCellValues();
      Object[] cellValues = new Object[cellValueList.size()];
      for (int j = 0; j < cellValues.length; j++) {
        Column column = this.columns.get(j);
        Object cellValue = cellValueList.get(j);
        cellValues[j] = column.getValueToExport(cellValue);
      }
      newRowValues.add(cellValues);
    }
    Map<String, Object> parameterValue = new HashMap<String, Object>();
    parameterValue.put(getName(), newRowValues);
    return Collections.unmodifiableMap(parameterValue);
  }

  /**
   * {@inheritDoc}
   */
  @SuppressWarnings("unchecked")
  @Override
  public void importValue(Map<String, Object> parameterValues) {
    if (parameterValues == null) {
      final String err = "O parmetro parameterValues est nulo.";
      throw new IllegalArgumentException(err);
    }
    clearRows();
    List<Object[]> values = (List<Object[]>) parameterValues.get(getName());
    if (values != null) {
      for (int i = 0; i < values.size(); i++) {
        createRow();
      }
      for (int i = 0; i < values.size(); i++) {
        Object[] cellValues = values.get(i);
        for (int j = 0; j < cellValues.length; j++) {
          Column<?> column = this.columns.get(j);
          setItemValue(column.getValueToImport(cellValues[j]), i, j);
        }
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetValue() {
    clearRows();
    createDefaultRows();
  }

  /**
   * <p>
   * Atribui um valor a uma clula.
   * </p>
   * 
   * <p>
   * Gera o evento {@link ParameterListener#valueWasChanged(Parameter)}.
   * </p>
   * 
   * @param value O valor (Aceita {@code null}).
   * @param rowIndex O ndice da linha (De 0 a quantidade de linhas - 1).
   * @param columnIndex O ndice da coluna (De 0 a quantidade de colunas - 1).
   * 
   * @return {@code true} em caso de sucesso ou {@code false} se o valor
   *         informado for igual ao valor armazenado na clula.
   */
  public boolean setItemValue(Object value, int rowIndex, int columnIndex) {
    if (getRowCount() == 0) {
      String message =
        MessageFormat.format(
          "No  possvel atribuir o valor {0}  clula ({1};{2}), "
            + "pois a tabela no possui linhas.", new Object[] { value,
              new Integer(rowIndex), new Integer(columnIndex) });
      throw new IllegalArgumentException(message);
    }
    if (getColumnCount() == 0) {
      String message =
        MessageFormat.format(
          "No  possvel atribuir o valor {0}  clula ({1};{2}), "
            + "pois a tabela no possui colunas.", new Object[] { value,
              new Integer(rowIndex), new Integer(columnIndex) });
      throw new IllegalArgumentException(message);
    }
    if (rowIndex < 0 || rowIndex >= getRowCount()) {
      throw new IllegalArgumentException(MessageFormat.format(
        "O parmetro rowIndex est fora da faixa esperada.\n"
          + "Faixa esperada: 0 a {0}.\nndice da linha (rowIndex): {1}.",
        new Object[] { new Integer(getRowCount()), new Integer(rowIndex) }));
    }
    if (columnIndex < 0 || columnIndex >= getColumnCount()) {
      throw new IllegalArgumentException(MessageFormat
        .format("O parmetro columnIndex est fora da faixa esperada.\n"
          + "Faixa esperada: 0 a {0}.\nndice da coluna (columnIndex): {1}.",
          new Object[] { new Integer(getColumnCount()),
              new Integer(columnIndex) }));
    }
    RowValue rowValue = this.rowValues.get(rowIndex);
    Object oldValue = rowValue.getCellValue(columnIndex);
    if (oldValue == null && value == null) {
      return false;
    }
    if (oldValue != null && oldValue.equals(value)) {
      return false;
    }
    rowValue.setCellValue(value, columnIndex);
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) throws ParseException {
    if (parameterValue != null) {
      TableParameterParser parser = new TableParameterParser();
      List<?> table;
      try {
        table = parser.parseText(parameterValue);
      }
      catch (AutomatonException e) {
        throw new ParseException("No  possivel atribuir o valor\n({0}),"
          + "j que ele no representa uma codificao vlida para tabelas.\n"
          + "Parmetro envolvido: {1}.", new Object[] { parameterValue, this });
      }
      for (int i = 0; i < table.size(); i++) {
        List<?> row = (List<?>) table.get(i);
        if (row.size() != getColumnCount()) {
          throw new ParseException("No  possivel atribuir o valor\n({0}),"
            + "j que ele representa uma tabela com {2} colunas, "
            + "porm esta tabela {1} possui {3} colunas.\n"
            + "Parmetro envolvido: {1}.", new Object[] { parameterValue, this,
              new Integer(row.size()), new Integer(getColumnCount()), });
        }
      }
      Object[][] newValues = new Object[table.size()][getColumnCount()];
      for (int i = 0; i < table.size(); i++) {
        List<?> row = (List<?>) table.get(i);
        for (int j = 0; j < row.size(); j++) {
          Column<?> column = getColumns().get(j);
          String itemValueAsText = (String) row.get(j);
          Object itemValue = column.getItemValueFromText(itemValueAsText);
          newValues[i][j] = itemValue;
        }
      }
      clearRows();
      for (int i = 0; i < newValues.length; i++) {
        createRow();
      }
      for (int i = 0; i < newValues.length; i++) {
        for (int j = 0; j < newValues[i].length; j++) {
          Object itemValue = newValues[i][j];
          setItemValue(itemValue, i, j);
        }
      }
    }
    else {
      clearRows();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void valueWasChanged(Object newValue, int columnIndex) {
    fireValueWasChangedEvent();
  }

  /**
   * Cria as linhas.
   */
  private void createDefaultRows() {
    if (this.fixedRowCount != null) {
      for (int rowIndex = 0; rowIndex < this.fixedRowCount; rowIndex++) {
        createRow();
      }
    }
  }

  /**
   * Remove as linhas.
   */
  private void clearRows() {
    if (getRowCount() != 0) {
      do {
        removeRow(getRowCount() - 1);
      } while (getRowCount() != 0);
      fireValueWasChangedEvent();
    }
  }

  /**
   * Atribui as colunas a esta tabela.
   * 
   * @param columns As colunas (No aceita {@code null} e no pode estar vazia).
   */
  private void setColumns(List<? extends Column<?>> columns) {
    if (columns == null) {
      throw new IllegalArgumentException("O parmetro columns est nulo.");
    }
    if (columns.isEmpty()) {
      throw new IllegalArgumentException("O parmetro columns est vazio.");
    }
    this.columns = new ArrayList<Column<?>>(columns);
  }

  /**
   * Dispara o evento
   * {@link TableParameterListener#rowWasCreated(TableParameter, RowValue, int)}
   * .
   * 
   * @param rowValue O valor da linha (Aceita {@code null}).
   * @param rowIndex O ndice da linha (De 0 a quantidade de linhas - 1).
   */
  private void fireRowWasCreated(RowValue rowValue, int rowIndex) {
    Iterator<TableParameterListener> listenerIterator =
      this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      TableParameterListener listener = listenerIterator.next();
      listener.rowWasCreated(this, rowValue, rowIndex);
    }
  }

  /**
   * Dispara o evento
   * {@link TableParameterListener#rowWasRemoved(TableParameter, RowValue, int)}
   * .
   * 
   * @param rowValue O valor da linha (Aceita {@code null}).
   * @param rowIndex O ndice da linha (De 0 a quantidade de linhas - 1).
   */
  private void fireRowWasRemoved(RowValue rowValue, int rowIndex) {
    Iterator<TableParameterListener> listenerIterator =
      this.listeners.iterator();
    while (listenerIterator.hasNext()) {
      TableParameterListener listener = listenerIterator.next();
      listener.rowWasRemoved(this, rowValue, rowIndex);
    }
  }

  /**
   * Cria os atributos transientes.
   * 
   * @param in stream com o objeto serializado.
   * 
   * @throws IOException em caso de erro de I/O.
   * @throws ClassNotFoundException se a classe do objeto serializado no for
   *         encontrada.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<TableParameterListener>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableParameterValidator createParameterValidator() {
    return new TableParameterValidator(isOptional(), getColumns());
  }

}
