/**
 * $Id$
 */
package csbase.logic.algorithms.parameters;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.LinkedList;
import java.util.List;

import tecgraf.openbus.algorithmservice.v1_0.parameters.TextParameterHelper;
import csbase.logic.algorithms.parameters.validators.TextParameterValidator;

/**
 * Parmetro do tipo Texto.
 * 
 * @author lmoreira
 */
public final class TextParameter extends SimpleParameter<String> {

  /**
   * Indica se o parmetro  editvel ou no.
   */
  private boolean isEditable;

  /**
   * Os obsevadores.
   */
  private transient List<TextParameterListener> listeners;

  /**
   * O nmero mximo de caracteres.
   */
  private Integer maxCharacters;

  /**
   * Cria um parmetro do tipo texto.
   * 
   * @param name O nome do parmetro (No aceita {@code null}).
   * @param label O rtulo do parmetro (No aceita {@code null}).
   * @param description A descrio do parmetro (No aceita {@code null}).
   * @param defaultValue O valor-padro (Aceita {@code null}).
   * @param isOptional Indica se  opcional.
   * @param isVisible Indica se o parmetro deve ficar visvel.
   * @param commandLinePattern O padro para construo da linha de comando. O
   *        padro ser utilizado para escrever o trecho da linha do comando
   *        referente ao parmetro. Esta string ser passada para o mtodo
   *        MessageFormat.format(String,Object...). O primeiro formato ({0}) 
   *        referente ao nome e o segundo formato ({1})  referente ao valor. Se
   *        {@code null} o parmetro no produzir sada na linha de comando.
   * @param maxCharacters O nmero mximo de caracteres para que o valor de uma
   *        clula seja considerado vlido (Aceita {@code null}).
   */
  public TextParameter(String name, String label, String description,
    String defaultValue, boolean isOptional, boolean isVisible,
    String commandLinePattern, Integer maxCharacters) {
    super(name, label, description, defaultValue, isOptional, isVisible,
      commandLinePattern);
    this.listeners = new LinkedList<TextParameterListener>();
    setMaxCharacters(maxCharacters);
    setEditable(true);
  }

  /**
   * Adiciona um observador a lista de observadores deste parmetro do tipo
   * texto.
   * 
   * @param listener O observador (No aceita {@code null}).
   */
  public void addTextParameterListener(TextParameterListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getExpressionValue() {
    return null;
  }

  /**
   * Obtm o nmero mximo de caracteres para que o valor de uma clula seja
   * considerado vlido.
   * 
   * @return O nmero mximo de caracteres ou {@code null} caso no haja um
   *         nmero mximo de caracteres definido.
   */
  public Integer getMaxCharacters() {
    return this.maxCharacters;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getType() {
    return "TEXT";
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getIDLType() {
    return TextParameterHelper.id();
  }

  /**
   * Indica se o parmetro  editvel ou no.
   * 
   * @return .
   */
  public boolean isEditable() {
    return this.isEditable;
  }

  /**
   * Permite/Proibe a edio do parmetro.
   * 
   * @param isEditable .
   */
  public void setEditable(boolean isEditable) {
    if (this.isEditable != isEditable) {
      this.isEditable = isEditable;
      for (TextParameterListener listener : listeners) {
        listener.editabilityWasChanged(this);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAsText(String parameterValue) {
    setValue(parameterValue);
  }

  /**
   * Atribui o nmero mximo de caracteres para que o valor de uma clula seja
   * considerado vlido.
   * 
   * @param maxCharacters O nmero mximo de caracteres ou {@code null} caso no
   *        haja um nmero mximo de caracteres definido.
   */
  private void setMaxCharacters(Integer maxCharacters) {
    if (maxCharacters != null && maxCharacters.intValue() <= 0) {
      throw new IllegalArgumentException(MessageFormat.format(
        "O parmetro maxCharacters deveria ser positivo.\n"
          + "Valor encontrado: {0}.", new Object[] { maxCharacters }));
    }
    this.maxCharacters = maxCharacters;
  }

  /**
   * Cria os atributos transientes.
   * 
   * @param in Leitor de objetos
   * 
   * @throws IOException em caso de erro na leitura
   * @throws ClassNotFoundException se no encontrar a classe do objeto sendo
   *         lido.
   */
  private void readObject(java.io.ObjectInputStream in) throws IOException,
    ClassNotFoundException {
    in.defaultReadObject();
    listeners = new LinkedList<TextParameterListener>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TextParameterValidator createParameterValidator() {
    return new TextParameterValidator(isOptional(), getMaxCharacters());
  }
}
