/*
 * $Id: ProjectSynchronizationUnit.java 116325 2011-03-18 03:17:58Z cassino $
 */
package csbase.logic;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Modela os dados de um projeto que participar de uma sincronizao.
 */
public class ProjectSynchronizationUnit implements Serializable {
  private String server;
  private String login;
  private String password;
  private Object projectId;
  private Object projectOwnerId;
  private Map<String, ProjectSynchronizationFileInfo> files;

  /**
   * Constri um objeto de dados de um projeto que participar de uma
   * sincronizao. Este objeto contm o nome do servidor, o usurio e senha a
   * serem utilizados na conexo com este servidor e o identificador do usurio
   * administrador deste projeto.<br>
   * O servidor nulo indica que o projeto pertence ao servidor que for chamado
   * para realizar a sincronizao. A lista de arquivos a serem transferidos so
   * criadas vazias. Note que a lista de arquivos  concretizada na forma de um
   * mapeamento entre caminhos e objetos do tipo ProjectSynchronizationFileInfo.
   * 
   * @param server - nome do servidor (pode ser null; neste caso ser
   *        considerado o servidor corrente)
   * @param login - login do usurio
   * @param password - senha do usurio
   * @param projectOwnerId - identificador do usurio dono do projeto
   */
  public ProjectSynchronizationUnit(String server, String login,
    String password, Object projectOwnerId) {
    this.server = server;
    this.login = login;
    this.password = password;
    this.projectOwnerId = projectOwnerId;
    this.files = new HashMap<String, ProjectSynchronizationFileInfo>();
  }

  /**
   * Retorna o nome do servidor.
   * 
   * @return Nome do servidor.
   */
  public String getServer() {
    return server;
  }

  /**
   * Retorna o usurio do servidor.
   * 
   * @return usurio do servidor.
   */
  public String getLogin() {
    return login;
  }

  /**
   * Retorna a senha no servidor.
   * 
   * @return Senha no servidor.
   */
  public String getPassword() {
    return password;
  }

  /**
   * Retorna o identificador do projeto.
   * 
   * @return Identificador do projeto.
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Atribui o identificador do projeto.
   * 
   * @param projectId Identificador do projeto.
   */
  public void setProjectId(Object projectId) {
    this.projectId = projectId;
  }

  /**
   * Retorna o identificador do usurio administrador do projeto.
   * 
   * @return Identificador do usurio administrador do projeto.
   */
  public Object getProjectOwnerId() {
    return projectOwnerId;
  }

  /**
   * Retorna a lista de arquivos. Os arquivos a serem transferidos para o outro
   * projeto que participar da sincronizao devero ter seu atributo
   * <code>transfer</code> ajustado para <code>true</code>. Note que as
   * listas de arquivos so concretizadas na forma de mapeamentos entre caminhos
   * e objetos do tipo ProjectSynchronizationFileInfo.
   * 
   * @return a lista de arquivos.
   */
  public Map<String, ProjectSynchronizationFileInfo> getFiles() {
    return Collections.unmodifiableMap(files);
  }

  /**
   * Altera a lista de arquivos. Os arquivos a serem transferidos para o outro
   * projeto que participar da sincronizao devero ter seu atributo
   * <code>transfer</code> ajustado para <code>true</code>. Note que as
   * listas de arquivos so concretizadas na forma de mapeamentos entre caminhos
   * e objetos do tipo ProjectSynchronizationFileInfo.
   * 
   * @param files lista de arquivos.
   */
  public void setFiles(Map<String, ProjectSynchronizationFileInfo> files) {
    this.files.clear();
    this.files.putAll(files);
  }
}
