package csbase.logic.algorithms.parameters.validators;

import java.io.Serializable;
import java.rmi.RemoteException;

import csbase.logic.algorithms.parameters.SimpleParameter;
import csbase.logic.algorithms.validation.LocalizedMessage;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationContext;
import csbase.logic.algorithms.validation.ValidationError;
import csbase.logic.algorithms.validation.ValidationMode;
import csbase.logic.algorithms.validation.ValidationSuccess;

/**
 * Validador de parmetro.
 * 
 * @param <V> O tipo do valor armazenado no parmetro.
 * 
 * @author lmoreira
 */
public abstract class SimpleParameterValidator<V> implements Serializable {

  /**
   * Indica se o valor do parmetro  opcional/obrigatrio.
   */
  private boolean isOptional;

  /**
   * Modo de validao ({@link ValidationMode#FULL} ou
   * {@link ValidationMode#ALLOW_EMPY_VALUES}).
   */
  private ValidationMode mode;

  /**
   * Cria o validador.
   * 
   * @param isOptional Indica se o valor do parmetro  opcional/obrigatrio.
   */
  protected SimpleParameterValidator(boolean isOptional) {
    this.isOptional = isOptional;
    this.mode = ValidationMode.FULL;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }
    if (!getClass().equals(obj.getClass())) {
      return false;
    }
    SimpleParameterValidator<?> validator = (SimpleParameterValidator<?>) obj;
    return isOptional() == validator.isOptional();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    return new Boolean(isOptional()).hashCode();
  }

  /**
   * Valida o parmetro.
   * 
   * @param parameter O parmetro envolvido (No aceita {@code null}).
   * @param value O valor (Aceita {@code null}).
   * @param context O contexto usado na validao.
   * 
   * @return O resultado da validao.
   * @throws RemoteException em caso de erro na comunicao com servidor.
   */
  public Validation validateValue(SimpleParameter<?> parameter, V value,
    ValidationContext context) throws RemoteException {
    if (parameter == null) {
      throw new IllegalArgumentException("O parmetro parameter est nulo.");
    }
    if (parameter.isEnabled() && parameter.isVisible()) {
      if (value == null) {
        if (isOptional() || getMode().equals(ValidationMode.ALLOW_EMPY_VALUES)) {
          return new ValidationSuccess();
        }
        LocalizedMessage message =
          new LocalizedMessage(SimpleParameterValidator.class, "empty_value",
            new Object[] { parameter.getLabel() });
        return new ValidationError(message);
      }
    }
    return new ValidationSuccess();
  }

  /**
   * Obtm o modo de validao atual.
   * 
   * @return modo de validao ({@link ValidationMode#FULL} ou
   *         {@link ValidationMode#ALLOW_EMPY_VALUES}).
   */
  public final ValidationMode getMode() {
    return mode;
  }

  /**
   * Modifica o modo de validao atual.
   * 
   * @param mode Modo de validao ({@link ValidationMode#FULL} ou
   *        {@link ValidationMode#ALLOW_EMPY_VALUES}).
   */
  public void setMode(ValidationMode mode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro this.mode est nulo.");
    }

    this.mode = mode;
  }

  /**
   * Indica se o valor do parmetro  opcional/obrigatrio.
   * 
   * @return .
   */
  public final boolean isOptional() {
    return this.isOptional;
  }

}
