/*
 * $Id: ProjectFileType.java 111078 2010-10-07 19:18:47Z clinio $
 */
package csbase.logic;

import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import csbase.remote.ClientRemoteLocator;
import csbase.util.StringUtils;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * A classe <code>ProjectFileType</code> mantm todos os tipos de arquivos
 * conhecidos pelo sistema.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectFileType implements Serializable,
  Comparable<ProjectFileType> {

  static {
    RestartManager.getInstance().addListener(new RestartListener() {
      @Override
      public void restart() {
        ProjectFileType.fileTypes = null;
        ProjectFileType.fileTypesToArray = null;
      }
    });
  }

  /**
   * Tipo indicativo de diretrio.
   */
  public static final String DIRECTORY_TYPE = "DIRECTORY_TYPE";

  /**
   * Tipo do arquivo que indica "Arquivo Desconhecido"
   */
  public static final String UNKNOWN = "UNKNOWN";

  /**
   * Guarda as instncias dos tipos de arquivo carregados.
   */
  private static Map<String, ProjectFileType> fileTypes;

  /**
   * Mesmas instancias carregas no fileTypes, porem ordenadas e em array.
   */
  private static ProjectFileType[] fileTypesToArray;

  /**
   * Prottipo que serve para criar a instncia de ProjectFileType apropriada
   * (cliente ou lgica)
   */
  protected static ProjectFileType projectFileTypePrototype =
    new ProjectFileType();

  /**
   * Descrio do tipo
   */
  private String description;

  /**
   * As extenses de um tipo de arquivo
   */
  private List<String> extensions;

  /**
   * MimeType do tipo
   */
  private String mimeType;

  /**
   * Cdigo do tipo
   */
  private String typeCode;

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(final ProjectFileType fileType) {
    if (this.description == null) {
      return -1;
    }
    if (fileType == null) {
      return -1;
    }
    final String desc = fileType.getDescription();
    if (desc == null) {
      return -1;
    }
    return StringUtils.compare(description, desc);
  }

  /**
   * Cria um ProjectFileType. (Ser sobreescrito na classe cliente para criar o
   * objeto apropriado.)
   * 
   * @param info .
   * 
   * @return .
   */
  protected ProjectFileType createProjectFileType(final ProjectFileTypeInfo info) {
    return new ProjectFileType(info);
  }

  /**
   * Obtm o cdigo do tipo.
   * 
   * @return .
   */
  public String getCode() {
    return this.typeCode;
  }

  /**
   * Obtm a descrio do tipo.
   * 
   * @return .
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * Consulta as extenses do arquivo.
   * 
   * @return as extenses
   */
  public List<String> getExtensions() {
    return Collections.unmodifiableList(this.extensions);
  }

  /**
   * Obtm o mimeType
   * 
   * @return .
   */
  public String getMimeType() {
    return this.mimeType;
  }

  /**
   * Atribui a descricao.
   * 
   * @param description .
   */
  public void setDescription(final String description) {
    this.description = description;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.description;
  }

  /**
   * Cria um prottipo do objeto ProjectFileType, para que o mtodo
   * createProjectFileType possa ser chamado de forma apropriada.
   */
  protected ProjectFileType() {
  }

  /**
   * Cria um ProjectFileType
   * 
   * @param info .
   */
  public ProjectFileType(final ProjectFileTypeInfo info) {
    this.typeCode = info.typeCode;
    this.description = info.description;
    this.mimeType = info.mimeType;
    this.extensions = new ArrayList<String>();
    if (info.extensions != null) {
      this.extensions.addAll(info.extensions);
    }
  }

  /**
   * Obtm todos os tipos de arquivos definidos.
   * 
   * @return um array com todos os tipos de arquivos.
   */
  public static ProjectFileType[] getAllFileTypes() {
    return fileTypesToArray;
  }

  /**
   * Obtm um tipo de arquivo que possui um determinado cdigo.
   * 
   * @return o tipo.
   */
  public static ProjectFileType getUnknownFileType() {
    return fileTypes.get(UNKNOWN);
  }

  /**
   * Obtm um tipo de arquivo que possui um determinado cdigo.
   * 
   * @param typeCode o cdigo do tipo.
   * 
   * @return o tipo.
   */
  public static ProjectFileType getFileType(final String typeCode) {
    ProjectFileType type = null;
    /*
     * Se o mapa de fileTypes no foi carregado, obtem do servidor, mas no
     * carrega o mapa local.
     */
    if (fileTypes == null) {
      ProjectFileTypeInfo info;
      try {
        info = ClientRemoteLocator.projectService.getFileType(typeCode);
      }
      catch (final Exception e) {
        return null;
      }
      type =
        (info != null) ? projectFileTypePrototype.createProjectFileType(info)
          : null;
    }
    else {
      type = fileTypes.get(typeCode);
      if (type == null) {
        type = fileTypes.get(UNKNOWN);
      }
    }
    return type;
  }

  /**
   * Consulta o tipo de arquivo sugerido dada uma extenso de arquivo.
   * 
   * @param extension extenso
   * @return tipo
   */
  public static ProjectFileType getProjectFileTypeFromExtension(
    final String extension) {
    if (extension == null) {
      return null;
    }
    for (final ProjectFileType pft : fileTypesToArray) {
      final List<String> exts = pft.getExtensions();
      for (final String ext : exts) {
        if (extension.equalsIgnoreCase(ext)) {
          return pft;
        }
      }
    }
    return null;
  }

  /**
   * Carrega os tipos de arquivos definidos no servidor.
   * 
   * @param locale definio de idioma
   * @throws RemoteException erro de rmi na chamada remota
   */
  public static void loadFileTypes(final Locale locale) throws RemoteException {
    if (fileTypes != null) {
      return;
    }
    fileTypes = new HashMap<String, ProjectFileType>();
    final Map<String, ProjectFileTypeInfo> fileTypesInfo =
      ClientRemoteLocator.projectService.getAllFileTypes(locale);
    fileTypesToArray = new ProjectFileType[fileTypesInfo.size()];
    int i = 0;
    for (final ProjectFileTypeInfo info : fileTypesInfo.values()) {
      final ProjectFileType pft =
        projectFileTypePrototype.createProjectFileType(info);
      fileTypes.put(info.typeCode, pft);
      fileTypesToArray[i++] = pft;
    }

    //Ordena o vetor
    if (fileTypesToArray.length > 0) {
      Arrays.sort(fileTypesToArray);
    }
  }
}
