package csbase.logic.algorithms;

import tecgraf.javautils.core.io.FileUtils;

/**
 * Construtor de linhas de comando simples ou de scripts. Contm mtodos
 * utilitrios para facilitar e padronizar as linhas de comando geradas pelo
 * sistema.
 */
public class CommandLineBuilder {

  /**
   * Caractere utilizado em referncias a variveis de ambiente.
   */
  public static final String REFERENCE_VAR_CHAR = "$";

  /** Caractere separador de comandos. */
  public static final String COMMAND_SEPARATOR = ";";

  /**
   * O separador de linha do script. *Precisa* que seja o "\n". No pode ser o
   * do sistema, para o script conseguir ser rodado pelos sgas de todas as
   * plataformas (at no caso do windows, por causa do cygwin).
   */
  public static final String SCRIPT_LINE_SEPARATOR = "\n";

  /**
   * Shell utilizado para executar a linha de comando
   */
  public static final String SHELL = "/bin/ksh";

  /** Caractere indicador de comentrios. */
  public static final String COMMENT = "# ";

  /**
   * Construtor da linha de comando.
   */
  private final StringBuilder buffer;

  /**
   * Separador de linhas do comando.
   */
  private final String lineSeparator;

  /**
   * Construror.
   * 
   * @param isScript indica se o comando deve ser construido como um script.
   */
  public CommandLineBuilder(boolean isScript) {
    this.buffer = new StringBuilder();
    if (isScript) {
      this.lineSeparator = SCRIPT_LINE_SEPARATOR;
    }
    else {
      this.lineSeparator = " ";
    }
  }

  /**
   * Retorna o comando com escape para o caractere de referncia a variveis de
   * ambiente. Para uso de comandos diretamente no shell,  necessrio "escapar"
   * esse caractere.
   * 
   * @param cmd O comando
   * @return O comando com espaos "escapados"
   */
  public static String escapeReferenceChar(String cmd) {
    if (cmd != null) {
      return cmd.replace(REFERENCE_VAR_CHAR, "\\" + REFERENCE_VAR_CHAR);
    }
    else {
      return null;
    }
  }

  /**
   * Retorna a string que representa uma referncia  varivel de ambiente na
   * linha de comando.
   * 
   * @param var varivel de ambiente ({@link EnvironmentVariable})
   * 
   * @return A string com a referncia  varivel de ambiente.
   */
  public static String makeEnvironmentVariableReference(EnvironmentVariable var) {
    return makeVariableReference(var.getName());
  }

  /**
   * Monta um caminho comeando com a varivel de ambiente especificada.
   * 
   * @param var a varivel de ambiente
   * @param path O caminho
   * @param fileSeparator o separador de arquivos
   * @return O caminho completo, comeando com a varivel de ambiente.
   */
  public static String makePathWithEnvironmentVariable(EnvironmentVariable var,
    String path, char fileSeparator) {
    String varRef = makeEnvironmentVariableReference(var);
    return FileUtils.joinPath(fileSeparator, varRef, path);
  }

  /**
   * Monta o nome de um diretrio de ligao de um n de um fluxo.
   * 
   * @param nodeId o identificador do n origem da ligao.
   * @param paramName o nome do parmetro de sada do n envolvido na ligao.
   * 
   * @return o nome do diretrio.
   */
  public static String makeLinkDirName(Integer nodeId, String paramName) {
    return makeLinkName("dir", nodeId, paramName);
  }

  /**
   * Monta o nome de um named pipe de um n de origem de uma ligao em um
   * fluxo.
   * 
   * @param nodeId o identificador do n origem da ligao.
   * @param paramName o nome do parmetro de sada do n envolvido na ligao.
   * 
   * @return o nome do named pipe.
   */
  public static String makeFromPipeName(Integer nodeId, String paramName) {
    return makeLinkName("from", nodeId, paramName);
  }

  /**
   * Monta o nome de um named pipe de um n de destino de uma ligao em um
   * fluxo.
   * 
   * @param nodeId o identificador do n destino da ligao.
   * @param paramName o nome do parmetro de entrada do n envolvido na ligao.
   * 
   * @return o nome do named pipe.
   */
  public static String makeToPipeName(Integer nodeId, String paramName) {
    return makeLinkName("to", nodeId, paramName);
  }

  /**
   * Monta o nome de uma ligao a ser utilizado na linha de comando de um
   * fluxo. As ligaes podem ser named-pipes de entrada ou de sada ou
   * diretrios.
   * 
   * @param prefix o prefixo que define o tipo de ligao.
   * @param nodeId o identificador do n origem da ligao.
   * @param paramName o nome do parmetro de sada do n envolvido na ligao.
   * 
   * @return o nome da ligao.
   */
  private static String makeLinkName(String prefix, Integer nodeId,
    String paramName) {
    return prefix + "_" + nodeId + "_" + paramName;
  }

  /**
   * Retorna a string que representa uma referncia  varivel de ambiente na
   * linha de comando.
   * 
   * @param var varivel de ambiente ({@link EnvironmentVariable})
   * 
   * @return A string com a referncia  varivel de ambiente.
   */
  public static String makeVariableReference(String var) {
    return REFERENCE_VAR_CHAR + var;
  }

  /**
   * Adiciona uma string ao final da linha de comando.
   * 
   * @param string a string.
   * 
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder append(String string) {
    buffer.append(string);
    return this;
  }

  /**
   * Adiciona uma string ao final da linha de comando seguida do separador de
   * comando. No caso de script, tambm adiciona o separador de linha.
   * 
   * @param command A string.
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendCommand(String command) {
    return append(command + COMMAND_SEPARATOR + lineSeparator);
  }

  /**
   * Adiciona uma string de comentrio ao final da linha de comando.
   * 
   * @param comment A string.
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendComment(String comment) {
    return append(COMMENT + comment + lineSeparator);
  }

  /**
   * Adiciona uma linha de separao de comentrio ao final da linha de comando.
   * 
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendCommentSeparator() {
    return appendComment("---------------------------------------------------");
  }

  /**
   * Adiciona uma linha vazia ao final da linha de comando.
   * 
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */

  public CommandLineBuilder appendEmptyLine() {
    return append(lineSeparator);
  }

  /**
   * Adiciona a declarao de uma varivel de ambiente ao final da linha de
   * comando. Caso o valor da varivel seja nulo ou vazio, nada  adicionado.
   * 
   * @param var A varivel
   * @param varValue O valor da varivel
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendEnvironmentVariableDeclaration(
    EnvironmentVariable var, String varValue) {
    return appendEnvironmentVariableDeclaration(var.getName(), varValue);
  }

  /**
   * Adiciona a declarao de uma varivel de ambiente ao final da linha de
   * comando. Caso o valor da varivel seja nulo ou vazio, nada  adicionado.
   * 
   * @param varName O nome da varivel
   * @param varValue O valor da varivel
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendEnvironmentVariableDeclaration(
    String varName, String varValue) {
    return appendCommand("export " + varName + "=" + varValue);
  }

  /**
   * Adiciona a captura do cdigo de sada de execuo do ltimo comando ao
   * final da linha de comando.
   * 
   * @param exitCodeFilePath Caminho para o arquivo onde ser armazenado o
   *        cdigo de sada do comando
   * @param context O contexto de execuo do comando
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendExitCodeCaptureCommand(
    String exitCodeFilePath, CommandLineContext context) {
    if (exitCodeFilePath != null) {
      char fileSeparator = context.getFileSeparator();
      String exitCodeVarName = "exit_code";
      String exitCodeVarValue = makeVariableReference(exitCodeVarName);
      String exitCodeLogFile =
        makePathWithEnvironmentVariable(EnvironmentVariable.PROJECT_DIR,
          exitCodeFilePath, fileSeparator);

      append(COMMAND_SEPARATOR).appendEmptyLine();
      /*
       * #1 Guarda o valor do cdigo de sada do comando na varivel
       * EXIT_CODE_VAR: (ex.) exit_code=$?
       */
      appendCommand(exitCodeVarName + "=" + REFERENCE_VAR_CHAR + "?");

      /*
       * #2 Salva o valor da varivel EXIT_CODE_VAR num arquivo: (ex.) echo
       * $exit_code >> arquivo
       */
      append("echo " + exitCodeVarValue)
        .redirectStdOutputToFile(exitCodeLogFile).append(COMMAND_SEPARATOR)
        .appendEmptyLine();

      /*
       * #3 Retorna o cdigo de sada capturado. (ex.) return $exit_code
       */
      appendCommand("return " + exitCodeVarValue);
    }
    return this;
  }

  /**
   * Adiciona o cdigo necessrio para fazer o descarte da sada padro do
   * comando ao final da linha de comando.
   * 
   * @return a linha de comando.
   */
  public CommandLineBuilder discardOutput() {
    return redirectStdOutputToFile("/dev/null");
  }

  /**
   * Adiciona o cdigo necessrio para fazer o descarte da sada padro e de
   * erro do ltimo comando ao final da linha de comando.
   * 
   * @return a linha de comando.
   */
  public CommandLineBuilder discardStdErrAndStdOutput() {
    return discardOutput().redirectStdErrToStdOutput();
  }

  /**
   * Adiciona o cdigo necessrio para fazer o redirecionamento da sada padro
   * e de erro do ltimo comando para um arquivo ao final da linha de comando.
   * 
   * @param standardOutputFile O arquivo para onde ser redirecionada as sadas.
   * @return a linha de comando.
   */
  public CommandLineBuilder redirectStdErrAndStdOutputToFile(
    String standardOutputFile) {
    return redirectStdOutputToFile(standardOutputFile)
      .redirectStdErrToStdOutput();
  }

  /**
   * Adiciona o cdigo necessrio para fazer o redirecionamento da sada de erro
   * para a sada padro do ltimo comando ao final da linha de comando.
   * 
   * @return a linha de comando.
   */
  public CommandLineBuilder redirectStdErrToStdOutput() {
    return append(" 2>&1");
  }

  /**
   * Adiciona o cdigo necessrio para fazer o redirecionamento da sada padro
   * do ltimo comando para um arquivo ao final da linha de comando.
   * 
   * @param standardOutputFile O arquivo para onde ser redirecionada a sada.
   * @return a linha de comando.
   */
  public CommandLineBuilder redirectStdOutputToFile(String standardOutputFile) {
    return append(" > " + standardOutputFile);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return buffer.toString();
  }

  /**
   * Adiciona um cabealho com o shell utilizado pelo script ao final da linha
   * de comando.
   * 
   * @param context o contexto de execuo do comando.
   * @return o prprio construtor do comando, para facilitar o encadeamento de
   *         chamadas de mtodos como esse.
   */
  public CommandLineBuilder appendScriptHeader(CommandLineContext context) {
    return append("#! ").append(SHELL).append(lineSeparator);
  }
}