/*
 * $Id:$
 */

package csbase.logic.filetypefinder;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import csbase.logic.ClientFile;
import csbase.logic.ProjectFileType;

/**
 * Buscador (por inferncia) de tipo de arquivo.
 *
 * @author Tecgraf
 */
public class FileTypeFinder {

  /**
   * Hash de tipos.
   */
  final private HashMap<ProjectFileType, FileTypeChecker> hash =
    new HashMap<ProjectFileType, FileTypeChecker>();

  /**
   * Construtor.
   */
  public FileTypeFinder() {
  }

  /**
   * Finaliza a insero de verificadores, travando a hash interna.
   */
  final public void unmodifiableFileTypeFinder() {
    Collections.unmodifiableMap(hash);
  }

  /**
   * Adiciona um verificador para um determinado tipo.
   *
   * @param checker verificador
   */
  final public void addFileChecker(FileTypeChecker checker) {
    ProjectFileType fileType = checker.getFileType();
    if (hash.containsKey(fileType)) {
      throw new IllegalArgumentException("Double type insertion: " + fileType);
    }
    hash.put(fileType, checker);
  }

  /**
   * Pergunta pelo tipo inferido de um arquivo (primeiro encontrado).
   *
   * @param file arquivo.
   * @param charset o charset para leitura do arquivo.
   * @return tipo
   */
  final public ProjectFileType inferType(ClientFile file, Charset charset) {
    Collection<FileTypeChecker> checkers = hash.values();
    for (FileTypeChecker checker : checkers) {
      if (checker.isOfType(file, charset)) {
        ProjectFileType fileType = checker.getFileType();
        return fileType;
      }
    }
    return null;
  }

  /**
   * Consulta verificador de um tipo
   *
   * @param fileType tipo
   * @return verificador
   */
  final public FileTypeChecker getFileTypeChecker(ProjectFileType fileType) {
    if (!hash.containsKey(fileType)) {
      return null;
    }
    FileTypeChecker checker = hash.get(fileType);
    return checker;
  }

  /**
   * Pergunta pelo tipos possveis de um arquivo.
   *
   * @param file arquivo.
   * @param charset o charset para leitura do arquivo.
   * @return lista de tipos.
   */
  final public List<ProjectFileType> inferPossibleTypes(ClientFile file,
    Charset charset) {
    List<ProjectFileType> list = new ArrayList<ProjectFileType>();
    Collection<FileTypeChecker> checkers = hash.values();
    for (FileTypeChecker checker : checkers) {
      if (checker.isOfType(file, charset)) {
        ProjectFileType fileType = checker.getFileType();
        list.add(fileType);
      }
    }
    Collections.unmodifiableList(list);
    return list;
  }
}
