package csbase.logic.algorithms.flows;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import csbase.exception.BugException;
import csbase.exception.ConfigurationException;
import csbase.exception.ParseException;
import csbase.exception.algorithms.AlgorithmNotFoundException;
import csbase.exception.algorithms.ParameterNotFoundException;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.remote.AlgorithmServiceInterface;
import csbase.remote.ClientRemoteLocator;

/**
 * <p>
 * O analisador dos metadados de fluxos de algoritmos.
 * </p>
 *
 * <p>
 * Esta classe prove mtodos para ler e escrever metadados sobre um fluxo de
 * algoritmos.
 * </p>
 *
 * @author lmoreira
 */
public final class FlowAlgorithmParser {
  /**
   * Atributo {@value #ALGORITHM_ATTRIBUTE_NAME} do elemento
   * {@value #ALGORITHM_ELEMENT_NAME}.
   */
  private static final String ALGORITHM_ATTRIBUTE_NAME = "nome";

  /**
   * Stributo {@value #ALGORITHM_ATTRIBUTE_VERSION} do elemento
   * {@value #ALGORITHM_ELEMENT_NAME}.
   */
  private static final String ALGORITHM_ATTRIBUTE_VERSION = "versao";

  /**
   * Elemento {@value #ALGORITHM_ELEMENT_NAME}.
   */
  private static final String ALGORITHM_ELEMENT_NAME = "algoritmo";

  /**
   * O encoding utilizado.
   */
  public static final String DEFAULT_ENCODING = "ISO-8859-1";

  /**
   * Elemento {@value #FLOW_ELEMENT_NAME}.
   */
  private static final String FLOW_ELEMENT_NAME = "fluxo";

  /**
   * Elemento {@value #FLOW_DESCRIPTION_NAME}.
   */
  private static final String FLOW_DESCRIPTION_NAME = "descricao";

  /**
   * Elemento {@value #FLOW_ELEMENT_NAME}.
   */
  private static final String FLOW_ATTRIBUTE_NAME = "nome";

  /**
   * Elemento {@value #INPUT_ELEMENT_NAME}.
   */
  private static final String INPUT_ELEMENT_NAME = "entrada";

  /**
   * Atributo {@value #LINK_ATTRIBUTE_ID} do elemento
   * {@value #LINK_ELEMENT_NAME}.
   */
  private static final String LINK_ATTRIBUTE_ID = "id";

  /**
   * Elemento {@value #LINK_ELEMENT_NAME}.
   */
  private static final String LINK_ELEMENT_NAME = "conexao";

  /**
   * Atributo {@value #LINK_PARAMETER_ATTRIBUTE_LABEL} do elemento
   * {@value #INPUT_ELEMENT_NAME} ou {@value #OUTPUT_ELEMENT_NAME}.
   */
  private static final String LINK_PARAMETER_ATTRIBUTE_LABEL = "rotulo";

  /**
   * Atributo {@value #LINK_PARAMETER_ATTRIBUTE_NAME} do elemento
   * {@value #INPUT_ELEMENT_NAME} ou {@value #OUTPUT_ELEMENT_NAME}.
   */
  private static final String LINK_PARAMETER_ATTRIBUTE_NAME = "parametro";

  /**
   * Atributo {@value #LINK_PARAMETER_ATTRIBUTE_NODE_ID} do elemento
   * {@value #INPUT_ELEMENT_NAME} ou {@value #OUTPUT_ELEMENT_NAME}.
   */
  private static final String LINK_PARAMETER_ATTRIBUTE_NODE_ID = "no";

  /**
   * Atributo {@value #NODE_ATTRIBUTE_ID} do elemento
   * {@value #NODE_ELEMENT_NAME}.
   */
  private static final String NODE_ATTRIBUTE_ID = "id";

  /**
   * Elemento {@value #NODE_ELEMENT_NAME}.
   */
  private static final String NODE_ELEMENT_NAME = "no";

  /**
   * Atributo {@value #NODE_ATTRIBUTE_ID} do elemento
   * {@value #NODE_PARAMETER_ELEMENT_NAME}.
   */
  private static final String NODE_PARAMETER_ATTRIBUTE_LABEL = "rotulo";

  /**
   * Atributo {@value #NODE_ATTRIBUTE_ID} do elemento
   * {@value #NODE_PARAMETER_ELEMENT_NAME}.
   */
  private static final String NODE_PARAMETER_ATTRIBUTE_NAME = "nome";

  /**
   * Atributo {@value #NODE_ATTRIBUTE_ID} do elemento
   * {@value #NODE_PARAMETER_ELEMENT_NAME}.
   */
  private static final String NODE_PARAMETER_ATTRIBUTE_TYPE = "tipo";

  /**
   * Atributo {@value #NODE_ATTRIBUTE_ID} do elemento
   * {@value #NODE_PARAMETER_ELEMENT_NAME}.
   */
  private static final String NODE_PARAMETER_ATTRIBUTE_VALUE = "valor";

  /**
   * Elemento {@value #NODE_PARAMETER_ELEMENT_NAME}.
   */
  private static final String NODE_PARAMETER_ELEMENT_NAME = "parametro";

  /**
   * Elemento {@value #NODE_PARAMETER_VALUE_ELEMENT_NAME}.
   */
  private static final String NODE_PARAMETER_VALUE_ELEMENT_NAME = "valor";

  /**
   * Elemento {@value #NODE_STDOUT_ELEMENT_NAME}.
   */
  private static final String NODE_STDOUT_ELEMENT_NAME = "stdout";

  /**
   * Atributo {@value #NODE_STDOUT_ATTRIBUTE_TYPE} do elemento
   * {@value #NODE_STDOUT_ELEMENT_NAME}.
   */
  private static final String NODE_STDOUT_ATTRIBUTE_TYPE = "tipo";

  /**
   * Elemento {@value #NODE_EXIT_CODE_ELEMENT_NAME}.
   */
  private static final String NODE_EXIT_CODE_ELEMENT_NAME = "codigo_saida";

  /**
   * Atributo {@value #NODE_EXIT_CODE_ATTRIBUTE_TYPE} do elemento
   * {@value #NODE_EXIT_CODE_ELEMENT_NAME}.
   */
  private static final String NODE_EXIT_CODE_ATTRIBUTE_TYPE = "tipo";

  /**
   * Quantidade de espaos para representar um caracter tabulao.
   */
  private static final int NUMBER_OF_SPACES_FOR_TABS = 3;

  /**
   * Elemento {@value #OUTPUT_ELEMENT_NAME}.
   */
  private static final String OUTPUT_ELEMENT_NAME = "saida";

  /**
   * Elemento {@value #POINT_ELEMENT_NAME}.
   */
  private static final String POINT_ELEMENT_NAME = "ponto";

  /**
   * Atributo {@value #POINT_X_ATTRIBUTE} do elemento
   * {@value #POINT_ELEMENT_NAME}.
   */
  private static final String POINT_X_ATTRIBUTE = "x";

  /**
   * Atributo {@value #POINT_Y_ATTRIBUTE} do elemento
   * {@value #POINT_ELEMENT_NAME}.
   */
  private static final String POINT_Y_ATTRIBUTE = "y";

  /**
   * Elemento {@value #SIZE_ELEMENT_NAME}.
   */
  private static final String SIZE_ELEMENT_NAME = "tamanho";

  /**
   * Atributo {@value #SIZE_HEIGHT_ATTRIBUTE} do elemento
   * {@value #SIZE_ELEMENT_NAME}.
   */
  private static final String SIZE_HEIGHT_ATTRIBUTE = "altura";

  /**
   * Atributo {@value #SIZE_WIDTH_ATTRIBUTE} do elemento
   * {@value #SIZE_ELEMENT_NAME}.
   */
  private static final String SIZE_WIDTH_ATTRIBUTE = "largura";

  /**
   * Elemento {@value #BYPASSED_ELEMENT_NAME}.
   */
  private static final String BYPASSED_ELEMENT_NAME = "desviar";

  /**
   * Atributo {@value #BYPASSED_ATTRIBUTE} do elemento
   * {@value #BYPASSED_ELEMENT_NAME}
   */
  private static final String BYPASSED_ATTRIBUTE = "valor";

  /**
   * O construtor de documentos <i>DOM</i>.
   */
  private DocumentBuilder builder;

  /**
   * A pilha com os elementos que esto abertos durante a escrita de metadados.
   * Se no houver elementos ou se no estiver escrevendo a pilha estar vazia.
   */
  private final Stack<String> elementNames;

  /**
   * Cria o parser de grafo de algoritmos.
   *
   * @throws ConfigurationException Em caso de erro de configurao.
   * @throws IllegalArgumentException Se o parmetro window estiver nulo.
   */
  public FlowAlgorithmParser() throws ConfigurationException {
    this.elementNames = new Stack<String>();
    try {
      DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      factory.setNamespaceAware(true);
      factory.setIgnoringComments(true);
      factory.setIgnoringElementContentWhitespace(true);
      factory.setValidating(false);
      this.builder = factory.newDocumentBuilder();
    }
    catch (ParserConfigurationException e) {
      String errorMessage =
        "Erro de configurao no analisador de configurador de "
          + "fluxo de algoritmos.\n";
      throw new ConfigurationException(e, errorMessage);
    }
  }

  /**
   * Cria um fluxo a partir de leitor com os metadados de um fluxo.
   *
   * @param inputStream O leitor (No aceita {@code null}).
   *
   * @return O fluxo.
   *
   * @throws ParseException Em caso de erro no formato do leitor.
   * @throws IOException Em caso de erro de ES no leitor.
   */
  public Flow read(InputStream inputStream) throws ParseException, IOException {
    if (inputStream == null) {
      throw new IllegalArgumentException("inputStream == null");
    }
    try {
      InputSource inputSource = new InputSource(inputStream);
      inputSource.setEncoding(DEFAULT_ENCODING);
      Document document = this.builder.parse(inputSource);
      document.normalize();
      Element rootElement = document.getDocumentElement();
      String name = getAttributeValue(rootElement, FLOW_ATTRIBUTE_NAME, true);
      String description =
        getAttributeValue(rootElement, FLOW_DESCRIPTION_NAME, true);
      Set<FlowNode> nodes = decodeNodes(rootElement);
      Set<FlowLink> links = decodeLinks(rootElement);
      Flow flow = new Flow(name, description, nodes, links);
      return flow;
    }
    catch (SAXException e) {
      Throwable throwable = (e.getException() != null ? e.getException() : e);
      String errorMessage =
        "Erro ao analisar o arquivo do configurador do fluxo de algoritmos.\n";
      throw new ParseException(throwable, errorMessage);
    }
  }

  /**
   * Escreve os metadados de um fluxo em um stream de sada utilizando o
   * encoding {@value #DEFAULT_ENCODING}.
   *
   * @param outputStream O stream de sada (No aceita {@code null}).
   * @param flow O fluxo (No aceita {@code null}).
   */
  public void write(OutputStream outputStream, Flow flow) {
    String encoding = DEFAULT_ENCODING;
    write(outputStream, encoding, flow);
  }

  /**
   * Escreve os metadados de um fluxo em um stream de sada utilizando o
   * encoding informado.
   *
   * @param outputStream O stream de sada (No aceita {@code null}).
   * @param encoding O enconding (No aceita {@code null}).
   * @param flow O fluxo (No aceita {@code null}).
   */
  public void write(OutputStream outputStream, String encoding, Flow flow) {
    Charset charset = Charset.forName(encoding);
    Writer writer = new OutputStreamWriter(outputStream, charset);
    PrintWriter printWriter = new PrintWriter(writer);
    write(printWriter, encoding, flow);
  }

  /**
   * Decodifica um fluxo.
   *
   * @param encodedFlow O fluxo codificado.
   * @return O fluxo decodificado.
   *
   * @throws ParseException Em caso de erro no formato do leitor.
   */
  public Flow decode(String encodedFlow) throws ParseException {
    try {
      byte[] content = encodedFlow.getBytes(DEFAULT_ENCODING);
      ByteArrayInputStream inputStream = new ByteArrayInputStream(content);
      return read(inputStream);
    }
    catch (IOException e) {
      throw new BugException(e);
    }
  }

  /**
   * Codifica um fluxo.
   *
   * @param flow O fluxo.
   * @return O fluxo codificado.
   */
  public String encode(Flow flow) {
    ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
    write(outputStream, DEFAULT_ENCODING, flow);
    byte[] content = outputStream.toByteArray();
    try {
      return new String(content, DEFAULT_ENCODING);
    }
    catch (UnsupportedEncodingException e) {
      throw new BugException(e);
    }
  }

  /**
   * Obtm o atributo com o nome fornecido do elemento fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   * @param attributeName O nome do atributo (No aceita {@code null}).
   * @param isOptional Indica se o atributo  opcional ({@code true}) ou
   *        obrigatrio ({@code false}).
   *
   * @return O atributo ou {@code null} se o atributo no existir e for
   *         opcional.
   *
   * @throws ParseException Se o atributo no for encontrado e for obrigatrio.
   */
  private Attr getAttribute(Element element, String attributeName,
    boolean isOptional) throws ParseException {
    NamedNodeMap nodeMap = element.getAttributes();
    Attr attribute = (Attr) nodeMap.getNamedItem(attributeName);
    if (attribute == null) {
      if (isOptional) {
        return null;
      }
      String elementName = element.getLocalName();
      String errorMessage =
        String.format("O atributo %1$s do elemento %0$s no foi encontrado.\n",
          elementName, attributeName);
      throw new ParseException(errorMessage);
    }
    return attribute;
  }

  /**
   * Obtm o valor do atributo com o nome fornecido do elemento fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   * @param attributeName O nome do atributo (No aceita {@code null}).
   *
   * @return O valor do atributo.
   *
   * @throws ParseException Se o atributo no for encontrado.
   */
  private String getAttributeValue(Element element, String attributeName)
    throws ParseException {
    return getAttributeValue(element, attributeName, false);
  }

  /**
   * Obtm o valor do atributo com o nome fornecido do elemento fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   * @param attributeName O nome do atributo (No aceita {@code null}).
   * @param isOptional Indica se o atributo  opcional ({@code true}) ou
   *        obrigatrio ({@code false}).
   *
   * @return O valor do atributo ou {@code null} se o atributo no existir e for
   *         opcional.
   *
   * @throws ParseException Se o atributo no for encontrado e se for
   *         obrigatrio.
   */
  private String getAttributeValue(Element element, String attributeName,
    boolean isOptional) throws ParseException {
    Attr attribute = getAttribute(element, attributeName, isOptional);
    if (attribute == null) {
      return null;
    }
    return attribute.getValue();
  }

  /**
   * Obtm o primeiro filho de um elemento cujo nome  igual ao nome fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   * @param elementName O nome do elemento procurado (No aceita {@code null}).
   *
   * @return O elemento procurado.
   *
   * @throws ParseException Se no houver um elemento com o nome fornecido.
   */
  private Element getChildElement(Element element, String elementName)
    throws ParseException {
    return getChildElement(element, elementName, false);
  }

  /**
   * Obtm o primeiro filho de um elemento cujo nome  igual ao nome fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   * @param elementName O nome do elemento procurado (No aceita {@code null}).
   * @param isOptional Indica se o elemento  opcional ({@code true}) ou 
   *        obrigatrio ({@code false}).
   *
   * @return O elemento procurado ou {@code null} se o elemento no existir e
   *         for opcional.
   *
   * @throws ParseException Se no houver um elemento com o nome fornecido.
   */
  private Element getChildElement(Element element, String elementName,
    boolean isOptional) throws ParseException {
    NodeList childNodes = element.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      Node childNode = childNodes.item(i);
      if (childNode.getNodeType() == Node.ELEMENT_NODE) {
        Element childElement = (Element) childNode;
        if (childElement.getLocalName().equals(elementName)) {
          return childElement;
        }
      }
    }
    if (isOptional) {
      return null;
    }
    throw new ParseException("O elemento {0} no foi encontrado.", elementName);
  }

  /**
   * <p>
   * Obtm os filhos de um elemento cujos nomes so iguais ao nome fornecido.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @param element O elemento (No aceita {@code null}).
   * @param elementName O nome do elemento procurado (No aceita {@code null}).
   *
   * @return A lista com os elementos ou uma lista vazia se no houver
   *         elementos.
   */
  private List<Element> getChildElements(Element element, String elementName) {
    List<Element> childElements = new LinkedList<Element>();
    NodeList childNodes = element.getChildNodes();
    for (int i = 0; i < childNodes.getLength(); i++) {
      Node childNode = childNodes.item(i);
      if (childNode.getNodeType() == Node.ELEMENT_NODE) {
        Element childElement = (Element) childNode;
        if (childElement.getLocalName().equals(elementName)) {
          childElements.add(childElement);
        }
      }
    }
    return Collections.unmodifiableList(childElements);
  }

  /**
   * Obtm o valor do elemento fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   *
   * @return O valor do atributo ou {@code null} se o atributo no existir.
   */
  private String getElementValue(Element element) {
    if (element == null) {
      throw new IllegalArgumentException("O parmetro element est nulo.");
    }
    String textContent = element.getTextContent();
    if (textContent == null) {
      return null;
    }
    textContent = textContent.trim();
    if (textContent.length() == 0) {
      return null;
    }
    return textContent;
  }

  /**
   * Obtm o valor do atributo com o nome fornecido do elemento fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   * @param attributeName O nome do atributo (No aceita {@code null}).
   *
   * @return O valor do atributo.
   *
   * @throws ParseException Se o atributo no for encontrado.
   */
  private int getIntegerAttributeValue(Element element, String attributeName)
    throws ParseException {
    String attributeValue = getAttributeValue(element, attributeName);
    try {
      return Integer.parseInt(attributeValue);
    }
    catch (NumberFormatException e) {
      String elementName = element.getLocalName();
      String errorMessage =
        String.format(
          "O atributo %1$s do elemento %0$s est com o valor %2$s que deveria "
            + "ser um nmero inteiro.", elementName, attributeName,
            attributeValue);
      throw new ParseException(e, errorMessage);
    }
  }

  /**
   * Escreve a identao da prxima linha de texto que ser escrita.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   */
  private void indent(PrintWriter printWriter) {
    for (int i = 0; i < this.elementNames.size(); i++) {
      for (int j = 0; j < NUMBER_OF_SPACES_FOR_TABS; j++) {
        printWriter.print(' ');
      }
    }
  }

  /**
   * Cria uma conexo representada pelo elemento <i>XML</i> fornecido.
   *
   * @param element O elemento (Precisa ser o elemento
   *        {@value #LINK_ELEMENT_NAME}).
   *
   * @return A conexo.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private FlowLink decodeLink(Element element) throws ParseException {
    int id = getIntegerAttributeValue(element, LINK_ATTRIBUTE_ID);

    Element outputElement = getChildElement(element, OUTPUT_ELEMENT_NAME, true);
    LinkParameter outputLinkParameter;
    if (outputElement != null) {
      outputLinkParameter = decodeLinkParameter(outputElement);
    }
    else {
      return null;
    }
    Element inputElement = getChildElement(element, INPUT_ELEMENT_NAME, true);
    LinkParameter inputLinkParameter;
    if (inputElement != null) {
      inputLinkParameter = decodeLinkParameter(inputElement);
    }
    else {
      return null;
    }
    List<Point> points = decodePoints(element);
    FlowLink link =
      new FlowLink(id, outputLinkParameter, inputLinkParameter, points);
    return link;
  }

  /**
   * Cria um parmetro de conexo representado pelo elemento <i>XML</i>
   * fornecido.
   *
   * @param element O elemento (Precisa ser o elemento
   *        {@value #INPUT_ELEMENT_NAME} ou {@value #OUTPUT_ELEMENT_NAME}).
   *
   * @return O parmetro de conexo.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private LinkParameter decodeLinkParameter(Element element)
    throws ParseException {
    int nodeId =
      getIntegerAttributeValue(element, LINK_PARAMETER_ATTRIBUTE_NODE_ID);
    String parameterName =
      getAttributeValue(element, LINK_PARAMETER_ATTRIBUTE_NAME);
    String parameterLabel =
      getAttributeValue(element, LINK_PARAMETER_ATTRIBUTE_LABEL);
    LinkParameter parameter =
      new LinkParameter(nodeId, parameterName, parameterLabel);
    return parameter;
  }

  /**
   * Cria as conexes representadas por elementos filhos do elemento <i>XML</i>
   * fornecido.
   *
   * @param parentElement O elemento pai dos elementos (No aceita {@code null}
   *        ).
   *
   * @return O conjunto de conexes ou um conjunto vazio caso no haja conexes.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private Set<FlowLink> decodeLinks(Element parentElement)
    throws ParseException {
    Set<FlowLink> links = new HashSet<FlowLink>();
    List<Element> elements = getChildElements(parentElement, LINK_ELEMENT_NAME);
    for (Element element : elements) {
      FlowLink link = decodeLink(element);
      if (link != null) {
        links.add(link);
      }
    }
    return links;
  }

  /**
   * Cria o n representado pelo elemento <i>XML</i> fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   *
   * @return O n.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   * @throws RemoteException Em caso de falha na comunicao com o servidor.
   */
  private FlowNode decodeNode(Element element) throws ParseException,
  RemoteException {
    int id = getIntegerAttributeValue(element, NODE_ATTRIBUTE_ID);
    Element algorithmElement = getChildElement(element, ALGORITHM_ELEMENT_NAME);
    String algorithmName =
      getAttributeValue(algorithmElement, ALGORITHM_ATTRIBUTE_NAME);
    String algorithmVersionIdAsText =
      getAttributeValue(algorithmElement, ALGORITHM_ATTRIBUTE_VERSION);
    AlgorithmVersionId algorithmVersionId =
      AlgorithmVersionId.create(algorithmVersionIdAsText);
    if (algorithmVersionId == null) {
      String errorMessage =
        String.format(
          "O valor para a verso (%s) est num formato invlido.\n",
          algorithmVersionIdAsText);
      throw new ParseException(errorMessage);
    }
    Set<NodeParameter> parameters = decodeNodeParameters(algorithmElement);
    FileURLValue stdout = decodeNodeStandardOutput(algorithmElement);
    FileURLValue exitCode = decodeNodeExitCodeFile(algorithmElement);
    Element pointElement = getChildElement(element, POINT_ELEMENT_NAME);
    Point point = decodePoint(pointElement);
    Element sizeElement = getChildElement(element, SIZE_ELEMENT_NAME);
    int width = getIntegerAttributeValue(sizeElement, SIZE_WIDTH_ATTRIBUTE);
    int height = getIntegerAttributeValue(sizeElement, SIZE_HEIGHT_ATTRIBUTE);
    Boolean bypassed = false;
    try {
      Element bypasseElement = getChildElement(element, BYPASSED_ELEMENT_NAME);
      String bypassedString =
        getAttributeValue(bypasseElement, BYPASSED_ATTRIBUTE);
      bypassed = Boolean.parseBoolean(bypassedString);
      AlgorithmConfigurator algorithmConfigurator =
        getAlgorithmService().createAlgorithmConfigurator(algorithmName,
          algorithmVersionId);

      if (algorithmConfigurator != null) {
        algorithmConfigurator.setStandardOutputFile(stdout);
        algorithmConfigurator.setExitCodeLogFile(exitCode);
        if (exitCode != null) {
          algorithmConfigurator.setHasExitCode(true);
        }
        try {
          fillParameterValues(algorithmConfigurator, parameters);
        }
        catch (ParameterNotFoundException e) {
          e.printStackTrace();
          // Deixa a execuo terminar normalmente.
        }
        // Por que precisa passar os parmetros se eles j foram
        // preenchidos no configurador?
        return new FlowNode(id, algorithmConfigurator, parameters,
          point.getX(), point.getY(), width, height, bypassed);
      }
      else {
        return new FlowNode(id, algorithmName, algorithmVersionId, parameters,
          point.getX(), point.getY(), width, height, bypassed, false);
      }
    }
    catch (AlgorithmNotFoundException e) {
      return new FlowNode(id, algorithmName, algorithmVersionId, parameters,
        point.getX(), point.getY(), width, height, bypassed, false);
    }
    catch (ParseException e) {
      return new FlowNode(id, algorithmName, algorithmVersionId, parameters,
        point.getX(), point.getY(), width, height, bypassed, false);
    }
  }

  /**
   * Preenche os valores dos parmetros do configurador.
   *
   * @param configurator O configurador (No aceita {@code null}).
   * @param parameters Os parmetros (No aceita {@code null}).
   * @throws ParseException Formato do valor de pelo menos um parmetro est no
   *         formato errado.
   * @throws ParameterNotFoundException Caso no exista um parmetro com o nome
   *         fornecido.
   */
  private void fillParameterValues(AlgorithmConfigurator configurator,
    Set<NodeParameter> parameters) throws ParseException,
    ParameterNotFoundException {
    Map<String, String> parameterValuesByName = new HashMap<String, String>();
    for (NodeParameter parameter : parameters) {
      parameterValuesByName.put(parameter.getName(), parameter.getValue());
    }
    configurator.setParameterValuesByName(parameterValuesByName);
  }

  /**
   * Retorna o servio de algoritmos.
   *
   * @return o servio de algoritmos.
   */
  private AlgorithmServiceInterface getAlgorithmService() {
    return ClientRemoteLocator.algorithmService;
  }

  /**
   * Cria um parmetro de n a partir do elemento <i>XML</i> fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   *
   * @return O parmetro de n.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private NodeParameter decodeNodeParameter(Element element)
    throws ParseException {
    String parameterName =
      getAttributeValue(element, NODE_PARAMETER_ATTRIBUTE_NAME);
    String parameterLabel =
      getAttributeValue(element, NODE_PARAMETER_ATTRIBUTE_LABEL);
    String parameterType =
      getAttributeValue(element, NODE_PARAMETER_ATTRIBUTE_TYPE);
    String parameterValue =
      getAttributeValue(element, NODE_PARAMETER_ATTRIBUTE_VALUE, true);
    if (parameterValue == null) {
      Element childElement =
        getChildElement(element, NODE_PARAMETER_VALUE_ELEMENT_NAME, true);
      if (childElement != null) {
        parameterValue = getElementValue(childElement);
      }
    }
    NodeParameter parameter =
      new NodeParameter(parameterName, parameterLabel, parameterType,
        parameterValue);
    return parameter;
  }

  /**
   * Cria os parmetro de n representadas por elementos filhos do elemento
   * <i>XML</i> fornecido.
   *
   * @param parentElement O elemento pai dos elementos (No aceita {@code null}
   *        ).
   *
   * @return O conjunto de parmetros de n ou um conjunto vazio caso no haja
   *         parmetro de n.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private Set<NodeParameter> decodeNodeParameters(Element parentElement)
    throws ParseException {
    Set<NodeParameter> parameters = new HashSet<NodeParameter>();
    List<Element> elements =
      getChildElements(parentElement, NODE_PARAMETER_ELEMENT_NAME);
    for (Element element : elements) {
      NodeParameter parameter = decodeNodeParameter(element);
      parameters.add(parameter);
    }
    return parameters;
  }

  /**
   * Cria um {@link FileURLValue} indicando a sada padro o elemento representada
   * pelo elemento {@code NODE_STDOUT_ELEMENT_NAME} filho do elemento <i>XML</i>
   * fornecido.
   *
   * @param parentElement O elemento pai do elemento (No aceita {@code null}).
   *
   * @return um {@link FileURLValue} indicando a sada padro ou null caso no haja
   *         uma.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private FileURLValue decodeNodeStandardOutput(Element parentElement)
    throws ParseException {
    Element stdoutElement =
      getChildElement(parentElement, NODE_STDOUT_ELEMENT_NAME, true);
    if (null == stdoutElement) {
      return null;
    }
    else {
      String path = getElementValue(stdoutElement);
      String type =
        getAttributeValue(stdoutElement, NODE_STDOUT_ATTRIBUTE_TYPE);
      return new FileURLValue(path, type);
    }
  }

  /**
   * Cria um {@link FileURLValue} indicando o arquivo que armazena o cdigo de sada
   * do elemento representada pelo elemento {@code NODE_EXIT_CODE_ELEMENT_NAME}
   * filho do elemento <i>XML</i> fornecido.
   *
   * @param parentElement O elemento pai do elemento (No aceita {@code null}).
   *
   * @return um {@link FileURLValue} indicando o arquivo que armazena o cdigo de
   *         sada ou null caso no haja uma.
   *
   * @throws ParseException Se no houver algum atributo ou elemento
   *         obrigatrio.
   */
  private FileURLValue decodeNodeExitCodeFile(Element parentElement)
    throws ParseException {
    Element exitCodeElement =
      getChildElement(parentElement, NODE_EXIT_CODE_ELEMENT_NAME, true);
    if (null == exitCodeElement) {
      return null;
    }
    else {
      String path = getElementValue(exitCodeElement);
      String type =
        getAttributeValue(exitCodeElement, NODE_EXIT_CODE_ATTRIBUTE_TYPE);
      return new FileURLValue(path, type);
    }
  }

  /**
   * Cria os ns representados por elementos filhos do elemento <i>XML</i>
   * fornecido.
   *
   * @param parentElement O elemento pai dos elementos (No aceita {@code null}
   *        ).
   *
   * @return O conjunto de ns ou um conjunto vazio caso no haja conexes.
   *
   * @throws ParseException Se houver algum erro na sub-rvore do documento.
   * @throws RemoteException Em caso de falha na comunicao com o servidor.
   */
  private Set<FlowNode> decodeNodes(Element parentElement)
    throws ParseException, RemoteException {
    Set<FlowNode> nodes = new HashSet<FlowNode>();
    List<Element> elements = getChildElements(parentElement, NODE_ELEMENT_NAME);
    for (Element element : elements) {
      FlowNode node = decodeNode(element);
      nodes.add(node);
    }
    return nodes;
  }

  /**
   * Cria um ponto a partir do elemento <i>XML</i> fornecido.
   *
   * @param element O elemento (No aceita {@code null}).
   *
   * @return O ponto.
   *
   * @throws ParseException Se no houver algum atributo obrigatrio.
   */
  private Point decodePoint(Element element) throws ParseException {
    int x = getIntegerAttributeValue(element, POINT_X_ATTRIBUTE);
    int y = getIntegerAttributeValue(element, POINT_Y_ATTRIBUTE);
    Point point = new Point(x, y);
    return point;
  }

  /**
   * Cria os pontos representadas por elementos filhos do elemento <i>XML</i>
   * fornecido.
   *
   * @param parentElement O elemento pai dos elementos (No aceita {@code null}
   *        ).
   *
   * @return O conjunto de pontos ou um conjunto vazio caso no haja pontos.
   *
   * @throws ParseException Se no houver algum atributo obrigatrio.
   */
  private List<Point> decodePoints(Element parentElement) throws ParseException {
    List<Point> points = new LinkedList<Point>();
    List<Element> elements =
      getChildElements(parentElement, POINT_ELEMENT_NAME);
    for (Element element : elements) {
      Point point = decodePoint(element);
      points.add(point);
    }
    return points;
  }

  /**
   * Escreve os metadados de um fluxo em um escritor.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param encoding O enconding (No aceita {@code null}).
   * @param flow O fluxo (No aceita {@code null}).
   */
  private void write(PrintWriter printWriter, String encoding, Flow flow) {
    printWriter.printf("<?xml version='1.0' encoding='%s'?>\n", encoding);

    List<Attribute> attributes = new ArrayList<Attribute>();
    String flowName = flow.getName();
    if (flowName != null) {
      attributes.add(new Attribute(FLOW_ATTRIBUTE_NAME, flowName));
    }

    String description = flow.getDescription();
    if (description != null && !description.trim().isEmpty()) {
      attributes.add(new Attribute(FLOW_DESCRIPTION_NAME, description));
    }

    Attribute[] attributesArray =
      attributes.toArray(new Attribute[attributes.size()]);
    writeElementStart(printWriter, FLOW_ELEMENT_NAME, attributesArray);

    writeNodes(printWriter, flow.getNodes());
    writeLinks(printWriter, flow.getLinks());
    writeElementEnd(printWriter);
    printWriter.flush();
  }

  /**
   * Escreve um elemento CDATA.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param value O valor (No aceita {@code null}).
   */
  private void writeCDATA(PrintWriter printWriter, String value) {
    indent(printWriter);
    printWriter.print("<![CDATA[");
    printWriter.print(value);
    printWriter.println("]]>");
  }

  /**
   * <p>
   * Escreve a tag que termina o elemento que est sendo construdo.
   * </p>
   *
   * <p>
   * Para que este mtodo funcione corretamente,  necessrio que haja um
   * elemento sendo construdo.
   * </p>
   *
   * @param printWriter O escritor (No aceita {@code null}).
   */
  private void writeElementEnd(PrintWriter printWriter) {
    String elementName = this.elementNames.pop();
    indent(printWriter);
    printWriter.print("</");
    printWriter.print(elementName);
    printWriter.println(">");
  }

  /**
   * Escreve a tag que inicia um elemento.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param elementName O nome do elemento (No aceita {@code null}).
   * @param attributes Os atributos do elemento (No aceita {@code null}).
   */
  private void writeElementStart(PrintWriter printWriter, String elementName,
    Attribute... attributes) {
    indent(printWriter);
    printWriter.print("<");
    printWriter.print(elementName);
    for (Attribute attribute : attributes) {
      printWriter.print(" ");
      printWriter.print(attribute);
    }
    printWriter.println(">");
    this.elementNames.push(elementName);
  }

  /**
   * Escreve os metadados de uma conexo.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param link A conexo (No aceita {@code null}).
   */
  private void writeLink(PrintWriter printWriter, FlowLink link) {
    LinkParameter inputLinkParameter = link.getInput();
    LinkParameter outputLinkParameter = link.getOutput();
    if (outputLinkParameter != null && inputLinkParameter != null) {
      Attribute idAttribute = new Attribute(LINK_ATTRIBUTE_ID, link.getId());
      writeElementStart(printWriter, LINK_ELEMENT_NAME, idAttribute);
      writeLinkParameter(printWriter, OUTPUT_ELEMENT_NAME, outputLinkParameter);
      writeLinkParameter(printWriter, INPUT_ELEMENT_NAME, inputLinkParameter);
      writePoints(printWriter, link.getPoints());
      writeElementEnd(printWriter);
    }
  }

  /**
   * Escreve os metadados de um parmetro de conexo.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param elementName O nome do elemento (No aceita {@code null}).
   * @param parameter O parmetro da conexo (No aceita {@code null}).
   */
  private void writeLinkParameter(PrintWriter printWriter, String elementName,
    LinkParameter parameter) {
    Attribute nodeIdAttribute =
      new Attribute(LINK_PARAMETER_ATTRIBUTE_NODE_ID, parameter.getNodeId());
    Attribute nameAttribute =
      new Attribute(LINK_PARAMETER_ATTRIBUTE_NAME, parameter.getName());
    Attribute labelAttribute =
      new Attribute(LINK_PARAMETER_ATTRIBUTE_LABEL, parameter.getLabel());
    writeElementStart(printWriter, elementName, nodeIdAttribute, nameAttribute,
      labelAttribute);
    writeElementEnd(printWriter);
  }

  /**
   * Escreve os metadados das conexes.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param links As conexes (No aceita {@code null}).
   */
  private void writeLinks(PrintWriter printWriter, Set<FlowLink> links) {
    for (FlowLink link : links) {
      writeLink(printWriter, link);
    }
  }

  /**
   * Escreve os metadados de um n.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param node O n (No aceita {@code null}).
   */
  private void writeNode(PrintWriter printWriter, FlowNode node) {
    Attribute idAttribute = new Attribute(NODE_ATTRIBUTE_ID, node.getId());
    writeElementStart(printWriter, NODE_ELEMENT_NAME, idAttribute);
    Attribute algorithmNameAttribute =
      new Attribute(ALGORITHM_ATTRIBUTE_NAME, node.getAlgorithmName());
    Attribute algorithmVersionAttribute =
      new Attribute(ALGORITHM_ATTRIBUTE_VERSION, node.getAlgorithmVersionId()
        .toString());
    writeElementStart(printWriter, ALGORITHM_ELEMENT_NAME,
      algorithmNameAttribute, algorithmVersionAttribute);
    writeNodeParameters(printWriter, node.getParameters());
    writeNodeStandardOutput(printWriter, node);
    writeNodeExitCodeFile(printWriter, node);
    writeElementEnd(printWriter);
    writePoint(printWriter, node.getX(), node.getY());
    Attribute widthAttribute =
      new Attribute(SIZE_WIDTH_ATTRIBUTE, node.getWidth());
    Attribute heightAttribute =
      new Attribute(SIZE_HEIGHT_ATTRIBUTE, node.getHeight());
    writeElementStart(printWriter, SIZE_ELEMENT_NAME, widthAttribute,
      heightAttribute);
    writeElementEnd(printWriter);
    Attribute bypassedAttribute =
      new Attribute(BYPASSED_ATTRIBUTE, String.valueOf(node.isBypassed()));
    writeElementStart(printWriter, BYPASSED_ELEMENT_NAME, bypassedAttribute);
    writeElementEnd(printWriter);
    writeElementEnd(printWriter);
  }

  /**
   * Escreve os metadados de parmetro de n.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param parameter O parmetro de n (No aceita {@code null}).
   */
  private void writeNodeParameter(PrintWriter printWriter,
    NodeParameter parameter) {
    Attribute nameAttribute =
      new Attribute(NODE_PARAMETER_ATTRIBUTE_NAME, parameter.getName());
    Attribute labelAttribute =
      new Attribute(NODE_PARAMETER_ATTRIBUTE_LABEL, parameter.getLabel());
    Attribute typeAttribute =
      new Attribute(NODE_PARAMETER_ATTRIBUTE_TYPE, parameter.getType());
    writeElementStart(printWriter, NODE_PARAMETER_ELEMENT_NAME, nameAttribute,
      labelAttribute, typeAttribute);
    String parameterValue = parameter.getValue();
    if (parameterValue != null) {
      writeElementStart(printWriter, NODE_PARAMETER_VALUE_ELEMENT_NAME);
      writeCDATA(printWriter, parameterValue);
      writeElementEnd(printWriter);
    }
    writeElementEnd(printWriter);
  }

  /**
   * Escreve os metadados dos parmetros de n.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param parameters Os parmetros de n (No aceita {@code null}).
   */
  private void writeNodeParameters(PrintWriter printWriter,
    Set<NodeParameter> parameters) {
    for (NodeParameter parameter : parameters) {
      writeNodeParameter(printWriter, parameter);
    }
  }

  /**
   * Escreve os metadados especificando a sada padro de um n.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param node O n (No aceita {@code null}).
   */
  private void writeNodeStandardOutput(PrintWriter printWriter, FlowNode node) {
    AlgorithmConfigurator nodeConfigurator = node.getAlgorithmConfigurator();
    if (null == nodeConfigurator) {
      return;
    }
    FileURLValue stdout = nodeConfigurator.getStandardOutputFile();
    if (null == stdout) {
      return;
    }

    Attribute typeAttribute =
      new Attribute(NODE_STDOUT_ATTRIBUTE_TYPE, stdout.getType());
    writeElementStart(printWriter, NODE_STDOUT_ELEMENT_NAME, typeAttribute);
    writeCDATA(printWriter, stdout.getPath());
    writeElementEnd(printWriter);
  }

  /**
   * Escreve os metadados especificando o arquivo que armazena o cdigo de sada
   * do elemento.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param node O n (No aceita {@code null}).
   */
  private void writeNodeExitCodeFile(PrintWriter printWriter, FlowNode node) {
    AlgorithmConfigurator nodeConfigurator = node.getAlgorithmConfigurator();
    if (null == nodeConfigurator) {
      return;
    }
    FileURLValue exitCodeFile = nodeConfigurator.getExitCodeLogFile();
    if (null == exitCodeFile || !nodeConfigurator.hasExitCode()) {
      return;
    }

    Attribute typeAttribute =
      new Attribute(NODE_EXIT_CODE_ATTRIBUTE_TYPE, exitCodeFile.getType());
    writeElementStart(printWriter, NODE_EXIT_CODE_ELEMENT_NAME, typeAttribute);
    writeCDATA(printWriter, exitCodeFile.getPath());
    writeElementEnd(printWriter);
  }

  /**
   * Escreve os metadados dos ns.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param nodes Os ns (No aceita {@code null}).
   */
  private void writeNodes(PrintWriter printWriter, Set<FlowNode> nodes) {
    for (FlowNode node : nodes) {
      writeNode(printWriter, node);
    }
  }

  /**
   * Escreve os metadados de um ponto.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param x A abscissa do ponto.
   * @param y A ordenada do ponto.
   */
  private void writePoint(PrintWriter printWriter, int x, int y) {
    Attribute xAttribute = new Attribute(POINT_X_ATTRIBUTE, x);
    Attribute yAttribute = new Attribute(POINT_Y_ATTRIBUTE, y);
    writeElementStart(printWriter, POINT_ELEMENT_NAME, xAttribute, yAttribute);
    writeElementEnd(printWriter);
  }

  /**
   * Escreve os metadados dos pontos.
   *
   * @param printWriter O escritor (No aceita {@code null}).
   * @param points Os pontos (No aceita {@code null}).
   */
  private void writePoints(PrintWriter printWriter, List<Point> points) {
    for (Point point : points) {
      writePoint(printWriter, point.getX(), point.getY());
    }
  }

  /**
   * Atributo de um elemento <i>XML</i>.
   *
   * @author lmoreira
   */
  private static final class Attribute {
    /**
     * O nome.
     */
    private String name;

    /**
     * O valor.
     */
    private String value;

    /**
     * Cria um atributo.
     *
     * @param name O nome (No aceita {@code null}).
     * @param value O valor.
     */
    Attribute(String name, int value) {
      this(name, Integer.toString(value));
    }

    /**
     * Cria um atributo.
     *
     * @param name O nome (No aceita {@code null}).
     * @param value O valor (No aceita {@code null}).
     */
    Attribute(String name, String value) {
      setName(name);
      setValue(value);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
      return this.name + "='" + this.value + "'";
    }

    /**
     * Atribui um nome a este atributo.
     *
     * @param name O nome (No aceita {@code null}).
     */
    private void setName(String name) {
      if (name == null) {
        throw new IllegalArgumentException("O parmetro this.name est nulo.");
      }

      this.name = name;
    }

    /**
     * Atribui um valor a este atributo.
     *
     * @param value O valor (No aceita {@code null}).
     */
    private void setValue(String value) {
      if (value == null) {
        throw new IllegalArgumentException("O parmetro this.value est nulo.");
      }

      this.value = value;
    }
  }
}
