package csbase.logic;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import csbase.exception.project.FileLockedException;

/**
 * Interface para representao de arquivos no lado cliente.
 * 
 * @author Tecgraf/PUC-Rio
 */
public interface ClientFile {

  /** Modo de otimizao: sem otimizao */
  public static int OPTIMIZATION_MODE_NONE = 0;

  /**
   * Modo de otimizao: Global ou simplificado - mesmo mounting point para
   * todos os projetos e usurios
   */
  public static int OPTIMIZATION_MODE_GLOBAL = 1;

  /**
   * Retorna a lista de arquivos deste diretrio.
   * 
   * @return A lista de arquivos deste diretrio.
   * @throws Exception Erro na busca de arquivos do diretrio.
   */
  ClientFile[] getChildren() throws Exception;

  /**
   * Obtm um fluxo de entrada para leitura do arquivo.
   * 
   * @return Fluxo de entrada para leitura do arquivo.
   * @throws IOException Caso ocorra algum erro de IO durante o procedimento.
   */
  InputStream getInputStream() throws IOException;

  /**
   * Retorna se existe o arquivo fsico.
   * 
   * @return indicativo
   * @throws IOException em caso de falha e I/O no procedimento.
   */
  boolean exists() throws IOException;

  /**
   * Consulta a data/hora da ltima modificao do arquivo.
   * 
   * @return a data
   */
  long getModificationDate();

  /**
   * Retorna o nome do arquivo.
   * 
   * @return Nome do arquivo.
   */
  String getName();

  /**
   * Obtm um fluxo de sada para escrita no arquivo.
   * 
   * @return Fluxo de sada para escrita no arquivo.
   * @throws IOException Caso ocorra algum erro de IO durante o procedimento.
   */
  OutputStream getOutputStream() throws IOException;

  /**
   * Retorna o diretrio-pai
   * 
   * @return o diretrio que contem este arquivo.
   */
  ClientFile getParent();

  /**
   * Retorna o caminho para o arquivo na forma de um array (no inclui o nome do
   * projeto).
   * 
   * @return Caminho para o arquivo na forma de um array (no inclui o nome do
   *         projeto).
   */
  String[] getPath();

  /**
   * Retorna o path pro arquivo na forma de string.
   * 
   * @return O path pro arquivo na forma de string.
   */
  String getStringPath();

  /**
   * Informa o tipo do arquivo.
   * 
   * @return O tipo do arquivo.
   */
  String getType();

  /**
   * Consulta se um arquivo  diretrio.
   * 
   * @return indicativo
   */
  boolean isDirectory();

  /**
   * Retorna true se o arquivo pode ser lido, false caso contrrio.
   * 
   * @return true se o arquivo pode ser lido, false caso contrrio.
   */
  boolean canRead();

  /**
   * Retorna true se o arquivo pode ser escrito, false caso contrrio.
   * 
   * @return - true se o arquivo pode ser escrito, false caso contrrio.
   */
  boolean canWrite();

  /**
   * Retorna true se o arquivo pode ser executado, false caso contrrio.
   * 
   * @return true se o arquivo pode ser executado, false caso contrrio.
   */
  boolean canExecute();

  /**
   * Abre o arquivo para leitura ou para leitura e escrita.
   * 
   * @param readOnly Boolean indicando se o arquivo deve ser aberto para leitura
   *        ou para leitura e escrita.
   * @throws Exception Erro na abertura do arquivo.
   */
  void open(final boolean readOnly) throws Exception;

  /**
   * L uma seqncia de bytes deste arquivo a partir da posio fornecida.
   * 
   * @param dst O array no qual os bytes sero escritos.
   * @param position A posio do arquivo a partir do qual os bytes sero lidos.
   * 
   * @return O nmero de bytes lidos.
   * @throws Exception Em caso de erro.
   */
  int read(byte[] dst, long position) throws Exception;

  /**
   * L uma seqncia de bytes deste arquivo a partir da posio fornecida.
   * 
   * @param dst O array no qual os bytes sero escritos.
   * @param off O ndice do array a partir do qual os bytes sero escritos.
   * @param len O nmero de bytes a ler do arquivo e escrever no array.
   * @param position A posio do arquivo a partir do qual os bytes sero lidos.
   * 
   * @return O nmero de bytes lidos.
   * @throws Exception Em caso de erro.
   */
  int read(byte[] dst, int off, int len, long position) throws Exception;

  /**
   * Fecha este arquivo. Aps chamado este mtodo, o objeto que representa o
   * arquivo no poder mais ser utilizado para leitura e/ou escrita.
   * 
   * @param force Indica que o arquivo deve ser considerado fechado mesmo em
   *        caso de erro.
   * 
   * @throws IOException se houver falha no acesso ao arquivo.
   */
  void close(boolean force) throws IOException;

  /**
   * Informa o tamanho corrente deste arquivo. Reflete o nmero corrente de
   * bytes do arquivo.
   * 
   * @return Tamanho corrente do arquivo.
   */
  public long size();

  /**
   * Retorna item de enumerao que identifica o tipo do arquivo quanto ao local
   * onde o arquivo est armazenado.
   * 
   * @return Item de enumerao que identifica o tipo do arquivo quanto ao local
   *         onde o arquivo est armazenado.
   */
  public ClientFileType getClientFileType();

  /**
   * Escreve uma sequncia de bytes do arquivo. Esse mtodo tenta fazer escritas
   * consecutivas at conseguir enviar a quantidade de bytes solicitada. Caso
   * no consiga, o mtodo lana uma exceo.
   * 
   * @param src O array contendo os bytes a serem escritos no arquivo.
   * @param off O deslocamento no array a partir do qual os bytes sero lidos.
   * @param len A quantidade de bytes a serem escritos do arquivo.
   * @param position A posio do arquivo a partir da qual a escrita deve ser
   *        iniciada.
   * 
   * @throws IOException Em caso de erro.
   * @throws FileLockedException caso o arquivo esteja bloqueado para escrita
   *         por outra sesso de usurio
   */
  public void write(byte[] src, int off, int len, long position)
    throws IOException, FileLockedException;

  /**
   * Escreve uma sequncia de bytes do arquivo. Esse mtodo tenta fazer escritas
   * consecutivas at conseguir enviar a quantidade de bytes solicitada. Caso
   * no consiga, o mtodo lana uma exceo.
   * 
   * @param src O array contendo os bytes a serem escritos no arquivo.
   * @param position A posio do arquivo a partir da qual a escrita deve ser
   *        iniciada.
   * 
   * @throws IOException Em caso de erro.
   * @throws FileLockedException caso o arquivo esteja bloqueado para escrita
   *         por outra sesso de usurio
   */
  public void write(byte[] src, long position)
    throws IOException, FileLockedException;

  /**
   * Informa a posio corrente deste arquivo. Reflete o nmero de bytes a
   * partir do incio do arquivo at a posio corrente.
   * 
   * @return Posio corrente do arquivo.
   * 
   * @throws IOException Em caso de erro.
   */
  public long position() throws IOException;

  /**
   * Altera a posio corrente deste arquivo. Reflete o nmero de bytes a partir
   * do incio do arquivo at a posio corrente.
   * 
   * @param newPosition Nova posio corrente do arquivo.
   * 
   * @throws IOException Em caso de erro.
   */
  public void position(long newPosition) throws IOException;

}