package csbase.logic.algorithms.parameters;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import csbase.exception.algorithms.ExpressionFunctionExecutionException;

/**
 * Assinatura de um mtodo que trabalha com tabelas.
 */
public abstract class TableExpressionFunction extends ExpressionFunction {

  /**
   * Cria uma assinatura.
   * 
   * @param name O nome do mtodo.
   * @param parameterTypes O array com os tipos dos parmetros.
   */
  public TableExpressionFunction(String name, Class<?>... parameterTypes) {
    super(name, createParameterTypes(parameterTypes));
  }

  /**
   * Executa a operao sobre a tabela.
   * 
   * @param values a lista de valores da tabela.
   * @param arguments os argumentos do mtodo.
   * @return object o resultado do mtodo.
   * 
   * @throws ExpressionFunctionExecutionException em caso de falha na execuo
   *         do mtodo.
   */
  protected abstract Object doOperation(List<Object> values,
    Object... arguments) throws ExpressionFunctionExecutionException;

  /**
   * {@inheritDoc}
   */
  @Override
  protected final Object doOperation(SimpleAlgorithmConfigurator configurator,
    Object... parameters) throws ExpressionFunctionExecutionException {
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    if (parameters == null) {
      throw new IllegalArgumentException("O parmetro parameters est nulo.");
    }
    if (parameters.length < 2) {
      throw new ExpressionFunctionExecutionException(
        "Esta funo trabalha sobre tabelas.\n"
          + "O primeiro parmetro tem que ser uma tabela e "
          + "o segundo parmetro tem que ser uma coluna.\n"
          + "Primeiro fornea o nome da lista a ser utilizada.\n"
          + "Funo: {0}.", this, parameters.length);
    }
    if (!(parameters[0] instanceof String)) {
      throw new ExpressionFunctionExecutionException(
        "Esta funo trabalha sobre tabelas.\n"
          + "O primeiro valor tem que ser uma tabela, "
          + "porm foi informado um valor que no  um nome de tabela.\n"
          + "Fornea o nome da tabela a ser utilizada.\n"
          + "Funo: {0}.\nValor: {1}.", this, parameters[0]);
    }
    String parameterName = (String) parameters[0];
    SimpleParameter<?> parameter =
      configurator.getSimpleParameter(parameterName);
    if (parameter == null) {
      throw new ExpressionFunctionExecutionException(
        "A tabela {0} no foi encontrada no configurador {1}.\nFuno: {2}.",
        parameterName, configurator, this);
    }
    if (!(parameter instanceof TableParameter)) {
      throw new ExpressionFunctionExecutionException(
        "O parmetro {0} do configurador {1} no  uma tabela.\n"
          + "Funo: {2}.", parameterName, configurator, this);
    }
    TableParameter tableParameter = (TableParameter) parameter;
    if (!(parameters[1] instanceof Double)) {
      throw new ExpressionFunctionExecutionException(
        "Esta funo trabalha sobre tabelas.\n"
          + "O segundo valor tem que ser uma coluna, "
          + "porm foi informado um valor que no  um ndice de coluna.\n"
          + "Fornea o nome da coluna a ser utilizada.\n"
          + "Funo: {0}.\nValor: {1}.", this, parameters[1]);
    }
    int columnIndex = ((Double) parameters[1]).intValue();
    if (columnIndex >= tableParameter.getColumnCount()) {
      throw new ExpressionFunctionExecutionException(
        "A coluna {1} da tabela {0} no foi encontrada no configurador {2}.\n"
          + "Funo: {3}.", parameterName, columnIndex, configurator, this);
    }
    List<Object> values = new ArrayList<Object>(tableParameter.getRowCount());
    for (int i = 0; i < tableParameter.getRowCount(); i++) {
      Object value = tableParameter.getItemValue(i, columnIndex);
      values.add(value);
    }
    values = Collections.unmodifiableList(values);
    Object[] newParameters = new Object[parameters.length - 2];
    for (int i = 0; i < newParameters.length; i++) {
      newParameters[i] = parameters[i + 2];
    }
    return doOperation(values, newParameters);
  }

  /**
   * Cria um novo array de tipos dos parmetros do mtodo a partir do array
   * original. Acrescenta os tipos dos parmetros necessrios para os mtodos
   * sobre tabelas.
   * 
   * @param parameterTypes o array original de tipos.
   * 
   * @return o novo array de tipos.
   */
  private static Class<?>[] createParameterTypes(Class<?>[] parameterTypes) {
    if (parameterTypes == null) {
      throw new IllegalArgumentException(
        "O parmetro parameterTypes est nulo.");
    }
    Class<?>[] newParameterTypes = new Class[parameterTypes.length + 2];
    newParameterTypes[0] = String.class;
    newParameterTypes[1] = Double.class;
    for (int i = 2; i < newParameterTypes.length; i++) {
      newParameterTypes[i] = parameterTypes[i - 2];
    }
    return newParameterTypes;
  }
}
