package csbase.logic.algorithms.parsers;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import csbase.exception.ParseException;
import csbase.logic.algorithms.parameters.ParameterGroup;
import csbase.logic.algorithms.parameters.SimpleAlgorithmConfigurator;
import csbase.logic.algorithms.parameters.TableColumn;
import csbase.logic.algorithms.parameters.TableParameter;
import csbase.logic.algorithms.parsers.columns.TableColumnFactory;

/**
 * <p>
 * Analisador de {@link TableParameter}.
 * </p>
 *
 * <p>
 * Este parser l os atributos de parmetros do tipo tabela. O elemento corrente
 * do {@link XmlParser analisador de XML} precisa ser um elemento
 * {@link TableParameter}.
 * </p>
 *
 */
public abstract class AbstractTableParameterFactory extends
  SimpleParameterParser<TableParameter> {

  /**
   * <p>
   * O elemento {@value #TABLE_PARAMETER_ELEMENT}: descreve as propriedades de
   * um {@link TableParameter parmetro do tipo tabela}.
   * </p>
   * <p>
   *  filho do elemento {@link TableParameter}.
   * </p>
   */
  public static final String TABLE_PARAMETER_ELEMENT = "tabela";

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica se o
   * {@link TableParameter parmetro do tipo tabela} deve delimitar as linhas da
   * tabela na linha de comando.  opcional, o seu valor-padro 
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   */
  private static final String TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE =
    "delimitar_linhas";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE}.
   * </p>
   */
  private static final boolean TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE =
    false;

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica se o
   * {@link TableParameter parmetro do tipo tabela} deve delimitar a tabela na
   * linha de comando.  opcional, o seu valor-padro 
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE} e o seu tipo 
   * booleano.
   */
  private static final String TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_ATTRIBUTE =
    "delimitar_tabela";

  /**
   * <p>
   * O valor-padro para o atributo
   * {@link #TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE}
   * .
   * </p>
   */
  private static final boolean TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE =
    false;

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_VISIBLE_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade de linhas do
   * {@link TableParameter parmetro do tipo tabela}.  opcional, seu tipo 
   * inteiro e o valor mnimo 
   * {@link #TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE}.
   */
  private static final String TABLE_PARAMETER_ELEMENT_VISIBLE_ROW_COUNT_ATTRIBUTE =
    "quantidade_de_linhas_visiveis";

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_MAX_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade mxima de
   * linhas do {@link TableParameter parmetro do tipo tabela};  opcional.
   */
  private static final String TABLE_PARAMETER_ELEMENT_MAX_ROW_COUNT_ATTRIBUTE =
    "quantidade_maxima_de_linhas";

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_MIN_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade mxima de
   * linhas do {@link TableParameter parmetro do tipo tabela};  opcional.
   */
  private static final String TABLE_PARAMETER_ELEMENT_MIN_ROW_COUNT_ATTRIBUTE =
    "quantidade_minima_de_linhas";

  /**
   * <p>
   * O valor mnimo para o atributo
   * {@link #TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE}.
   * </p>
   * <p>
   * O seu valor  {@value #TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE}.
   * </p>
   */
  private static final int TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE = 1;

  /**
   * O atributo {@value #TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE} do
   * elemento {@link #TABLE_PARAMETER_ELEMENT}: indica a quantidade de linhas
   * sero visveis no {@link TableParameter parmetro do tipo tabela}. 
   * opcional, seu tipo  inteiro e o valor mnimo 
   * {@link #TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE}.
   */
  private static final String TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE =
    "quantidade_de_linhas";

  /**
   * <p>
   * O elemento {@link #CELL_VALUE_ELEMENT}: indica o valor de uma clula do
   * {@link TableParameter parmetro tipo tabela} indicado pelo elemento que 
   * pai deste elemento.
   * </p>
   */
  private static final String CELL_VALUE_ELEMENT = "celula";

  /**
   * O atributo {@value #CELL_VALUE_ELEMENT_COLUMN_ID_ATTRIBUTE} do elemento
   * {@link #CELL_VALUE_ELEMENT}: indica o identificador da coluna, 
   * obrigatrio e  do tipo string.
   */
  private static final String CELL_VALUE_ELEMENT_COLUMN_ID_ATTRIBUTE =
    "id_da_coluna";

  /**
   * O atributo {@value #CELL_VALUE_ELEMENT_ROW_INDEX_ATTRIBUTE} do elemento
   * {@link #CELL_VALUE_ELEMENT}: indica o ndice da linha,  obrigatrio,  do
   * tipo inteiro e o valor mnimo  0.
   */
  private static final String CELL_VALUE_ELEMENT_ROW_INDEX_ATTRIBUTE =
    "indice_da_linha";

  /**
   * O atributo {@value #CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE} do elemento
   * {@link #CELL_VALUE_ELEMENT}: indica o valor da clula,  obrigatrio, o
   * tipo depende do tipo da coluna.
   */
  protected static final String CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE = "valor";

  /**
   * Mapa que associa as fbricas ao identificador de sua coluna.
   */
  private Map<String, TableColumnFactory> factoriesByColumnId;

  /**
   * Cria o analisador.
   */
  protected AbstractTableParameterFactory() {
    factoriesByColumnId = new HashMap<String, TableColumnFactory>();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TableParameter createSimpleParameter(XmlParser parser, String name,
    String label, String description, boolean isOptional, boolean isVisible,
    String commandLinePattern, ParameterGroup group,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    boolean delimitTable = parser.extractAttributeValueAsBoolean(
      TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_ATTRIBUTE,
      TABLE_PARAMETER_ELEMENT_DELIMIT_TABLE_DEFAULT_VALUE);
    boolean delimitRows = parser.extractAttributeValueAsBoolean(
      TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_ATTRIBUTE,
      TABLE_PARAMETER_ELEMENT_DELIMIT_ROWS_DEFAULT_VALUE);
    Integer rowCount = parser.extractAttributeValueAsInteger(
      TABLE_PARAMETER_ELEMENT_ROW_COUNT_ATTRIBUTE, null, null,
      TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE);
    Integer visibleRowCount = parser.extractAttributeValueAsInteger(
      TABLE_PARAMETER_ELEMENT_VISIBLE_ROW_COUNT_ATTRIBUTE, null, null,
      TABLE_PARAMETER_ELEMENT_ROW_COUNT_MINIMUM_VALUE);

    Integer maxRowCount = parser.extractAttributeValueAsInteger(
      TABLE_PARAMETER_ELEMENT_MAX_ROW_COUNT_ATTRIBUTE, null, null, null);

    Integer minRowCount = parser.extractAttributeValueAsInteger(
      TABLE_PARAMETER_ELEMENT_MIN_ROW_COUNT_ATTRIBUTE, null, null, null);

    List<TableColumn<?>> columns = loadColumns(parser, name, configurator);
    TableParameter parameter = new TableParameter(name, label, description,
      isOptional, isVisible, commandLinePattern, columns, delimitTable,
      delimitRows, rowCount, visibleRowCount, minRowCount, maxRowCount);
    return parameter;
  }

  /**
   * <p>
   * Carregas as colunas da tabela.
   * </p>
   *
   * @param parser O analisador (No aceita {@code null}).
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   *
   * @return As colunas.
   *
   * @throws ParseException Se houver um erro no XML, inclusive se no houver
   *         colunas.
   */
  private List<TableColumn<?>> loadColumns(XmlParser parser,
    String parameterName, SimpleAlgorithmConfigurator configurator)
      throws ParseException {
    List<TableColumn<?>> columns = new LinkedList<TableColumn<?>>();
    if (!parser.goToFirstChild()) {
      throw new ParseException("A lista de colunas da tabela est vazia.");
    }
    do {
      String elementName = parser.getElementName();
      if (elementName.equals(CELL_VALUE_ELEMENT)) {
        setCellValue(parser, parameterName, columns);
      }
      else {
        createColumn(parser, parameterName, columns, elementName, configurator);
      }
    } while (parser.goToNextSibling());
    parser.goToParent();
    return columns;
  }

  /**
   * <p>
   * Cria uma coluna.
   * </p>
   *
   * <p>
   * Adiciona a coluna na lista de colunas em construo.
   * </p>
   *
   * <p>
   * Se o elemento corrente no for de um tipo vlido de coluna, ele ignora o
   * elemento e nada faz.
   * </p>
   *
   * @param parser O analisador (No aceita {@code null}).
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param columns As colunas que esto em construo (No aceita {@code null}
   *        ).
   * @param elementName O nome do parmetro (No aceita {@code null}).
   * @param configurator O configurador de algoritmos (No aceita {@code null}).
   *
   * @throws ParseException Se houver um erro no XML.
   */
  private void createColumn(XmlParser parser, String parameterName,
    List<TableColumn<?>> columns, String elementName,
    SimpleAlgorithmConfigurator configurator) throws ParseException {
    TableColumnFactory factory = createFactory(elementName);
    if (factory != null) {
      TableColumn<?> column = factory.createColumn(parser, parameterName,
        configurator);
      factoriesByColumnId.put(column.getId(), factory);
      if (columns.contains(column)) {
        throw new ParseException("J existe uma coluna {0}.", column);
      }
      columns.add(column);
    }
  }

  /**
   * <p>
   * Atribui o valor padro de uma cluna da tabela.
   * </p>
   *
   * <p>
   * O elemento corrente precisa ser {@link #CELL_VALUE_ELEMENT} (
   * {@value #CELL_VALUE_ELEMENT}).
   * </p>
   *
   * @param parser O analisador (No aceita {@code null}).
   * @param parameterName O nome do parmetro (No aceita {@code null}).
   * @param columns As colunas que esto em construo (No aceita {@code null}
   *        ).
   *
   * @throws ParseException Se houver um erro no XML.
   */
  private void setCellValue(XmlParser parser, String parameterName,
    List<TableColumn<?>> columns) throws ParseException {
    Integer rowIndex = parser.extractAttributeValueAsInteger(
      CELL_VALUE_ELEMENT_ROW_INDEX_ATTRIBUTE, null, 0);
    String columnId = parser.extractAttributeValue(
      CELL_VALUE_ELEMENT_COLUMN_ID_ATTRIBUTE);
    TableColumn<?> theColumn = null;
    for (TableColumn<?> aColumn : columns) {
      if (aColumn.getId().equals(columnId)) {
        theColumn = aColumn;
        break;
      }
    }
    if (theColumn == null) {
      throw new ParseException(
        "A coluna {2} da tabela {0} referenciada pela clula {1} {2} "
          + "no existe na tabela ou ainda no foi definida.", parameterName,
        rowIndex, columnId);
    }
    TableColumnFactory factory = factoriesByColumnId.get(columnId);
    factory.setCellValue(parser, parameterName, theColumn, rowIndex,
      CELL_VALUE_ELEMENT_VALUE_ATTRIBUTE);
  }

  /**
   * Cria a fbrica de colunas.
   *
   * @param elementName O nome do elemento que est sendo processado.
   *
   * @return A fbrica ou {@code null} caso no haja fbrica cadastrada para
   *         este elemento.
   */
  protected abstract TableColumnFactory createFactory(String elementName);
}
