package csbase.logic.algorithms.validation;

import java.io.Serializable;
import java.util.Arrays;

/**
 * Mensagem internacionalizvel para descrever o resultado de validaes.
 */
public class LocalizedMessage implements Serializable {

  /**
   * Chave de internacionalizao.
   */
  private String key;
  /**
   * Parmetros para a mensagem internacionalizvel.
   */
  private Object[] args;

  /**
   * Construtor.
   * 
   * @param clazz a classe que define a chave de internacionalizao. Se for
   *        definida, o nome da classe ser usado como sufixo para a chave de
   *        internacionalizao.
   * @param key a chave de internacionalizao.
   * @param args parmetros para a mensagem internacionalizvel.
   */
  public LocalizedMessage(Class<?> clazz, String key, Object[] args) {
    setKey(clazz, key);
    this.args = args;
  }

  /**
   * Construtor.
   * 
   * @param key a chave de internacionalizao.
   * @param args parmetros para a mensagem internacionalizvel.
   */
  public LocalizedMessage(String key, Object[] args) {
    this(null, key, args);
  }

  /**
   * Construtor.
   * 
   * @param clazz a classe que define a chave de internacionalizao. Se for
   *        definida, o nome da classe ser usado como sufixo para a chave de
   *        internacionalizao.
   * @param key a chave de internacionalizao.
   */
  public LocalizedMessage(Class<?> clazz, String key) {
    this(clazz, key, null);
  }

  /**
   * Construtor.
   * 
   * @param key a chave de internacionalizao.
   */
  public LocalizedMessage(String key) {
    this(key, null);
  }

  /**
   * Atribui a chave de internacionalizao.
   * 
   * @param clazz a classe que define a chave de internacionalizao. Se for
   *        definida, o nome da classe ser usado como sufixo para a chave de
   *        internacionalizao.
   * @param key a chave de internacionalizao.
   */
  protected void setKey(Class<?> clazz, String key) {
    if (key == null) {
      throw new IllegalArgumentException("O parmetro key est nulo.");
    }
    String newKey;
    if (clazz != null) {
      newKey = clazz.getSimpleName() + "." + key;
    }
    else {
      newKey = key;
    }
    this.key = newKey;
  }

  /**
   * Retorna a chave de internacionalizao da mensagem.
   * 
   * @return a chave de internacionalizao.
   */
  public String getKey() {
    return key;
  }

  /**
   * Retorna os parmetros da mensagem.
   * 
   * @return os parmetros da mensagem.
   */
  public Object[] getArgs() {
    return args;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    String str = key;
    if (args != null && args.length > 0) {
      str += " " + Arrays.toString(args);
    }
    return str;
  }

}
