package csbase.remote;

import java.rmi.RemoteException;

import tecgraf.javautils.core.filter.IFilter;
import csbase.util.messages.Message;

/**
 * Interface remota para acesso ao servio de mensagens.
 * 
 * @author Tecgraf
 */
public interface MessageServiceInterface extends ServiceInterface {

  /** Nome do servio para o <code>ServiceManager</code> */
  public static final String SERVICE_NAME = "MessageService";

  /**
   * Recebe novas mensagens, destinadas a seo do usurio conectado ao servio.
   * 
   * @param filter filtro que determina as mensagens que sero retornadas.
   * 
   * @return Mensagens destinadas a seo do usurio.
   * 
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  public Message[] receive(IFilter<Message> filter) throws RemoteException;

  /**
   * Atribui o ouvinte de mensagens da seo do usurio, ao servio de
   * mensagens.
   * 
   * @param listener Ouvinte de mensagens.
   * @param filter Filtro que determina as mensagens que sero repassadas ao
   *        ouvinte.
   * 
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  public void setMessageListener(IRemoteMessageListener listener,
    IFilter<Message> filter) throws RemoteException;

  /**
   * Remove do servio de mensagens, o ouvinte de mensagens para a seo do
   * usurio conetado.
   * 
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  public void clearMessageListener() throws RemoteException;

  /**
   * Envia uma mensagem a vrios destinatrios.
   * 
   * @param message Mensagem a ser enviada.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   * @param usersLogin Login dos usurios de destino.
   * 
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  public void send(Message message, long timeToLive, String... usersLogin)
    throws RemoteException;

  /**
   * Envia uma mensagem para todos os usurios.
   * 
   * @param message Mensagem a ser enviada.
   * @param timeToLive Tempo, em milisegundos, que essa mensagem deve persistir
   *        at que seja consumida.
   * 
   * @throws RemoteException Caso haja problemas de comunicao com o servidor.
   */
  public void broadcast(Message message, long timeToLive)
    throws RemoteException;
}
