/*
 * $Id$
 */
package csbase.logic;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Map.Entry;

/**
 * Representa uma URI de conexo em um servidor CSBase.
 * <p>
 * Ex: <br>
 * csbase://localhost:1090?info1=xpto&info2=xpto
 * </p>
 * 
 * @author Tecgraf/PUC-Rio
 * 
 */
public final class ServerURI implements Serializable, Comparable<ServerURI> {

  /** Esquema CSBase */
  public static String CSBASE_SCHEME = "csbase";

  /** Instncia da URI */
  protected URI uri;

  /**
   * Constri uma nova URI de conexo com um servidor CSBase
   * 
   * @param str A string que representa a URI
   * @throws IllegalArgumentException Se no  uma URI CSBase vlida. Ex:
   *         csbase://csbaseserver:2099
   */
  private ServerURI(String str) {
    try {
      if (str == null)
        throw new IllegalArgumentException("str == null");
      this.uri = new URI(str);
      this.uri = this.uri.normalize();
      if (this.uri.getScheme() == null
        || !this.uri.getScheme().equals(CSBASE_SCHEME))
        throw new IllegalArgumentException(str + " no  uma URI CSBase vlida");
    }
    catch (URISyntaxException e) {
      throw new IllegalArgumentException(str + " no  uma URI CSBase vlida");
    }
  }

  /**
   * @return Recupera host e porta da URI
   */
  public String getHostAndPort() {
    return this.getHost() + ":" + this.getPort();
  }

  /**
   * Recupera a porta. Se a porta no foi explicitamente definida na URI ento a
   * porta RMI default 1099 ser retornada.
   * 
   * @return A porta especificada na URI ou 1099 como sendo a default.
   */
  public int getPort() {
    return this.uri.getPort() != -1 ? this.uri.getPort() : 1099;
  }

  /**
   * Recupera o host da URI
   * 
   * @return O host da URI
   */
  public String getHost() {
    return this.uri.getHost();
  }

  /**
   * @return String com os parmetros desta URI
   */
  public String getParams() {
    return this.uri.getQuery();
  }

  /**
   * @return Um mapa com os parmetros passados na URI
   */
  public Map<String, String> getParamsMap() {
    Map<String, String> params = null;
    if (this.getParams() != null) {
      params = new HashMap<String, String>();
      StringTokenizer st = new StringTokenizer(this.getParams(), "&");
      while (st.hasMoreTokens()) {
        String tk = st.nextToken();
        String[] p = tk.split("=");
        if (p.length == 2)
          params.put(tk.split("=")[0], tk.split("=")[1]);
      }
    }
    if (params != null)
      return Collections.unmodifiableMap(params);
    return null;
  }

  /**
   * Adiciona parmetros a URI. Os parmetros que existirem sero sobreescritos.
   * 
   * @param params O mapa de parmetros a ser adicionado
   * @return Uma nova URI com os parmetros adicionados
   */
  public ServerURI appendParams(Map<String, String> params) {
    if (params == null)
      throw new IllegalArgumentException("params == null");

    if (params.size() != 0) {
      String strURI = CSBASE_SCHEME + "://" + this.getHostAndPort();

      Map<String, String> mergedParams = null;

      if (this.getParamsMap() != null)
        mergedParams = new HashMap<String, String>(this.getParamsMap());

      if (mergedParams == null)
        mergedParams = params;
      else
        mergedParams.putAll(params);

      strURI += '?';

      int c = mergedParams.size();

      for (Entry<String, String> entry : mergedParams.entrySet()) {
        if (--c > 0)
          strURI += entry.getKey() + '=' + entry.getValue() + '&';
        else
          strURI += entry.getKey() + '=' + entry.getValue();
      }

      return new ServerURI(strURI);
    }
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    return this.uri.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int hashCode() {
    Map<String, String> p = this.getParamsMap();

    if (p == null) {
      return this.getHostAndPort().hashCode();
    }

    long keyHash = 0;
    long valueHash = 0;

    for (String k : p.keySet())
      keyHash += Math.abs(k.hashCode());

    for (Serializable v : p.values())
      valueHash += Math.abs(v.hashCode());

    return this.getHostAndPort().hashCode() + (int) (keyHash * valueHash);
  }

  /**
   * {@inheritDoc}
   * 
   * Uma {@link ServerURI}  igual a outra mesmo que os parmetros estejam fora
   * de ordem mas sejam os mesmos
   */
  @Override
  public boolean equals(Object obj) {
    if (obj == null) {
      return false;
    }

    if (!obj.getClass().equals(this.getClass())) {
      return false;
    }

    ServerURI sURI = (ServerURI) obj;

    if (!this.getHostAndPort().equals(sURI.getHostAndPort())) {
      return false;
    }

    Map<String, String> p1 = sURI.getParamsMap();
    Map<String, String> p2 = this.getParamsMap();

    if (p1 != null && p2 != null) {
      if (p1.size() != p2.size()) {
        return false;
      }

      for (String k : p1.keySet()) {
        if (!p2.containsKey(k))
          return false;
      }

      for (Serializable v : p1.values()) {
        if (!p2.containsValue(v))
          return false;
      }
    }
    else if (p1 != null || p2 != null) {
      return false;
    }

    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int compareTo(ServerURI serverURI) {
    return serverURI.uri.compareTo(this.uri);
  }

  /**
   * Cria uma URI para acesso a um servidor.
   * 
   * @param host O nome do servidor.
   * @param port A porta do servidor.
   * 
   * @return A URI.
   */
  public static ServerURI create(String host, int port) {
    return new ServerURI((CSBASE_SCHEME + "://" + host + ":" + String
      .valueOf(port)));
  }

  /**
   * Retorna uma URI CSBase a partir de uma String
   * 
   * @param uriStr A String representando a URI
   * @return A instncia da URI ou null caso uriStr no represente uma URI
   *         CSBase vlida
   */
  public static ServerURI parse(String uriStr) {
    try {
      ServerURI uri = new ServerURI(uriStr);
      if (uri.getHostAndPort() != null) {
        return new ServerURI(uriStr);
      }
    }
    catch (Exception e) {
      /* nada a fazer */
    }
    return null;
  }

  /**
   * Indica se a URI CSBase  vlida
   * 
   * @param uriStr A String representando a URI CSBase
   * @return true se vlida, false caso contrrio
   */
  public static boolean isValid(String uriStr) {
    return parse(uriStr) != null;
  }

  /**
   * Criado para testes
   * 
   * @param a argumento.
   */
  public static void main(String a[]) {
    ServerURI sUri1 = ServerURI.parse("csbase:/localhost?x=3");

    System.out.println(sUri1);

    Map<String, String> m = new HashMap<String, String>();
    m.put("z", "");

    System.out.println(sUri1.appendParams(m));

  }

}
