package csbase.logic.algorithms.parameters;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * Valor que representa a linha de uma tabela.
 */
public final class RowValue implements Serializable {

  /** Lista de valores de cada uma das clulas da tabela nesta linha. */
  private List<Object> cellValues;

  /** Observadores de mudanas nos valores desta linha. */
  private List<RowValueListener> listeners;

  /**
   * Construtor.
   * 
   * @param cellValues a lista de valores desta linha.
   */
  public RowValue(Object... cellValues) {
    if (cellValues == null) {
      throw new IllegalArgumentException("O parmetro cellValues est nulo.\n");
    }
    this.cellValues = new ArrayList<Object>(cellValues.length);
    for (Object cellValue : cellValues) {
      this.cellValues.add(cellValue);
    }
    listeners = new LinkedList<RowValueListener>();
  }

  /**
   * Adiciona um novo observador de mudanas nos valores desta linha.
   * 
   * @param listener o observador.
   */
  public void addRowValueListener(RowValueListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    listeners.add(listener);
  }

  /**
   * Obtm o valor da clula em uma determinada coluna.
   * 
   * @param columnIndex o ndice numrico da coluna.
   * @return o valor da clula no ndice.
   */
  public Object getCellValue(int columnIndex) {
    return cellValues.get(columnIndex);
  }

  /**
   * Remove um novo observador de mudanas nos valores desta linha.
   * 
   * @param listener o observador.
   * @return verdadeiro, se bem sucedido ou falso, caso contrrio.
   */
  public boolean removeRowValueListener(RowValueListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    return listeners.remove(listener);
  }

  /**
   * Atribui um novo valor  clula de uma determinada coluna.
   * 
   * @param cellValue o novo valor da clula.
   * @param columnIndex o ndice da coluna.
   */
  public void setCellValue(Object cellValue, int columnIndex) {
    cellValues.set(columnIndex, cellValue);
    for (RowValueListener listener : listeners) {
      listener.valueWasChanged(cellValue, columnIndex);
    }
  }

  /**
   * Obtm a lista de valores das clulas desta linha.
   * 
   * @return a lista de valores.
   */
  public List<Object> getCellValues() {
    return Collections.unmodifiableList(cellValues);
  }
}
